import java.util.*;
import java.util.regex.*;


/**
 * This class represents regular expressions. It will allow to generate random legal and illegal strings
 * taking as starting point the reference of one regular expression
 * 
 * TODO Explain syntax (remember {}^$ must always be preceded by the escape char)
 * ?
 * +
 * *
 * ^
 * \\
 * $
 * ()
 * []
 * {}
 * \\.
 * \\d
 * \\w
 * \\s
 * \\a
 * \\D
 * \\W
 * \\S
 * \\A
 * 
 * @author I054742
 *
 */
class RegularExpression {

	// Future improvements:
	//		- Add \p{Lower} and \p{Upper} and \p{ASCII} ...
	private final int NUMB_OF_ILLEGAL_TRIES = 10;
	private final int NUMB_OF_DIFF_CHARS = 1<<8;
	
	private final int MAX_LEGAL_REPETITIONS = 8;
	
	
	private String origRegExp = null;
	String getOrigRegExp () {return origRegExp;}
	
	/* Will contain the set of character set related with the meta letters */
	private String charSetOf[] = new String[this.NUMB_OF_DIFF_CHARS];
	//private String metaSymbol = ".*+?^()[]\\{}$";
	/* Set of characters associated with set of characters when they are preceded by the escape character */
	private String metaLetter = "dwsaDWSA.";
	/* Indicates if this regular expression was compiled successfully */
	private boolean compiled = false;
	/* Translated regular expression to be used with java.util.regex.*; */
	private String translated = null;
	/* It will hold the compiled translated regular expression */
	private Pattern pattern = null;
	/* It will hold the data structure result of the compilation */
	private Vector compilationResult = null;
	/* It will hold all possible legal characters of the whole regular expression */
	private boolean exists[] = new boolean[this.NUMB_OF_DIFF_CHARS];
	/* Indicates if the regular expression represents an integer */
	private boolean regExpInteger = false;
	/* Indicates if the regular expression represents a double */
	private boolean regExpDouble = false;
	/* Indicates if the process of initialization for illegal integer was initialized */
	static private boolean initIllegalInteger = false;
	//static private boolean initIllegalDouble = false;
	//static private boolean initLegalInteger = false;
	//static private boolean initLegalDouble = false;
	/* Set of illegal integer values */
	static private String illegalIntegerValues[] = null;
	/* Set of illegal double values */
	static private String illegalDoubleValues[] = {
		"1.7976931348623159E308", 
		"1.7976931348623160E308", 
		"1.7976931348623158E309", 
		"1.7976931348623158E7897",
		"1.7986931348623157E308", 
		".7976931348623157E328", 
		"2.7976931348623157E308", 
		"127E3588",
		"9E-6324", 
		".01E-566", 
		".01E-2002", 
		"4.9E-325", 
		"1.7E-324", 
		".1E-324", 
		"1.E-324", 
		"11.E-325"
	};
	//static private String legalIntegerValues[] = null;
	//static private String legalDoubleValues[] = null;
	
	/**
	 * Set of predefined identifiers
	 */
	static private final String[] PREDEFINED_IDENTIFIERS = {
		"INTEGER",
		"DOUBLE",
		"BIG_INTEGER",
		"BIG_DOUBLE",
		"CHAR",
		"ALPHA",
		"NUMERIC",
		"TELEPHONE",
		"STRING",
		"ALPHANUMERIC"
	};
	
	/**
	 * 
	 * @return
	 */
	static String[] getPREDEFINED_IDENTIFIERS () {
		
		return PREDEFINED_IDENTIFIERS;
	}
	
	/**
	 * Related predefined regular expressions
	 */
	static private final String[] PREDEFINED_REGEXP = {
		"",
		"",
		"-?[1-9][0-9]*|0", // BIG_INTEGER
		"(-?[1-9][0-9]*|0)(.|.[0-9]+)?", // BIG_DOUBLE
		"\\.", // CHAR
		"[a-zA-Z]*", // ALPHA
		"[0-9]*", // NUMERIC
		"(\\(\\+[0-9]\\)|\\(00[0-9]+\\))(-[0-9]+)+", // TELEPHONE
		"[-a-zA-Z0-9.,øæå\\\\+?*|!'#£\\$¤%&/()=\\{\\}~_:;\\^ ]*", // STRING
		"[a-zA-Z0-9]*" // ALPHANUMERIC
	};
	
	/* Indicates if the set of predefined identifiers was initialized */
	static private boolean initIdentifier = false;
	/* Will contain the set of identifiers, each related with its regular expression */
	static private MyHashMap identifiers = new MyHashMap();
	
	
	static private final MyHashMap PREDEFINED_REGULAR_EXPRESSIONS = new MyHashMap() {{
		
		put("INTEGER",getRegExpFromId("INTEGER"));
		put("DOUBLE",getRegExpFromId("DOUBLE"));
		put("BIG_INTEGER",getRegExpFromId("BIG_INTEGER"));
		put("BIG_DOUBLE",getRegExpFromId("BIG_DOUBLE"));
		put("CHAR",getRegExpFromId("CHAR"));
		put("ALPHA",getRegExpFromId("ALPHA"));
		put("NUMERIC",getRegExpFromId("NUMERIC"));
		put("TELEPHONE",getRegExpFromId("TELEPHONE"));
		put("STRING",getRegExpFromId("STRING"));
		put("ALPHANUMERIC",getRegExpFromId("ALPHANUMERIC"));
	}};
	
	static MyHashMap getPREDEFINED_REGULAR_EXPRESSIONS () {
		
		return PREDEFINED_REGULAR_EXPRESSIONS;
	}
	
	
	/**
	 * Constructor
	 */
	RegularExpression () {
		
		this.initCharSetOf();
	}
	
	
	/**
	 * Gives an illegal integer
	 * 
	 * @param generator Random generator
	 * @return An illegal integer
	 */
	static private String illegalInteger (Random generator) {
		
		/* Number of precalculated integer values */
		int LEN_PRECALCULATED_VALUES = 120;
		
		/* Creates a set of illegal values, most of then close to the high and low limits */
		if (RegularExpression.initIllegalInteger==false) {
			long x = Integer.MAX_VALUE; x++;
			long y = Integer.MIN_VALUE; y--;
			RegularExpression.illegalIntegerValues= new String[LEN_PRECALCULATED_VALUES];
			for (int i=0, len=LEN_PRECALCULATED_VALUES/6, j=0; i<len; i++) {
				RegularExpression.illegalIntegerValues[j++]=Long.toString(x);
				RegularExpression.illegalIntegerValues[j++]=Long.toString(y);
				RegularExpression.illegalIntegerValues[j++]=Long.toString(x+generator.nextInt(3)+1);
				RegularExpression.illegalIntegerValues[j++]=Long.toString(y-generator.nextInt(3)-1);
				RegularExpression.illegalIntegerValues[j++]=Long.toString(x+generator.nextInt(Integer.MAX_VALUE)+1);
				RegularExpression.illegalIntegerValues[j++]=Long.toString(y-generator.nextInt(Integer.MAX_VALUE)-1);
			}
			RegularExpression.initIllegalInteger=true;
		}
		return RegularExpression.illegalIntegerValues[generator.nextInt(RegularExpression.illegalIntegerValues.length)];
	}
	
	
	/**
	 * Gives an illegal double value
	 * 
	 * @param generator A random generator
	 * @return An illegal double value
	 */
	static private String illegalDouble (Random generator) {
		
		return RegularExpression.illegalDoubleValues[generator.nextInt(RegularExpression.illegalDoubleValues.length)];
	}
	
	
	/**
	 * Gives a legal integer value
	 * 
	 * @param generator Random generator
	 * @return A legal integer value
	 */
	static private String legalInteger (Random generator) {
		
		if (generator.nextInt(100)<33) {
			int x = generator.nextInt(100);
			if (x<9) return Integer.toString(0);
			else if (x<18) return Integer.toString(1);
			else if (x<27) return Integer.toString(2);
			else if (x<36) return Integer.toString(-1);
			else if (x<45) return Integer.toString(-2);
			else if (x<54) return Integer.toString(Integer.MAX_VALUE);
			else if (x<63) return Integer.toString(Integer.MAX_VALUE-1);
			else if (x<72) return Integer.toString(Integer.MAX_VALUE-2);
			else if (x<81) return Integer.toString(Integer.MIN_VALUE);
			else if (x<90) return Integer.toString(Integer.MIN_VALUE+1);
			else return Integer.toString(Integer.MIN_VALUE+2);
		}
		else {
			if (generator.nextInt(100)<50) {
				return Integer.toString(generator.nextInt(Integer.MAX_VALUE));
			}
			else {
				return Integer.toString(generator.nextInt(Integer.MAX_VALUE)*-1);
			}
		}
	}
	
	
	/**
	 * Gives a legal double value
	 * 
	 * @param generator Random generator
	 * @return A legal value
	 */
	static private String legalDouble (Random generator) {
		
		if (generator.nextInt(100)<33) {
			int x = generator.nextInt(100);
			if (x<9) return Double.toString(0);
			else if (x<18) return Double.toString(0.00000000001);
			else if (x<27) return Double.toString(1.0);
			else if (x<36) return Double.toString(-1);
			else if (x<45) return Double.toString(-0.00000000001);
			else if (x<54) return Double.toString(Double.MAX_VALUE);
			else if (x<63) return Double.toString(Double.MAX_VALUE-1);
			else if (x<72) return Double.toString(Double.MAX_VALUE-0.000000000001);
			else if (x<81) return Double.toString(Double.MIN_VALUE);
			else if (x<90) return Double.toString(Double.MIN_VALUE+1);
			else return Double.toString(Double.MIN_VALUE+0.000000000001);
		}
		else {
			if (generator.nextInt(100)<50) {
				return Double.toString(generator.nextDouble());
			}
			else {
				return Double.toString(generator.nextDouble()*-1.0);
			}
		}
	}
	
	
	/**
	 * This method receives 'id'. 'id' is either a regular expression or an identifier. This method will check
	 * what 'id' is. If it is an identifier it will treated for its particular case. After finishing this
	 * first step then the resultant regular expression will be compiled
	 * 
	 * @param id Either a regular expression or an identifier
	 * @return A RegularExpression object if the resulting regular expression was compiled successfully
	 * 		   null otherwise
	 */
	static RegularExpression getRegExpFromId (String id) {
		
		RegularExpression re = new RegularExpression();
		if (id.equalsIgnoreCase("INTEGER")) {
			re.regExpInteger=true;
		}
		else if (id.equalsIgnoreCase("DOUBLE")) {
			re.regExpDouble=true;
		}
		if (RegularExpression.initIdentifier==false) {
			for (int i=0; i<RegularExpression.PREDEFINED_IDENTIFIERS.length; i++) {
				RegularExpression.identifiers.put(RegularExpression.PREDEFINED_IDENTIFIERS[i],RegularExpression.PREDEFINED_REGEXP[i]);
			}
			RegularExpression.initIdentifier=true;
		}
		String identifier = (String)RegularExpression.identifiers.get(id);
		if (identifier!=null) id=identifier;
		
		if (re.compile(id)) return re;
		else return null;
	}
	
	
	/**
	 * Checks if a char is a digit
	 * 
	 * @param c The char
	 * @return 'true' if 'c' is a digit
	 * 		   'false' otherwise
	 */
	private boolean isDigit (char c) {
		
		return (c>='0' && c<='9');
	}
	
	
	/**
	 * Checks if 'c' is a letter
	 * 
	 * @param c The char to be checked
	 * @return 'true' if 'c' is a letter
	 * 		   'false' otherwise
	 */
	private boolean isLetter (char c) {
		
		return ((c>='A' && c<='Z') || (c>='a' && c<='z'));
	}
	
	
	/**
	 * Checks if a char is letter or digit
	 * 
	 * @param c The char
	 * @return 'true' if 'c' is a letter or digit
	 * 		   'false' otherwise
	 */
	private boolean isLetterOrDigit (char c) {
		
		return (this.isLetter(c) || this.isDigit(c));
	}
	
	
	/**
	 * Checks if 'c' is a space character
	 * 
	 * @param c The char to be checked
	 * @return 'true' if 'c' is a space character
	 * 		   'false' otherwise
	 */
	private boolean isSpace (char c) {
		
		return Character.isSpace(c);
	}
	
	
	/**
	 * Checks if 'c' is a new line char
	 * 
	 * @param c The char
	 * @return 'true' if 'c' is a new line char
	 * 		   'false' otherwise
	 */
	private boolean isNewLineChar (char c) {
		
		return (c=='\n' || c=='\r');
	}
	
	
	/**
	 * Initializes the char of set related with the meta letter set
	 */
	private void initCharSetOf () {
		
		for (int i=0; i<this.metaLetter.length(); i++) {
			this.charSetOf[this.metaLetter.charAt(i)]=new String();
		}
		for (char c=0; c<this.NUMB_OF_DIFF_CHARS; c++) {
			if (/*Character.isDigit(c)*/this.isDigit(c)) this.charSetOf['d']+=c;
			else this.charSetOf['D']+=c;
			if (/*Character.isLetter(c)*/this.isLetter(c)) this.charSetOf['a']+=c;
			else charSetOf['A']+=c;
			if (/*Character.isLetterOrDigit(c)*/this.isLetterOrDigit(c)) this.charSetOf['w']+=c;
			else this.charSetOf['W']+=c;
			if (/*Character.isSpace(c)*/this.isSpace(c)) this.charSetOf['s']+=c;
			else this.charSetOf['S']+=c;
			/*if (!this.isNewLineChar(c))*/ this.charSetOf['.']+=c;
		}
	}
	
	
	/**
	 * Checks if a character 'c' is a meta symbol
	 * 
	 * @param c The character
	 * @return 'true' if 'c' is a meta symbol
	 * 		   'false' otherwise
	 */
	private boolean isMetaSymbol (char c) {
		
		return (c=='[') || (c==']') || (c=='(') || (c==')') || (c=='*') || (c=='+') ||
			   (c=='?') || (c=='^') || (c=='|') || (c=='{') || (c=='}') || (c=='$') || (c=='\\');
	}
	
	
	/**
	 * Checks if a character 'c' is a meta letter
	 * 
	 * @param c The character
	 * @return 'true' if 'c' is a meta letter
	 * 		   'false' otherwise
	 */
	private boolean isMetaLetter (char c) {
		
		return (c=='d') || (c=='w') || (c=='s') || (c=='a') || (c=='.') ||
			   (c=='D') || (c=='W') || (c=='S') || (c=='A');
	}
	
	
	/**
	 * Checks if a character 'c' is a quantizer
	 * 
	 * @param c The character
	 * @return 'true' if 'c' is a quantizer
	 * 		   'false' otherwise
	 */
	private boolean isQuantizer (char c) {
		
		return (c=='*') || (c=='+') || (c=='?');
	}
	
	
	/**
	 * Checks if a character 'c' is a meta char
	 * 
	 * @param c The character
	 * @return 'true' if 'c' is a meta char
	 * 		   'false' otherwise
	 */
	private String slashBackCharSet (char c) {
		
		if (this.isMetaSymbol(c)) {
			return (new String())+c;
		}
		if (this.isMetaLetter(c)) {
			return charSetOf[c];
		}
		return null;
	}
	
	
	/**
	 * Marks in 'exists' the position corresponding to the chars contained in 'charSet'
	 * 
	 * @param charSet The set of chars
	 * @param exists The set of existing chars
	 */
	private void merge (String charSet, boolean exists[]) {
		
		for (int i=0; i<charSet.length(); i++) exists[charSet.charAt(i)]=true;
	}
	
	
	/**
	 * Creates a range of chars between c1 and c2 if the fits with the adequate restrictions
	 * 
	 * @param c1 Char
	 * @param c2 Char
	 * @return A String with the range or chars if they fits the adequate restrictions
	 * 		   null otherwise
	 */
	private String createRange (char c1, char c2) {
		
		if (c1>c2) return null;
		String res = null;
		if (!Character.isLetterOrDigit(c1) || !Character.isLetterOrDigit(c2)) return null;
		if (Character.isDigit(c1) && !Character.isDigit(c2)) return null;
		if (Character.isLetter(c1) && !Character.isLetter(c2)) return null;
		
		if ((Character.isDigit(c1) && Character.isDigit(c2)) || 
			(Character.isLowerCase(c1) && Character.isLowerCase(c2)) || 
			(Character.isUpperCase(c1) && Character.isUpperCase(c2))) {
			
			res = new String();
			for (char c=(char)Math.min(c1,(int)c2), last=(char)Math.max((int)c1,(int)c2); c<=last; c++) {
				res+=c;
			}
		}
		
		return res;
	}
	
	
	/**
	 * Analyze literally 'str' from the position 'from' until it finds either ']' or the end of 'str'
	 * 
	 * @param from Position where to analyze from
	 * @param str The set of char
	 * @return A set of char if the syntax is correct
	 * 		   null otherwise
	 */
	private Vector literalSet (int from, String str) {
		
		if (from>=str.length()) return null;
		Vector res = new Vector();
		boolean close = false;
		boolean negation = (str.charAt(from)=='^');
		int i=from+(negation?1:0);
		if (i>=str.length()) return null;
		boolean exists[] = new boolean[this.NUMB_OF_DIFF_CHARS];
		for (int k=0; k<this.NUMB_OF_DIFF_CHARS; k++) exists[k]=false;
		
		if (str.charAt(i)=='-') {
			i+=1;
			this.merge("-",exists);
		}
		
		for (; i<str.length(); i++) {
			
			char c = str.charAt(i);
			if ((c=='&') && (i+1>=str.length() || str.charAt(i+1)=='&')) return null;
			else if (c==']') {close=true; break;}
			else if (c=='\\') {
				if (i+1>=str.length()) return null;
				c = str.charAt(++i);
				String charSet = this.slashBackCharSet(c);
				if (charSet==null) return null;
				if (i+1>=str.length()) return null;
				if (str.charAt(i+1)=='-') {
					if (i+2>=str.length() || str.charAt(i+2)!=']') return null;
					else {
						this.merge(charSet,exists);
						this.merge("-",exists);
						i+=2; 
						close=true; 
						break;
					}
				}
				this.merge(charSet,exists);
			}
			else if (this.isProhibitedChar(c)) return null;
			else if (c=='-') {
				if (i+1>=str.length() || str.charAt(i+1)!=']') return null;
				else {
					this.merge("-",exists);
					i+=1; 
					close=true; 
					break;
				}
			}
			else {
				if (i+1>=str.length()) return null;
				if (str.charAt(i+1)=='-') {
					if (i+2>=str.length()) return null;
					if (str.charAt(i+2)==']') {
						this.merge(c+"-",exists);
						i+=2; 
						close=true; 
						break;
					}
					else {
						char c2 = str.charAt(i+2);
						String charSet = this.createRange(c,c2);
						if (charSet==null) return null;
						this.merge(charSet,exists);
						i+=2;
					}
				}
				else {
					this.merge((new String())+c,exists);
				}
			}
		}
		
		if (!close) return null;
		String resCharSet = new String();
		for (char c=0; c<this.NUMB_OF_DIFF_CHARS; c++) {
			if (exists[c]^negation) resCharSet+=c;
		}
		res.add(resCharSet);
		res.add(new Integer(i));
		return res;
	}
	
	
	/**
	 * Checks if 'c' is a prohibited char (some of these char can be used only in a special manner)
	 * 
	 * @param c The char to be checked
	 * @return 'true' if 'c' is a prohibited char
	 * 		   'false' otherwise
	 */
	private boolean isProhibitedChar (char c) {
		
		return ((c=='{') || (c=='}') || (c=='^') || (c=='$')); 
	}
	
	
	/**
	 * Compiles the regular expression 'str'
	 * 
	 * @param str The regular expression
	 * @return 'true' if the compilation was done correctly
	 * 		   'false' otherwise
	 */
	boolean compile (String str) {
		
		this.origRegExp = str;
		this.compiled=false;
		for (int i=0; i<this.NUMB_OF_DIFF_CHARS; i++) this.exists[i]=false;
		this.compilationResult = this.compileString(0,str,false);
		
		if (this.compilationResult==null) {
			Loggin.Log(Globals.indent()+"ERROR: Compilation error when compiling the regular expression -> \""+str+"\"");
			return false;
		}
		this.compiled=true;
		this.translated=this.translate(str);
		this.pattern = Pattern.compile(this.translated, Pattern.DOTALL);
		
		if (Certification.DEBUGGING_INFO) System.out.println("Translated: "+this.translated);
		
		return true;
	}
	
	
	/**
	 * Compiles a part of the regular expression 'str'
	 * 
	 * @param from Position where to compile from
	 * @param str Regular expression
	 * @param open Indicates if a parenthesis is open in this moment
	 * @return The corresponding data structure for the portion of analyzed regular expression
	 * 		   null otherwise
	 */
	private Vector compileString (int from, String str, boolean open) {
		
		if (str.length()==0) return new Vector();
		if (from>=str.length()) return null;
		Vector res = new Vector();
		Vector group = new Vector();
		int i=from;
		
		for (; i<str.length(); i++) {
			
			char c = str.charAt(i);
			if (c==')') {
				if (open) {open=false; break;}
				else return null;
			}
			else if (c==']') {
				return null;
			}
			ExpressionSet current = new ExpressionSet();
			if (c=='*') {return null;}
			else if (c=='+') {return null;}
			else if (c=='?') {return null;}
			else if (c=='{' || c=='}') {return null;}
			else if (c=='\\') {
				if (i+1==str.length()) {return null;}
				c = str.charAt(++i);
				String charSet = this.slashBackCharSet(c);
				if (charSet==null) return null;
				current.setCharSet(charSet);
			}
			else if (this.isProhibitedChar(c)) {
				return null;
			}
			else if (c=='|') {
				res.add(group);
				group=new Vector();
				continue;
			}
			else if (c=='(') {
				Vector v = this.compileString(i+1,str,true);
				if (v==null) {
					return null;
				}
				i=((Integer)v.elementAt(v.size()-1)).intValue();
				current.setGroup(v);
			}
			else if (c=='[') {
				Vector v = this.literalSet(i+1,str);
				if (v==null) {
					return null;
				}
				i=((Integer)v.elementAt(1)).intValue();
				current.setCharSet((String)v.elementAt(0));
			}
			else {
				current.setCharSet((new String())+c);
			}
			if (i+1<str.length()) {
				if (this.isQuantizer(str.charAt(i+1))) {
					current.setQuantizer(str.charAt(++i));
				}
				else if (str.charAt(i+1)=='{') {
					i++;
					String s = new String();
					while (i+1<str.length()) {
						if (str.charAt(i+1)=='}') break;
						s+=str.charAt(++i);
					}
					if (i+1>=str.length()) return null;
					if (this.evaluateMargins(s,current)==false) return null;
					current.setQuantizer('{');
					i++;
				}
			}
			current.mark();
			this.mark(current.getCharSet());
			group.add(current);
		}
		if (open) {
			return null;
		}
		res.add(group);
		res.add(new Integer(i));
		return res;
	}
	
	// TODO Check if it is useful 'to mark'
	/**
	 * Marks the chars of 'charSet' as existing ones in the regular expression
	 * 
	 * @param charSet The set of chars
	 */
	private void mark (String charSet) {
		
		if (charSet==null) return;
		for (int i=0; i<charSet.length(); i++) this.exists[charSet.charAt(i)]=true;
	}
	
	
	/**
	 * Evaluates a part of the regular expression with the form "{x,y}" where "," and "y" can exists or not.
	 * But this method only will receive "x,y"
	 * 
	 * @param s The portion of the regular expression to be analyzed
	 * @param current ExpressionSet that will be set as consequence of this analysis
	 * @return 'true' if the expression is legal
	 * 		   'false' otherwise
	 */
	private boolean evaluateMargins (String s, ExpressionSet current) {
		
		s=GenericMethods.cleanSpaces(s);
		String parts[] = s.split(",");
		if (parts.length<=0 || parts.length>2) return false;
		int ns[] = new int[parts.length];
		for (int i=0; i<parts.length; i++) {
			try {
				ns[i]=Integer.parseInt(parts[i]);
			}
			catch (Exception exc) {
				if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in RegularExpression.evaluateMargins");
				return false;
			}
		}
		if (ns[0]<0) return false;
		if (parts.length==2 && ns[1]<ns[0]) return false;
		if (parts.length==1 && s.indexOf(",")<0) current.setMiniMaxi(ns[0],ns[0]);
		else if (parts.length==1) current.setMiniMaxi(ns[0],Integer.MAX_VALUE);
		else current.setMiniMaxi(ns[0],ns[1]);
		return true;
	}
	
	
	/**
	 * Calculates the number of repetitions that will be made in 'es'
	 * 
	 * @param es The ExpressionSet
	 * @param generator A random generator
	 * @return Number of times that 'es' will be repeated
	 */
	private int repetitions (ExpressionSet es, Random generator, boolean maximum) {
		
		int TOP = 7;
		if (maximum && generator.nextInt(100)<2) TOP = MAX_LEGAL_REPETITIONS;
		char q = es.getQuantizer();
		if (q=='*') return generator.nextInt(TOP);
		else if (q=='+') return generator.nextInt(TOP)+1;
		else if (q=='?') return generator.nextInt(2);
		else if (q=='{') return Math.min(TOP,generator.nextInt(es.getMaxi()-es.getMini()+1))+es.getMini();
		else return 1;
	}
	
	
	/**
	 * Creates a random legal String from this regular expression
	 * 
	 * @param seed Seed for the random generator
	 * @return A random legal String
	 * @throws Exception This exception will be thrown if no regular expression was compiled properly
	 */
	String randomLegalString (int seed) throws Exception {
		
		if (this.compiled==false) {
			throw new Exception("The regular expression must be legal and properly compiled before this operation");
		}
		return this.randomLegalString(this.compilationResult,new Random(seed));
	}
	
	
	/**
	 * Creates a random legal String from this regular expression
	 * 
	 * @return A random legal String
	 * @throws Exception This exception will be thrown if no regular expression was compiled properly
	 */
	String randomLegalString () throws Exception {
		
		if (this.compiled==false) {
			throw new Exception("The regular expression must be legal and properly compiled before this operation");
		}
		return this.randomLegalString(this.compilationResult,new Random());
	}
	
	
	/**
	 * Creates a random legal String taking as initial point 'compExp'
	 * 
	 * @param compExp The point where the creation will start from
	 * @param generator Random generator
	 * @return The random legal String
	 */
	private String randomLegalString (Vector compExp, Random generator) {
		
		if (this.regExpInteger) return RegularExpression.legalInteger(generator);
		if (this.regExpDouble) return RegularExpression.legalDouble(generator);
		if (compExp.size()==0) return "";
		
		String res = new String();
		
		Vector group = (Vector)compExp.elementAt(generator.nextInt(compExp.size()-1));
		for (int i=0; i<group.size(); i++) {
			
			ExpressionSet expSet = (ExpressionSet)group.elementAt(i);
			for (int rep=this.repetitions(expSet,generator,true); rep>0; rep--) {
				if (expSet.getCharSet()!=null) {
					res+=expSet.getCharSet().charAt(generator.nextInt(expSet.getCharSet().length()));
				}
				else {
					res+=this.randomLegalString(expSet.getGroup(),generator);
				}
				if (res.length()>3000) rep = Math.min(rep,this.repetitions(expSet,generator,false));
			}
		}
		
		return res;
	}
	
	
	/**
	 * Tries to find an illegal char according to the legal ones recorded in 'exists'
	 * 
	 * @param exists Set of legal chars
	 * @param generator Random generator
	 * @return An illegal char if it was found
	 */
	private char tryIllegalChar (boolean exists[], Random generator) {
		
		if (generator.nextInt(100)<97) {
			
			char startsInterval[] = {'0','a','A'};
			int lens[] = {10, 26, 26};
			
			for (int from=generator.nextInt(3), i=0; i<3; i++, from++) {
				from%=3;
				for (int c = startsInterval[from], j=0, rand=generator.nextInt(lens[from]); j<lens[from]; j++, rand++) {
					rand%=lens[from];
					if (!exists[c+rand]) return (char)(c+rand);
				}
			}
		}
		
		return (char)(generator.nextInt(256));
	}
	
	
	/**
	 * Tries to add 'n' illegal chars to 'str'
	 * 
	 * @param str The String where adding the chars
	 * @param n The number of chars to be added
	 * @param generator Random generator
	 * @return The new string with the added chars
	 */
	private String addChars (String str, int n, Random generator) {
		
		for (int i=0; i<n; i++) {
			
			if (generator.nextInt(100)<50) {
				str=this.tryIllegalChar(this.exists,generator)+str;
			}
			else {
				str+=this.tryIllegalChar(this.exists,generator);
			}
		}
		return str;
	}
	
	
	/**
	 * Tries to create a random legal String
	 * 
	 * @param seed Seed used for the random generator
	 * @param mini Minimum number of chars of the illegal String
	 * @return The possible illegal String
	 * @throws Exception An exception will be thrown if the regular expression was not properly expressed
	 */
	String randomIllegalString (int seed, int mini) throws Exception {
		
		if (this.regExpInteger) return RegularExpression.illegalInteger(new Random(seed));
		if (this.regExpDouble) return RegularExpression.illegalDouble(new Random(seed));
		if (this.compiled==false) {
			throw new Exception("The regular expression must be legal and properly compiled before this operation");
		}
		for (int i=0; i<this.NUMB_OF_ILLEGAL_TRIES; i++) {
			String res = this.randomIllegalString(this.compilationResult,new Random(seed+i*i));
			if (res.length()<mini) res=this.addChars(res,mini-res.length(),new Random(seed+i*i));
			if (this.pattern.matcher(res).matches()==false) return res;
		}
		return null;
	}
	
	
	/**
	 * Tries to create a random legal String
	 * 
	 * @param mini Minimum number of chars of the illegal String
	 * @return The possible illegal String
	 * @throws Exception An exception will be thrown if the regular expression was not properly expressed
	 */
	String randomIllegalString (int mini) throws Exception {
		
		if (this.regExpInteger) return RegularExpression.illegalInteger(new Random());
		if (this.regExpDouble) return RegularExpression.illegalDouble(new Random());
		if (this.compiled==false) {
			throw new Exception("The regular expression must be legal and properly compiled before this operation");
		}
		for (int i=0; i<this.NUMB_OF_ILLEGAL_TRIES; i++) {
			String res = this.randomIllegalString(this.compilationResult,new Random());
			if (res.length()<mini) res=this.addChars(res,mini-res.length(),new Random());
			if (this.pattern.matcher(res).matches()==false) return res;
		}
		return null;
	}
	
	
	/**
	 * Checks if the String 'str' matches this regular expression
	 * 
	 * @param str String to be checked
	 * @return 'true' if it matches
	 * 		   'false' otherwise
	 */
	boolean matches (String str) {
		
		try {
			if (this.regExpInteger) {
				(new Integer(str)).intValue();
				return true;
			}
			if (this.regExpDouble) {
				(new Double(str)).doubleValue();
				return true;
			}
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in RegularExpression.matches");
			return false;
		}
		return this.pattern.matcher(str).matches();
	}
	
	
	/**
	 * Tries to create a random illegal String taking as initial point 'comExp'
	 * 
	 * @param compExp The initial point
	 * @param generator Random generator
	 * @return A possible random illegal String
	 */
	private String randomIllegalString (Vector compExp, Random generator) {
		
		if (this.regExpInteger) return RegularExpression.illegalInteger(generator);
		if (this.regExpDouble) return RegularExpression.illegalDouble(generator);
		if (compExp.size()==0) return "";
		
		String res = new String();
		
		int len = compExp.size()-1;
		for (int n=generator.nextInt(len), j=0; j<len; j++, n++) {
			n%=len;
			Vector group = (Vector)compExp.elementAt(n);
			for (int i=0; i<group.size(); i++) {
				
				ExpressionSet expSet = (ExpressionSet)group.elementAt(i);
				for (int rep=generator.nextInt(3); rep>0; rep--) {
					if (expSet.getCharSet()!=null) {
						char c = tryIllegalChar(expSet.getExists(),generator);
						if (c!=0) res+=c;
					}
					else {
						res+=this.randomIllegalString(expSet.getGroup(),generator);
					}
				}
			}
		}
		
		return res;
	}
	
	
	/**
	 * Data structure used as representation of the regular expression
	 * 
	 * @author I054742
	 *
	 */
	private class ExpressionSet {
		
		/* It will contain independent parts of a regular expression as the ones between parenthesis */
		Vector group = null;
		/* Charset of a portion of a regular expression */
		String charSet = null;
		/* Quantizer of this part of the regular expression */
		char quantizer='0';
		/* Minimum and maximum number of repetitions of this part of the regular expression */
		int mini, maxi;
		/* Set of existing chars in this part of the regular expression */
		boolean exists[] = new boolean[NUMB_OF_DIFF_CHARS];
		
		
		/**
		 * Marks the set of chars of this ExpressionSet
		 */
		private void mark () {
			
			for (int i=0; i<NUMB_OF_DIFF_CHARS; i++) this.exists[i]=false;
			if (this.charSet!=null) {
				for (int i=0; i<this.charSet.length(); i++) {
					this.exists[this.charSet.charAt(i)]=true;
				}
			}
		}
		
		
		/**
		 * Gives the group of this ExpressionSet
		 * 
		 * @return The group
		 */
		private Vector getGroup () {
			
			return this.group;
		}
		
		
		/**
		 * Gives the char set of this ExpressionSet
		 * 
		 * @return The char set
		 */
		private String getCharSet () {
			
			return this.charSet;
		}
		
		
		/**
		 * Gives the quantizer of this ExpressionSet
		 * 
		 * @return The quantizer
		 */
		private char getQuantizer () {
			
			return this.quantizer;
		}
		
		
		/**
		 * Gives the minimum number of repetitions of this ExpressionSet
		 * 
		 * @return The minimum number of repetitions
		 */
		private int getMini () {
			
			return this.mini;
		}
		
		
		/**
		 * Gives the maximum number of repetitions of this ExpressionSet
		 * 
		 * @return The maximum number of repetitions
		 */
		private int getMaxi () {
			
			return this.maxi;
		}
		
		
		/**
		 * Gives the existing set of chars of this ExpressionSet
		 * 
		 * @return The set of existing chars
		 */
		private boolean[] getExists () {
			
			return this.exists;
		}
		
		
		/**
		 * Sets the group of this ExpressionSet to 'aGroup'
		 * 
		 * @param aGroup The new group
		 */
		private void setGroup (Vector aGroup) {
			
			this.group=aGroup;
		}
		
		
		/**
		 * Sets the char set of this ExpressionSet to 'aCharSet'
		 * 
		 * @param aCharSet The new char set
		 */
		private void setCharSet (String aCharSet) {
			
			this.charSet=aCharSet;
		}
		
		
		/**
		 * Sets the quantizer of this ExpressionSet to 'aQuantizer'
		 * 
		 * @param aQuantizer The new quantizer
		 */
		private void setQuantizer (char aQuantizer) {
			
			this.quantizer=aQuantizer;
		}
		
		
		/**
		 * Sets the minimum and maximum number of repetitions of this ExpressionSet to 'aMini' and 'aMaxi' respectively
		 * 
		 * @param aMini The minimum number of repetitions
		 * @param aMaxi The maximum number of repetitions
		 */
		private void setMiniMaxi (int aMini, int aMaxi) {
			
			this.mini=aMini;
			this.maxi=aMaxi;
		} 
	}
	
	
	/**
	 * Translate this regular expression to the one used by java.util.regex.*;
	 * 
	 * @param str The regular expression to be translated
	 * @return The translated regular expression
	 */
	private String translate (String str) {
		
		
		String originals[] = {"\\\\a", "\\\\A", "\\\\\\.", "\\.", "\\\\h"};
		String translations[] = {"\\\\p{Alpha}", "[^\\\\p{Alpha}]", "\\\\h", "\\\\.", "."};
		String res = str.substring(0);
		
		for (int i=0; i<originals.length; i++) {
			res=(Pattern.compile(originals[i])).matcher(res).replaceAll(translations[i]);
		}
		
		return res;
	}

	
///////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////// TESTING METHODS ///////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////
	
	
	static void main (String[] args) {
		
		/*for (char c=1; c<256; c++) {
			
			if (Character.isLetter(c)) {
				System.out.println(c+" -> "+(int)c);
			}
		}*/
		
		String regExpTrans[] = {
								"\\a..\\A\\.*|(kio)?",
								"\\A",
								"\\.",
								"."
		};
	
		Pattern p = Pattern.compile(".*", Pattern.DOTALL);
		System.out.println(p.matcher("¬Sôäz\r ").matches());
		
		if (true) return;
		
		RegularExpression rExp = new RegularExpression();
		
		for (int i=0; i<regExpTrans.length; i++) {
			
			System.out.println(regExpTrans[i]+" -> "+rExp.translate(regExpTrans[i]));
		}
		
		
		String regExp[] = {/*"[a-z0-9]+@sap.nor*.com",
						   "\\(\\)(ab|cd|01)*",
						   "[(\\)\\\\]*,[-a-b]+,[7-9-]+,[a1-3]+",
						   "-?[1-9][0-9]*|0",
						   "(a[b-y]*z|A[B-Y]*Z)*",
						   "(file=[a-zA-Z]+[a-zA-Z.0-9]*,)?(dir=[a-zA-Z]+[a-zA-Z.0-9]*,)root=[a-cA-Z]",
						   "(ab)*"*/
				/*"INTEGER",
				"DOUBLE",
				"BIG_INTEGER",
				"BIG_DOUBLE",
				"CHAR",
				"ALPHA",
				"NUMERIC",*/
				"TELEPHONE",
				"STRING",
				"ALPHANUMERIC"
		};
		
		
		for (int i=0; i<regExp.length; i++) {
			
			rExp = RegularExpression.getRegExpFromId(regExp[i]);
			boolean compiled = rExp!=null;
			//boolean compiled = rExp.compile(regExp[i]);
			System.out.println("Evaluating "+regExp[i]);
			if (compiled==false) {
				System.out.println("--null");
			}
			else {
				for (int k=0; k<5; k++) {
					//System.out.println("--Random string: "+rExp.randomLegalString(compExp,new Random()));
					try {
						String s;
						System.out.println("--Random Legal string: "+(s=rExp.randomLegalString())+" -> "+Pattern.matches(rExp.translated,s));
						System.out.println("--Random Illegal string: "+rExp.randomIllegalString(5));
					}
					catch (Exception exc) {
						if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in RegularExpression.main");
						System.out.println(exc.getMessage());
					}
				}
			}
			System.out.println();
		}	
	}
}
