import java.util.Iterator;
import java.util.Vector;

import com.sap.idm.vds.MVDSearchResultEntry;
import com.sap.idm.vds.MVDSearchResults;


/**
 * Class in charge of checking the VDS connection and that the initial state fits with the specifications given in the configuration files
 * 
 * @author I054742
 *
 */
public class Preprocessing {

	static final int VIRTUAL_DN = 1;
	static final int COMPLETE_DN = 2;
	static final int INVALID_DN = -1;
	static private int lastDnCheck;
	static int getLastDnCheck () {return lastDnCheck;}
	
	static boolean connect () {
		
		Loggin.Log(Globals.indent()+"Connecting with VDS...");
		Globals.incIndent();
		
		try {
			Globals.mServer=GeneralConfiguration.getServer();
			Globals.mPort=GeneralConfiguration.getPort();
			Globals.mUser=GeneralConfiguration.getUser();
			Globals.mPassword=GeneralConfiguration.getPassword();
			Globals.szLimit=GeneralConfiguration.getSizeLimit();
			Globals.tmLimit=GeneralConfiguration.getTimeLimit();
			Globals.securityOption=GeneralConfiguration.getSecurityOption();
			Globals.ldapm = new LDAPManagement(Globals.mServer,Globals.mPort,Globals.mUser,Globals.mPassword,Globals.securityOption);
			
			Loggin.Log(Globals.indent()+"Successful connection");
			return true;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Preprocessing.connect");
			Loggin.Log(Globals.indent()+"ERROR: Error while trying to connect to VDS"+(exc.getMessage()==null?"":" -> "+exc.getMessage()));
			return false;
		}
		finally {
			Globals.decIndent();
			Loggin.Log(Globals.indent()+"Connection process finished");
		}
	}
	
	
	/**
	 * Checks if the initial state is legal according to the configuration files information
	 * @return 'true' if the initial state is legal according to the configuration files information
	 * 		   'false' otherwise
	 */
	static boolean initialChecking () {
		
		Loggin.Log(Globals.indent()+"Checking if the initial state is legal according to the configuration files information...");
		Globals.incIndent();
		
		try { 
			
			boolean res = initialChecking(GeneralConfiguration.getStartingPoint());
			
			if (res) Loggin.Log(Globals.indent()+"No error was found during the initial checking proccess");
			else Loggin.Log(Globals.indent()+"ERROR: Error in initial checking process");
			
			return res;
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Preprocessing.initialChecking");
			Loggin.Log(Globals.indent()+"ERROR: Error in initial checking process");
			return false;
		}
		finally {
			Globals.decIndent();
			Loggin.Log(Globals.indent()+"Initial checking finished");
		}
	}
	
	
	static private int isLegalDn (Vector rDnParts, int i_rDn, String[] dnParts, int i_dn, String partialDn, boolean checkVals, boolean login) {
		
		if (i_dn<0) return VIRTUAL_DN;
		if (i_rDn<0 && i_dn<0) return COMPLETE_DN;
		if (i_rDn<0 || i_dn<0) return INVALID_DN;
		Vector part = (Vector)rDnParts.elementAt(i_rDn);
		String dnPart[] = GenericMethods.collectStrings(dnParts[i_dn],' ',' ',"=");
		String idP = (String)part.elementAt(EntryDefinition.L_DN_ID);
		String idD = dnPart[0];
		if (idP.equalsIgnoreCase(idD)==false) return INVALID_DN;
		String val = dnPart[1];
		Vector regExps = (Vector)part.elementAt(EntryDefinition.L_DN_RE);
		Vector atts = (Vector)part.elementAt(EntryDefinition.L_DN_ATTS);
		if (partialDn.length()>0) partialDn=dnParts[i_dn]+","+partialDn;
		else partialDn=dnParts[i_dn];
		if (checkVals) {
			if (atts.size()==0) {
				RegularExpression re = (RegularExpression)regExps.elementAt(0);
				if (re.matches(val)==false) {
					Loggin.Log(Globals.indent()+"ERROR: The value: \""+val+"\" of the entry \""+partialDn+"\" is not legal");
					return INVALID_DN;
				}
			}
			else {
				String expectVal = new String();
				Entry e = new Entry(partialDn,true,true);
				boolean skipValCompareTest = false;
				String completeOrigRegExp = new String();
				for (int i=0; i<atts.size(); i++) {
					String attName = (String)atts.elementAt(i);
					boolean isMandatory = false;
					try {
						isMandatory = Globals.hasPropertyByRdnId(e.getRdnId(),attName,Globals.PROP_MUST);
					}
					catch (Exception exc) {
						if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in Preprocessing.isLegalDn");
						return INVALID_DN;
					}
					Vector attVals = e.getAttrValue(attName);
					RegularExpression re = (RegularExpression)regExps.elementAt(i);
					if (completeOrigRegExp.length()>0) completeOrigRegExp+=Globals.SEP_BETWEEN_ATT_VALS;
					completeOrigRegExp+=re.getOrigRegExp();
					if (attVals==null) {
						if (isMandatory) {
							if (login) Loggin.Log(Globals.indent()+"ERROR: The mandatory attribute: \""+attName+"\" of the entry \""+partialDn+"\" has no value");
							return INVALID_DN;
						}
						else {
							skipValCompareTest = true;
							continue;
						}
					}
					String attVal = new String();
					for (int j=0; j<attVals.size(); j++) {
						String partialVal = (String)attVals.elementAt(j);
						if (re.matches(partialVal)==false) {
							if (login) Loggin.Log(Globals.indent()+"ERROR: The value: \""+partialVal+"\" of the entry \""+partialDn+"\" is not legal");
							return INVALID_DN;
						}
						if (j>0) attVal+=MyAttribute.ATT_VALS_SEP;
						attVal+=partialVal;
					}
					if (expectVal.length()>0) expectVal+=Globals.SEP_BETWEEN_ATT_VALS;
					expectVal+=attVal;
				}
				if (skipValCompareTest==false) {
					if (expectVal.equalsIgnoreCase(val)==false) {
						if (login) Loggin.Log(Globals.indent()+"ERROR: The values of the entry \""+partialDn+"\" are not the expected ones according to the specifications of the configuration files");
						return INVALID_DN;
					}
				}
				else {
					RegularExpression completeRe = new RegularExpression();
					boolean compileRes = completeRe.compile(completeOrigRegExp);
					if (compileRes==false) {
						if (login) Loggin.Log(Globals.indent()+"WARNING: Not possible to make 100% sure that the initial configuration is complete within the initial specification specified into the configuration files");
					}
					else {
						if (completeRe.matches(val)==false) {
							if (login) Loggin.Log(Globals.indent()+"ERROR: The DN of the entry \""+partialDn+"\" does not match with the one specified into the configuration files");
							return INVALID_DN;
						}
					}
				}
			}
		}
		String symb = (String)part.elementAt(EntryDefinition.L_DN_SYMB);
		boolean tries[] = {false,false,false};
		int n = 1+(symb.equals("+")?1:0)+((symb.equals("*") || symb.equals("?"))?2:0);
		int res = INVALID_DN;
		for (int u=0; res!=COMPLETE_DN && u<n; u++) {
			int x = Globals.randGen.nextInt(n);
			while (tries[x]) x=Globals.randGen.nextInt(n);
			tries[x]=true;
			int temp = 0;
			if (x==0) temp = isLegalDn(rDnParts,i_rDn-1,dnParts,i_dn-1,partialDn,checkVals,login);
			else if (x==1 && symb.equals("?")==false) temp = isLegalDn(rDnParts,i_rDn,dnParts,i_dn-1,partialDn,checkVals,login);
			else temp = isLegalDn(rDnParts,i_rDn-1,dnParts,i_dn,partialDn,checkVals,login);
			res = Math.max(res,temp);
		}
		return res;
	}
	
	
	static String isLegalDn (String dn, boolean checkVals, boolean login) {
		
		String dnParts[] = GenericMethods.collectStrings(dn,' ',' ',",");
		MyHashMap eDefByRdnId = Globals.getEntDefByRDNID();
		int tempRes = INVALID_DN;
		Vector possibRes = new Vector();
		for (Iterator it=eDefByRdnId.keySet().iterator(); tempRes!=COMPLETE_DN && it.hasNext(); ) {
			String rDnId = (String)it.next();
			EntryDefinition eDef = (EntryDefinition)eDefByRdnId.get(rDnId);
			Vector rDnParts = eDef.getRdnParts();
			int partRes = isLegalDn(rDnParts,rDnParts.size()-1,dnParts,dnParts.length-1,new String(),checkVals,login);
			if (partRes>tempRes) {
				tempRes = partRes;
				possibRes = new Vector();
				possibRes.add(rDnId);
			}
			else if (partRes==tempRes && partRes>INVALID_DN) {
				
			}
		}
		lastDnCheck = tempRes;
		if (possibRes.size()==0) return null;
		return (String)possibRes.elementAt(Globals.randGen.nextInt(possibRes.size()));
	}
	
	
	static String getAssociatedRdnId (String dn, boolean checkVals, boolean login) {
		
		return isLegalDn(dn,checkVals,login);
	}
	
	
	/**
	 * Checks the legality of one entry
	 * @param dn The entry DN
	 * @return 'true' if the entry is legal
	 * 		   'false' otherwise
	 */
	static private boolean checkIfEntryIsLegal (String dn, boolean checkVals, boolean login) {
		
		String rDnId = isLegalDn(dn,checkVals,login);
		if (rDnId==null) {
			Loggin.Log(Globals.indent()+"ERROR: The entry with DN: \""+dn+"\" has a DN which does not match with any legal DN specification");
			return false;
		}
		
		if (lastDnCheck!=COMPLETE_DN) return true;
		
		EntryDefinition eDef = Globals.getEntDefFromRDNID(rDnId);
		String eId = eDef.getId();
		/* Creates the attribute set for each entry */
		Entry e = new Entry(dn);
		for (int k=0; k<e.getNumbOfAttrs(); k++) {
			MyAttribute ma = e.getAttributeAt(k);
			if (Globals.existsAtt(eId,ma.getName())==false) {
				Loggin.Log(Globals.indent()+"WARNING: The attribute: \""+ma.getName()+"\" of the entry: \""+e.getDn()+"\" has not definition");
			}
			else {
				for (int j=0; j<ma.getNumbOfValues(); j++) {
					String val = ma.getValueAt(j);
					if (Globals.valIsLegal(eId,ma.getName(),val)==false) {
						Loggin.Log(Globals.indent()+"ERROR: The value: \""+val+"\" of the attribute: \""+ma.getName()+"\" of the entry: "+e.getDn()+" is an illegal value");
						return false;
					}
				}
			}
		}
		
		return true;
	}
	
	
	/**
	 * Checks if every entry and every attribute is legal
	 * @param sp Initial starting point
	 * @return 'true' if everything is legal
	 * 		   'false' otherwise
	 */
	static private boolean initialChecking (String sp) {
		
		MVDSearchResults sr = Globals.ldapm.search(sp,LDAPManagement.ONE_LEVEL,Globals.attr,
												   Globals.urlfilter,Globals.szLimit,Globals.tmLimit,false);
		/* If the call to this method is for the initial starting point then we will check this node independently */
		if (sp.equalsIgnoreCase(GeneralConfiguration.getStartingPoint())) {
			if (checkIfEntryIsLegal(sp,true,true)==false) return false;
		}
		if (sr==null) return true;
		/* For each of the entries directly under this starting point we will check their legality */
		for (int i=0; i<sr.size(); i++) {
			
			MVDSearchResultEntry me = (MVDSearchResultEntry)sr.elementAt(i);

			if (checkIfEntryIsLegal(me.getDn(),true,true)==false) return false;
			
			if (initialChecking(me.getDn())==false) return false;
		}
		return true;
	}
}
