import java.util.Vector;


/**
 * This class will parse every instruction of the instructions file 'Instructions.inst'. It will check the correctness of every instruction and it will create a special data structure for each one of them
 * @author I054742
 *
 */
public class Instruction {
	
	/* The different instruction types */
	static final String ADD = "add";
	static final String DEL = "del";
	static final String MOD_ADD = "mod_add";
	static final String MOD_DEL = "mod_del";
	static final String MOD_REP = "mod_rep";
	
	/* Will hold the original instruction line */
	private String original = null;
	/* Will hold the number of line where the instruction is into the file */
	private int nLine = -1;
	/* Instruction type */
	private String type = null;
	/* The instruction DN */
	private String dn = null;
	/* The instruction attributes */
	private Vector values = new Vector();
	/* An entry with the instruction DN and the instruction attributes */
	private Entry entry = null;
	
	/* They define any legal entry */
	static private final String ATTR_DEF = "\\{\\[[a-zA-Z_]+=\\.*\\]\\}";
	static private final String ATTRS_DEF = "\\{((\\[[a-zA-Z_]+=\\.*\\],)*\\[[a-zA-Z_]+=\\.*\\])?\\}";
	static private final String DN_DEF = "(\\.+=\\.+,)*\\.+=\\.+";
	static private final String MOD_OPTS = "("+MOD_ADD+"|"+MOD_DEL+"|"+MOD_REP+"|"+")";
	static private final String ADD_DEF = ADD+"[ ]+"+DN_DEF+"[ ]+"+ATTRS_DEF;
	static private final String DEL_DEF = DEL+"[ ]+"+DN_DEF;
	static private final String MOD_DEF = MOD_OPTS+"[ ]+"+DN_DEF+"[ ]+"+ATTR_DEF;
	static private final String INST_DEF = "("+ADD_DEF+"|"+DEL_DEF+"|"+MOD_DEF+"|"+")";
	
	/* Regular expression used to identify any legal instruction */
	static private RegularExpression addRegDef = RegularExpression.getRegExpFromId(ADD_DEF);
	static private RegularExpression delRegDef = RegularExpression.getRegExpFromId(DEL_DEF);
	static private RegularExpression modRegDef = RegularExpression.getRegExpFromId(MOD_DEF);
	static private RegularExpression instRegDef = RegularExpression.getRegExpFromId(INST_DEF);
	
	
	
	/**
	 * Makes an initial parsing of the instruction 'inst' and split it in its different parts
	 * @param inst The instruction
	 * @return One vector with the parts of the instruction
	 */
	static private Vector splitInstruction (String inst, boolean log) {
		
		inst = GenericMethods.cleanTrailingAndLeadingSpaces(inst);
		
		if (instRegDef.matches(inst.toLowerCase())==false) {
			
			if (log) Loggin.Log(Globals.indent()+"ERROR: The instruction: "+inst+" does not match with any legal instruction definition");
			return null;
		}
		
		int index = inst.indexOf(" ");
		if (index<0) return null;
		String command = inst.substring(0,index);
		command = GenericMethods.cleanTrailingAndLeadingSpaces(command.toLowerCase());
		
		inst = GenericMethods.cleanTrailingAndLeadingSpaces(inst.substring(index));
		
		String dn = null;
		String vals = null;
		if (command.equalsIgnoreCase(DEL)) {
			dn = inst;
			if (dn.length()==0) {
				return null;
			}
		}
		else {
			index = inst.indexOf("{");
			if (index<0) return null;
			dn = inst.substring(0,index);
			dn = GenericMethods.cleanTrailingAndLeadingSpaces(dn);
		
			inst = GenericMethods.cleanTrailingAndLeadingSpaces(inst.substring(index));
			
			vals = null;
			if (inst.length()>0) {
				if (inst.startsWith("{")==false) return null;
				if (inst.endsWith("}")==false) return null;
				vals = GenericMethods.cleanBrackets(inst,'{','}');
				vals = GenericMethods.cleanTrailingAndLeadingSpaces(vals);
			}
		}
		
		Vector res = new Vector();
		
		res.add(command);
		res.add(dn);
		res.add(vals);
		
		return res;
	}
	
	
	/**
	 * Creates the data structure for an add entry instruction
	 * @param parts The instruction parts resulting of the first parsing
	 * @param orig The original instruction
	 * @param nLine The number of line where the instruction is into the instructions file
	 * @return One Instruction object
	 */
	static private Instruction makeAddInstruction (Vector parts, String orig, int nLine, boolean log) {
		
		String command = (String)parts.elementAt(0);
		if (command.equalsIgnoreCase(ADD)==false) return null;
		String dn = (String)parts.elementAt(1);
		if (dn.indexOf("=")<0) {
			if (log) Loggin.Log(Globals.indent()+"ERROR: Bad formed DN: "+dn);
			return null;
		}
		String values = (String)parts.elementAt(2);
		Instruction res = new Instruction();
		if (values==null) {
			if (log) Loggin.Log(Globals.indent()+"ERROR: Values missed in add operation");
			return null;
		}
		if (values.length()>0) {
			if (values.endsWith(",")) {
				if (log) Loggin.Log(Globals.indent()+"ERROR: Total blank attribute");
				return null;
			}
			String vals[] = GenericMethods.collectStrings(values,'[',']',"]");
			for (int i=0; i<vals.length; i++) {
				String val = GenericMethods.cleanTrailingAndLeadingSpaces(vals[i]);
				int index = val.indexOf("=");
				if (index<0) {
					if (log) Loggin.Log(Globals.indent()+"ERROR: Bad formed attribute: "+val);
					return null;
				} 
				String attrName = GenericMethods.cleanTrailingAndLeadingSpaces(val.substring(0,index));
				if (attrName.length()==0) {
					if (log) Loggin.Log(Globals.indent()+"ERROR: Bad formed attribute: "+val);
					return null;
				}
				String attrValue = GenericMethods.cleanTrailingAndLeadingSpaces(val.substring(index+1,val.length()));
				Vector v = new Vector();
				String justVals[] = GenericMethods.collectStrings(attrValue,' ',' ',"|");
				for (int j=0; j<justVals.length; j++) {
					v.add(justVals[j]);
				}
				res.addParameter(new MyAttribute(attrName,v,null));
			}
		}
		res.setOriginal(orig);
		res.setNLine(nLine);
		res.setDn(dn);
		res.setType(command);
		res.setEntry(new Entry(dn,res.getAttributes(),true));
		
		return res;
	}
	
	
	/**
	 * Creates the data structure for an delete entry instruction
	 * @param parts The instruction parts resulting of the first parsing
	 * @param orig The original instruction
	 * @param nLine The number of line where the instruction is into the instructions file
	 * @return One Instruction object
	 */
	static private Instruction makeDelInstruction (Vector parts, String orig, int nLine, boolean log) {
		
		String command = (String)parts.elementAt(0);
		if (command.equalsIgnoreCase(DEL)==false) return null;
		String dn = (String)parts.elementAt(1);
		if (dn.indexOf("=")<0) {
			if (log) Loggin.Log(Globals.indent()+"ERROR: Bad formed DN: "+dn);
			return null;
		}
		String values = (String)parts.elementAt(2);
		if (values!=null) {
			if (log) Loggin.Log(Globals.indent()+"ERROR: Values missed in add operation");
			return null;
		}
		Instruction res = new Instruction();
		res.setOriginal(orig);
		res.setNLine(nLine);
		res.setDn(dn);
		res.setType(command);
		
		return res;
	}
	
	
	/**
	 * Creates the data structure for an modification attribute instruction
	 * @param parts The instruction parts resulting of the first parsing
	 * @param orig The original instruction
	 * @param nLine The number of line where the instruction is into the instructions file
	 * @return One Instruction object
	 */
	static private Instruction makeModInstruction (Vector parts, String orig, int nLine, boolean log) {
		
		String command = (String)parts.elementAt(0);
		if (command.equalsIgnoreCase(MOD_ADD)==false && command.equalsIgnoreCase(MOD_DEL)==false && command.equalsIgnoreCase(MOD_REP)==false) return null;
		String dn = (String)parts.elementAt(1);
		if (dn.indexOf("=")<0) {
			if (log) Loggin.Log(Globals.indent()+"ERROR: Bad formed DN: "+dn);
			return null;
		}
		String values = (String)parts.elementAt(2);
		if (values==null) {
			if (log) Loggin.Log(Globals.indent()+"ERROR: Values missed in modify operation");
			return null;
		}
		
		Instruction res = new Instruction();
		String val = GenericMethods.cleanBrackets(values,'[',']');
		int index = val.indexOf("=");
		if (index<0) {
			if (log) Loggin.Log(Globals.indent()+"ERROR: Bad formed attribute: "+val);
			return null;
		} 
		String attrName = GenericMethods.cleanTrailingAndLeadingSpaces(val.substring(0,index));
		if (attrName.length()==0) {
			if (log) Loggin.Log(Globals.indent()+"ERROR: Bad formed attribute: "+val);
			return null;
		}
		res.setOriginal(orig);
		res.setNLine(nLine);
		String attrValue = GenericMethods.cleanTrailingAndLeadingSpaces(val.substring(index+1,val.length()));
		Vector v = new Vector();
		String justVals[] = GenericMethods.collectStrings(attrValue,' ',' ',"|");
		for (int j=0; j<justVals.length; j++) {
			v.add(justVals[j]);
		}
		v.add(attrValue);
		res.addParameter(new MyAttribute(attrName,v,null));
		res.setDn(dn);
		res.setType(command);
		
		return res;
	}
	
	
	/**
	 * * Creates one Instruction object parting from the original instruction 'inst' which is in line 'nLine' into the instructions file 'Instructions.inst'
	 * @param inst The original instruction
	 * @param nLine The line number where the instruction is
	 * @return One Instruction object if the instruction is legal
	 * 		   A null value otherwise
	 */
	static public Instruction makeInstruction (String inst, int nLine, boolean log) {
		
		Vector parts = splitInstruction(inst,log);
		if (parts==null) return null;
		
		Instruction res = makeAddInstruction(parts,inst,nLine,log);
		if (res!=null) return res;
		res=makeDelInstruction(parts,inst,nLine,log);
		if (res!=null) return res;
		res=makeModInstruction(parts,inst,nLine,log);
		if (res!=null) return res;
		if (log) Loggin.Log(Globals.indent()+"ERROR: Illegal operation identifier: "+inst);
		return null;
	}
	
	
	private void setOriginal (String aOriginal) {
		
		this.original=aOriginal;
	}
	
	
	private void setNLine (int aNLine) {
		
		this.nLine=aNLine;
	}
	
	
	private void addParameter (Object o) {
		
		this.values.add(o);
	}
	
	
	private void setType (String aType) {
		
		this.type=aType;
	}
	
	
	private void setDn (String aDn) {
		
		this.dn=aDn;
	}
	
	private void setEntry (Entry aEntry) {
		
		this.entry=aEntry;
	}
	
	
	public String getOriginal () {
		
		return this.original;
	}
	
	
	public int getNLine () {
		
		return this.nLine;
	}
	
	
	public Vector getAttributes () {
		
		return this.values;
	}
	
	
	public String getType () {
		
		return this.type;
	}
	
	
	public String getDn () {
		
		return this.dn;
	}
	
	
	public Entry getEntry () {
		
		return this.entry;
	}
}
