import java.util.*;

/**
 * Class in charge of creating entry definitions from the set of parameters of a entry specification
 * @author I054742
 *
 */
public class EntryDefinition extends Definition {
	
	/* Identifiers for the legal RDN specifications vectors */
	static int L_DN_ID = 0;
	static int L_DN_SYMB = 1;
	static int L_DN_RE = 2;
	static int L_DN_VAL = 3;
	static int L_DN_ATTS = 4;
	static int L_DN_RDN = 5;
	static int L_DN_ACCU = 6;
	
	/* Particular identifiers */
	static final String RDN = "RDN";
	
	/* Mandatory parameters */
	static private final MyHashMap MANDATORY = new MyHashMap() {{
		put(ID,null);
	}};
	
	private Vector extend = null;
	private String rDN = null;
	private String id = null;
	
	private MyHashMap linkedAttr = null;
	private MyHashMap attrList = null;
	private Vector rDnParts = null;
	private String rDnId = null;
	private String rDnIdRegExp = null;
	private String possibEntRegExp = null;
	private RegularExpression regExpPossibEnt = null;
	
	
	/**
	 * Constructs an entry definition
	 * @param params Set of parameters of the entry specification
	 * @throws Exception
	 */
	EntryDefinition (MyHashMap params) throws Exception {
		
		/* Checks if it contains all the mandatory parameters */
		if (params.keySet().containsAll(MANDATORY.keySet())==false) {
			Loggin.Log(Globals.indent()+"ERROR: Not all the mandatory parameters were found in one of the entry definitions");
			throw new Exception();
		}
		
		/* Makes an initial parsing of the received parameters */
		MyHashMap tempTestCases = new MyHashMap();
		MyHashMap tempProperties = new MyHashMap();
		for (Iterator it=params.keySet().iterator(); it.hasNext(); ) {
			
			String key = (String)it.next();
			String val = (String)params.get(key);
			if (this.addParam(key,val,tempTestCases,tempProperties)==false) throw new Exception();
		}
		/* Checks the legality of the RDN */
		if (this.rDN!=null && (this.createLegalDNSpec(this.rDN))==false) throw new Exception();
		/* Checks the legality of the 'extend' parameter */
		if (this.extend!=null && this.checkExtend()==false) throw new Exception();
		/* Checks if the inheritance is legal */
		if (this.legalInheritance()==false) throw new Exception();
		
		this.inherit();
		this.createTestCases(tempTestCases);
		this.createProperties(tempProperties);
	}
	
	
	/**
	 * Checks if the inheritance is legal
	 * @return 'true' if there is no illegality in the inheritance parameters
	 * 		   'false' otherwise
	 */
	private boolean legalInheritance () {
		
		if (this.extend==null) return true;
		for (int i=0; i<this.extend.size(); i++) {
			String eId = (String)this.extend.elementAt(i);
			if (eId.equalsIgnoreCase(this.id)) {
				Loggin.Log(Globals.indent()+"ERROR: The entry definition with ID: \""+this.id+"\" creates recursive inheritance since it inherits itself");
				return false;
			}
			MyHashMap endDefs = Globals.getEntDefByID();
			EntryDefinition eDef = (EntryDefinition)endDefs.get(eId);
			if (eDef==null) {
				Loggin.Log(Globals.indent()+"ERROR: The entry definition with ID: \""+this.id+"\" inherits from another entry with ID: \""+eId+"\" which either is a not declared entry or is declared after the entry definition with ID: \""+this.id+"\"");
				return false;
			}
			if (eDef.getRDN()!=null) {
				Loggin.Log(Globals.indent()+"ERROR: The entry definition with ID: \""+this.id+"\" inherits the RDN of the entry with ID: \""+eDef.getId()+"\". The RDN only can be part of an entry which is not inherited by any other entry");
				return false;
			}
		}
		return true;
	}
	
	
	private void inherit () {	
	
	}
	
	
	/**
	 * 
	 * @param paramId
	 * @param paramVal
	 * @param tempTestCases
	 * @param tempProperties
	 * @return
	 */
	private boolean addParam (String paramId, String paramVal, MyHashMap tempTestCases, MyHashMap tempProperties) {
		
		if (paramId.equalsIgnoreCase(EXTENDS)) {
			this.extend = GenericMethods.fromArrStringToVector(GenericMethods.collectStrings(paramVal,'[',']',","));
		}
		else if (paramId.equalsIgnoreCase(RDN)) {
			this.rDN = GenericMethods.cleanTrailingAndLeadingSpaces(paramVal);
		}
		else if (paramId.equalsIgnoreCase(ID)) {
			this.id = GenericMethods.cleanTrailingAndLeadingSpaces(paramVal);
		}
		else if (Globals.PROPERTIES.containsKey(paramId)) {
			tempProperties.put(paramId,GenericMethods.fromArrStringToVector(GenericMethods.collectStrings(paramVal,'{','}',",")));
		}
		else {
			tempTestCases.put(paramId,paramVal);
		}
		return true;
	}
	
	
	/**
	 * 
	 * @return
	 */
	private boolean checkExtend () {
		
		if (GenericMethods.vectorContainsDuplicates(this.extend,true)) {
			Loggin.Log(Globals.indent()+"WARNING: Duplicate entry identifier(s) in inherited entries definition for entry with ID: \""+this.id+"\"");
		}
		MyHashMap eDefs = Globals.getEntDefByID();
		for (int i=0; i<this.extend.size(); i++) {
			String eId = (String)this.extend.elementAt(i);
			if (eDefs.containsKey(eId)==false) {
				Loggin.Log(Globals.indent()+"ERROR: Not visible entry identifier: \""+eId+"\" which is part of the inherited entries definition for entry with ID: \""+this.id+"\"");
				return false;
			}
		}
		return true;
	}
	
	
	/**
	 * 
	 * @param rDn
	 * @return
	 */
	private boolean createLegalDNSpec (String rDn) {
		
		String ids[] = GenericMethods.collectStrings(rDn,' ',' ',",");
		String accum = new String();
		this.rDnParts = new Vector();
		for (int i=0; i<ids.length; i++) this.rDnParts.add("");
		this.rDnId = new String();
		this.rDnIdRegExp = new String();
		this.possibEntRegExp = new String();
		for (int i=ids.length-1; i>=0; i--) {
			this.possibEntRegExp = this.rDnIdRegExp;
			String idVal[]=GenericMethods.collectStrings(ids[i],' ',' ',"=");
			String id = idVal[0];
			String val = idVal[1];
			String symbol = "";
			if (val.endsWith("*") || val.endsWith("+") || val.endsWith("?")) {
				symbol=val.substring(val.length()-1);
				val=val.substring(0,val.length()-1);
			}
			if (accum.length()>0) accum+=Globals.ACC_SEP;
			accum+=id;
			this.rDnId=accum;
			Vector dnPart = new Vector();
			dnPart.add(id);
			dnPart.add(symbol);
			dnPart.add("regExps");
			dnPart.add(val);
			dnPart.add("attrList");
			dnPart.add(rDn);
			dnPart.add(accum);
			Vector realDnPart = null;
			if ((realDnPart=Globals.addLegalDNSpec(accum,dnPart))==null) return false;
			this.rDnParts.setElementAt(realDnPart,i);
			if (this.rDnIdRegExp.length()==0) {
				if (symbol.equals("?") || symbol.equals("*")) {
					Loggin.Log(Globals.indent()+"ERROR: The RDN definition of the entry definition with ID: \""+this.id+"\" is illegal since in the right most has a symbol '*' or '?'");
					return false;
				}
				if (symbol.equals("+")) id = id + "(,"+id+")*";
				this.rDnIdRegExp = id;
			}
			else {
				this.rDnIdRegExp += "(,"+id+")"+symbol;
			}
			if (symbol.equals("*") || symbol.equals("+")) this.possibEntRegExp = this.rDnIdRegExp;
		}
		if ((this.regExpPossibEnt = RegularExpression.getRegExpFromId(this.possibEntRegExp))==null) {
			Loggin.Log(Globals.indent()+"ERROR: The RDN definition of the entry definition with ID: \""+this.id+"\" is illegal since in the right most has a symbol '*' or '?'");
			return false;
		}
		return true;
	}
	
	
	/**
	 * Creates the list of attributes of this entry
	 * @param temp
	 */
	private void createInitialAttrListAndPropList (MyHashMap temp) {
		
		this.attrList = new MyHashMap();
		this.properties = new MyHashMap();
		
		for (Iterator it=temp.keySet().iterator(); it.hasNext(); ) {
			
			String prop = (String)it.next();
			Vector v = (Vector)temp.get(prop);
			for (int i=0; i<v.size(); i++) {
				String attrName = (String)v.elementAt(i);
				if (this.attrList.containsKey(attrName)==false) {
					this.attrList.put(attrName,"");
					this.properties.put(attrName,new MyHashMap());
				}
				MyHashMap props = (MyHashMap)this.properties.get(attrName);
				props.put(prop,"yes");
			}
		}
	}
	

	/**
	 * 
	 * @param temp
	 */
	private void createProperties (MyHashMap temp) {
		
		this.createInitialAttrListAndPropList(temp);
		if (this.extend!=null) {
			MyHashMap eDefs = Globals.getEntDefByID();
			for (int i=this.extend.size()-1; i>=0; i--) {
				String eId = (String)this.extend.elementAt(i);
				EntryDefinition eDef = (EntryDefinition)eDefs.get(eId);
				MyHashMap properties = eDef.getProperties();
				for (Iterator it=properties.keySet().iterator(); it.hasNext(); ) {
					String attrName = (String)it.next();
					if (this.attrList.containsKey(attrName)==false) {
						this.attrList.put(attrName,"");
						this.properties.put(attrName,new MyHashMap());
					}
					MyHashMap extendedProps = eDef.getPropertiesOfAttr(attrName);
					MyHashMap props = (MyHashMap)this.properties.get(attrName);
					props.putAll(extendedProps);
				}
			}
		}
	}
	
	
	/**
	 * 
	 * @param temp
	 */
	private void createTestCases (MyHashMap temp) {
		
		this.testCases = new MyHashMap();
		this.testCases.putAll(GeneralConfiguration.getTestCases());
		if (this.extend!=null) {
			MyHashMap eDefs = Globals.getEntDefByID();
			for (int i=this.extend.size()-1; i>=0; i--) {
				String eId = (String)this.extend.elementAt(i);
				EntryDefinition eDef = (EntryDefinition)eDefs.get(eId);
				MyHashMap tCases = eDef.getTestCases();
				this.testCases.putAll(tCases);
			}
		}
		this.testCases.putAll(temp);
	}
	
	
	MyHashMap getPropertiesOfAttr (String attrName) {
		
		return (MyHashMap)this.properties.get(attrName);
	}
	
	
	String getId () {
		
		return this.id;
	}
	
	
	String getRDN () {
		
		return this.rDN;
	}
	
	
	MyHashMap getAttrList () {
		
		return this.attrList;
	}
	
	
	String getRdnId () {
		
		return this.rDnId;
	}
	
	
	Vector getRdnParts () {
		
		return this.rDnParts;
	}
	
	
	boolean matchPossibEnt (String aRdnId) {
		
		return this.regExpPossibEnt.matches(aRdnId);
	}
	
	
	void setLinkedAttrs (Vector v) {
		
		this.linkedAttr = new MyHashMap();
		MyHashMap linkAtts = new MyHashMap();
		for (int i=0; i<v.size(); i++) {
			String attName = (String)v.elementAt(i);
			this.linkedAttr.put(attName,"");
			MyHashMap props = new MyHashMap();
			props.put(Globals.PROP_LINKED,"yes");
			linkAtts.put(attName,props);
		}
		Globals.addProps(this.id,linkAtts);
	}
}
