import java.util.Iterator;
import java.util.Vector;

/**
 * Checks the legality of attribute specifications and create attribute definitions for attribute specifications in the case where these ones are legal
 * @author I054742
 *
 */
public class AttributeDefinition extends Definition {
	
	/* Particular identifiers */
	static final String ATTRS = "attrs";
	static final String LEGAL_VALUES = "legalValues";
	
	/* Mandatory parameters */
	static private final MyHashMap MANDATORY = new MyHashMap() {{
		put(ID,"");
	}};
	
	
	/* Inherited AttributeDefinitions */
	private Vector extend = null;
	/* List of attributes related to this definition */
	private MyHashMap attrs = null;
	/* ID */
	private String id = null;
	/* Regular expression specifying the set of legal values */
	private String legalValues = null;
	
	/* RegularExpression which defines the set of legal values */
	private RegularExpression legals = null;
	/* Set of illegal values */
	private Vector illegals = null ;
	/* Definitive list of attributes (after parsing 'attrs') */
	private MyHashMap attrList = null;
	
	
	/**
	 * Constructs a new AttributeDefinition
	 * @param params List of parameters of the attribute definition
	 * @throws Exception If some illegality was found while creating the new definition
	 */
	AttributeDefinition (MyHashMap params) throws Exception {
		
		/* Checks if it contains all the mandatory parameters */
		if (params.keySet().containsAll(MANDATORY.keySet())==false) {
			Loggin.Log(Globals.indent()+"ERROR: Not all the mandatory parameters were found in the current attribute definition");
			throw new Exception();
		}
		
		/* Makes an initial parsing and preparation of the received parameters */
		MyHashMap tempTestCases = new MyHashMap();
		MyHashMap tempProperties = new MyHashMap();
		for (Iterator it=params.keySet().iterator(); it.hasNext(); ) {
			
			String key = (String)it.next();
			String val = (String)params.get(key);
			if (this.addParam(key,val,tempTestCases,tempProperties)==false) throw new Exception();
		}
		/* Checks if there is some illegality in the inherited attribute definitions */
		if (this.extend!=null && this.checkExtend()==false) throw new Exception();
		/* Tries creating a set of legal and illegal values */
		if (this.legalValues!=null && this.createLegalAndIllegalValues()==false) throw new Exception();
		this.inherit();
		/* If there is some attribute assigned to this attribute definition but there is no associated set of legal values, then this is an illegal situation */
		if (this.legalValues==null && this.attrs!=null) {
			Loggin.Log(Globals.indent()+"ERROR: There are attributes assigned to the attribute definition with ID: \""+this.id+"\" which have not any assigned legal value");
			throw new Exception();
		}
		this.createTestCases(tempTestCases);
		this.createProperties(tempProperties);
		this.createAttrsList();
	}
	
	
	/**
	 * Makes a pre-processing of the received parameters 
	 * @param paramId Parameter identifier
	 * @param paramVal Parameter value
	 * @param tempTestCases Temporary test cases
	 * @param tempProperties Temporary properties
	 * @return 'true' if no illegality was discovered
	 * 		   'false' otherwise
	 */
	private boolean addParam (String paramId, String paramVal, MyHashMap tempTestCases, MyHashMap tempProperties) {
		
		if (paramId.equalsIgnoreCase(EXTENDS)) {
			this.extend = GenericMethods.fromArrStringToVector(GenericMethods.collectStrings(paramVal,'[',']',","));
		}
		else if (paramId.equalsIgnoreCase(ATTRS)) {
			this.attrs = GenericMethods.fromArrStringToMyHashMap(GenericMethods.collectStrings(paramVal,'{','}',","));
		}
		else if (paramId.equalsIgnoreCase(ID)) {
			this.id = GenericMethods.cleanTrailingAndLeadingSpaces(paramVal);
		}
		else if (paramId.equalsIgnoreCase(LEGAL_VALUES)) {
			this.legalValues = paramVal;
		}
		else if (Globals.PROPERTIES.containsKey(paramId)) {
			tempProperties.put(paramId,paramVal);
		}
		else {
			tempTestCases.put(paramId,paramVal);
		}
		return true;
	}
	
	
	/**
	 * Develops the inheritance
	 */
	private void inherit () {
		
		if (this.legalValues!=null) return;
		if (this.extend==null) return;
		MyHashMap aDefs = Globals.getAttDefByID();
		for (int i=0; this.legalValues==null && i<this.extend.size(); i++) {
			String aId = (String)this.extend.elementAt(i);
			AttributeDefinition aDef = (AttributeDefinition)aDefs.get(aId);
			this.legalValues = aDef.getLegalValues();
			this.legals = aDef.getLegals();
			this.illegals = aDef.getIllegals();
		}
	}
	
	
	/**
	 * Checks if there is some irregularity in the parameter 'extend'
	 * @return 'true' if there is no irregularity
	 * 		   'false' otherwise
	 */
	private boolean checkExtend () {
		
		if (GenericMethods.vectorContainsDuplicates(this.extend,true)) {
			Loggin.Log(Globals.indent()+"WARNING: Duplicate attribute identifier(s) in inherited attributes definition for attribute with ID: \""+this.id+"\"");
		}	
		MyHashMap aDefs = Globals.getAttDefByID();
		for (int i=0; i<this.extend.size(); i++) {
			String aId = (String)this.extend.elementAt(i);
			if (aDefs.containsKey(aId)==false) {
				Loggin.Log(Globals.indent()+"ERROR: Either not existing attribute definition with ID: \""+aId+"\" which is part of the inherited attributes definition for attribute with ID: \""+this.id+"\", or the definition of the inherited one is after the one that wish to inherit");
				return false;
			}
		}
		return true;
	}
	
	
	/**
	 * Creates a list of attributes which have this attribute definition as their definition
	 * This will be a map [attrName[eId,Object]]
	 * @throws Exception
	 */
	private void createAttrsList () throws Exception {
		
		// NOTE The attributes are not inherited ;)
		this.attrList = new MyHashMap();
		if (this.attrs==null) return;
		for (Iterator it=this.attrs.keySet().iterator(); it.hasNext(); ) {
			String attrId = (String)it.next();
			String attrName = attrId;
			String eId = "*";
			int ind = attrId.indexOf('@');
			if (ind>=0) {
				attrName = attrId.substring(0,ind);
				eId = attrId.substring(ind+1);
			}
			/* Checks if an attribute is part of some entry definition */
			MyHashMap propByAtt = Globals.getPropsByAtt();
			if (propByAtt.containsKey(attrName)==false) {
				Loggin.Log(Globals.indent()+"ERROR: The attribute: \""+attrName+"\" is not part of any entry definition. The error was found in the attribute definition with ID: \""+this.id+"\"");
				throw new Exception();
			}
			MyHashMap ents = (MyHashMap)propByAtt.get(attrName);
			MyHashMap entList = new MyHashMap();
			if (eId.equals("*")) {
				entList.putAll(ents);
			}
			else {
				/* Checks if the entry identifier associated to an attribute exists */
				if (ents.containsKey(eId)==false) {
					Loggin.Log(Globals.indent()+"ERROR: The entry identifier: \""+eId+"\" associated with the attribute: \""+attrName+"\" does not exist. The error was found in the attribute definition with ID: \""+this.id+"\"");
					throw new Exception();
				}
				entList.put(eId,"");
			}
			this.attrList.put(attrName,entList);
		}
	}
	
	
	/**
	 * Creates the RegularExpression for the legal values and a set of illegal values
	 * @return 'true' if no irregularity was found
	 * 		   'false' otherwise
	 */
	private boolean createLegalAndIllegalValues () {
		
		this.legals = Globals.getRegExp(this.legalValues);
		if (this.legals==null) {
			Loggin.Log(Globals.indent()+"ERROR: Legal values: \""+this.legalValues+"\" of the attribute definition with ID: \""+this.id+"\" has not related regular expression");
			return false;
		}
		this.illegals = new Vector();
		for (int i=0; i<100; i++) {
			try {
				String str = this.legals.randomIllegalString(Globals.randGen.nextInt(GeneralConfiguration.getSeed()),10);		
				if (str!=null) this.illegals.add(str);
			}
			catch (Exception exc) {
				if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in AttributeDefinition.createLegalAndIllegalValues");
			}
		}
		return true;
	}
	
	
	/**
	 * Creates the definitive set of properties having into account both the specific ones for this attribute definition and the inherited ones
	 * @param temp Set of properties specially specified for this attribute definition
	 */
	private void createProperties (MyHashMap temp) {
		
		this.properties = new MyHashMap();
		if (this.extend!=null) {
			MyHashMap aDefs = Globals.getAttDefByID();
			for (int i=this.extend.size()-1; i>=0; i--) {
				String aId = (String)this.extend.elementAt(i);
				AttributeDefinition aDef = (AttributeDefinition)aDefs.get(aId);
				MyHashMap props = aDef.getProperties();
				this.properties.putAll(props);
			}
		}
		this.properties.putAll(temp);
	}
	
	
	/**
	 * Creates the definitive set of test cases having into account both the specific ones for this attribute definition and the inherited ones
	 * @param temp Set of test cases specially specified for this attribute definition
	 */
	private void createTestCases (MyHashMap temp) {
		
		this.testCases = new MyHashMap();
		//this.testCases.putAll(GeneralConfiguration.getTestCases());
		if (this.extend!=null) {
			MyHashMap aDefs = Globals.getAttDefByID();
			for (int i=this.extend.size()-1; i>=0; i--) {
				String aId = (String)this.extend.elementAt(i);
				AttributeDefinition aDef = (AttributeDefinition)aDefs.get(aId);
				MyHashMap tCases = aDef.getTestCases();
				this.testCases.putAll(tCases);
			}
		}
		this.testCases.putAll(temp);
	}
	
	
	String getId () {
		
		return this.id;
	}
	
	
	String getLegalValues () {
		
		return this.legalValues;
	}
	
	
	MyHashMap getAttrList () {
		
		return this.attrList;
	}
	
	
	RegularExpression getLegals () {
		
		return this.legals;
	}
	
	
	String getLegalVal () {
		
		try {
			return this.legals.randomLegalString(Globals.randGen.nextInt());
		}
		catch (Exception exc) {
			if (Certification.DEBUGGING_INFO) Loggin.Log(Globals.indent()+"Exception: \""+exc.getMessage()+"\" in AttributeDefinition.getLegalVal");
			return null;
		}
	}
	
	
	Vector getIllegals () {
		
		return this.illegals;
	}
	
	
	String getIllegalVal () {
		
		if (this.illegals==null) return null;
		return (String)this.illegals.elementAt(Globals.randGen.nextInt(this.illegals.size()));
	}
	
	
	boolean valIsLegal (String value) {
		
		return this.legals.matches(value);
	}
}
