## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
    collapse = TRUE,
    comment = "#>",
    fig.path = "figures/",
    dpi = 36
)

## ----eval=TRUE, include=F-----------------------------------------------------
start.time <- Sys.time()

## ----eval=TRUE, warning=FALSE, message=FALSE----------------------------------
# Load libs
library(Banksy)

library(SummarizedExperiment)
library(SpatialExperiment)
library(scuttle)

library(scater)
library(cowplot)
library(ggplot2)

# Load data
data(hippocampus)
gcm <- hippocampus$expression
locs <- as.matrix(hippocampus$locations)

## ----eval=TRUE----------------------------------------------------------------
head(gcm[,1:5])
head(locs)

## ----eval=TRUE, message=FALSE-------------------------------------------------
se <- SpatialExperiment(assay = list(counts = gcm), spatialCoords = locs)
colData(se) <- cbind(colData(se), spatialCoords(se))

# QC based on total counts
qcstats <- perCellQCMetrics(se)
thres <- quantile(qcstats$total, c(0.05, 0.98))
keep <- (qcstats$total > thres[1]) & (qcstats$total < thres[2])
se <- se[, keep]

## ----eval=TRUE, message=FALSE-------------------------------------------------
# Normalization to mean library size
se <- computeLibraryFactors(se)
aname <- "normcounts"
assay(se, aname) <- normalizeCounts(se, log = FALSE)

## ----eval=TRUE----------------------------------------------------------------
k_geom <- c(15, 30)
se <- computeBanksy(se, assay_name = aname, compute_agf = TRUE, k_geom = k_geom)

## ----eval=TRUE----------------------------------------------------------------
se

## ----eval=TRUE----------------------------------------------------------------
lambda <- c(0, 0.2)
se <- runBanksyPCA(se, use_agf = c(FALSE, TRUE), lambda = lambda, seed = 1000)

## ----eval=TRUE----------------------------------------------------------------
reducedDimNames(se)

## ----eval=TRUE----------------------------------------------------------------
k <- 50
res <- 1
se <- clusterBanksy(se, use_agf = c(FALSE, TRUE), lambda = lambda, k_neighbors = k, resolution = res, seed = 1000)

## ----eval=TRUE----------------------------------------------------------------
colnames(colData(se))

## ----eval=TRUE----------------------------------------------------------------
se <- connectClusters(se)

## ----parameter-selection-spatial, eval=TRUE, fig.height=7, out.width='90%'----
cnames <- colnames(colData(se))
cnames <- cnames[grep("^clust", cnames)]
cplots <- lapply(cnames, function(cnm) {
    plotColData(se, x = "sdimx", y = "sdimy", point_size = 0.1, colour_by = cnm) +
        coord_equal() +
        labs(title = cnm) +
        theme(legend.title = element_blank()) +
        guides(colour = guide_legend(override.aes = list(size = 2)))
})

plot_grid(plotlist = cplots, ncol = 2)

## ----eval=TRUE----------------------------------------------------------------
compareClusters(se, func = "ARI")

## ----eval=TRUE----------------------------------------------------------------
compareClusters(se, func = "NMI")

## ----eval=TRUE, echo=FALSE----------------------------------------------------
Sys.time() - start.time

## ----sess---------------------------------------------------------------------
sessionInfo()

