unit LaterHelpers;

interface

uses Windows, Controls, Forms, FileCtrl, SysUtils, OpDialog, Classes;

procedure Usage;

type
  TOperation = (unk, del, ren, mov, inv);

  TOptions = class
  private
    procedure ReportError(nError: Integer; SourceName: String);
  public
    NeedsFile, NeedsMain: Boolean;
    Verbose, Quiet: Boolean;
    Operation: TOperation;
    SourceNames: TStringList;
    DestName: String;
    constructor Create;
    procedure ParseCmdLine;
    function Execute: Boolean;
  end;

implementation

constructor TOptions.Create;
begin
  NeedsFile := false;
  NeedsMain := false;
  Verbose := false;
  Quiet := false;
  Operation := unk;
  SourceNames := TStringList.Create;
  DestName := '';
end;

procedure TOptions.ParseCmdLine;
var i, state: Integer;
    p: String;
begin
  i := 1;
  state := 1; // Options
  while (i <= ParamCount) and (state = 1) and (Operation <> inv)
  do begin
    p := ParamStr(i);
    if (p[1] = '/') or (p[1] = '-')
    then begin
      Delete(p, 1, 1);
      if AnsiSameText(p, 'i')
      then begin
        NeedsMain := true;
        Inc(i);
      end
      else begin
        if AnsiSameText(p, 'v')
        then begin
          Verbose := true;
          Inc(i);
        end
        else begin
          if AnsiSameText(p, 'q')
          then begin
            Quiet := true;
            Inc(i);
          end
          else
            Operation := inv;
        end
      end
    end
    else
      state := 2; // Operation
  end;
  if (i <= ParamCount) and (Operation <> inv)
  then begin
    p := ParamStr(i);
    if AnsiSameText(p, 'del') or AnsiSameText(p, 'delete')
    then begin
      Operation := del;
      Inc(i);
    end
    else begin
      if AnsiSameText(p, 'ren') or AnsiSameText(p, 'rename')
      then begin
        Operation := ren;
        Inc(i);
      end
      else begin
        if AnsiSameText(p, 'mov') or AnsiSameText(p, 'move')
        then begin
          Operation := mov;
          Inc(i);
        end
      end;
    end;
  end;
  if (Operation = unk) and (i <= ParamCount)
  then begin
    Operation := del;
    if not Quiet
    then NeedsFile := true;
  end;
  state := 3; // Source
  if Operation in [del, ren, mov]
  then begin
    while (i <= ParamCount) and (state = 3)
    do begin
      p := ParamStr(i);
      if (Operation in [ren, mov]) and (SourceNames.Count > 0) and AnsiSameText(p, 'to')
      then begin
        state := 4; // Destination
      end
      else begin
        SourceNames.Add(p);
      end;
      Inc(i);
    end;
    if i <= ParamCount
    then DestName := ParamStr(i);
    if (SourceNames.Count = 0) or ((Operation in [ren, mov]) and (DestName = ''))
    then NeedsFile := true;
  end;
  if (Operation = inv) and not Quiet
  then Usage;
  if operation = unk
  then NeedsMain := true;
end;

function TOptions.Execute: Boolean;
var nError, iSrc: Integer;
    drvs: Char;
    pths, fils: String;
    goOn: Boolean;
    SourceName, DestNameBak: String;
begin
  nError := 0;
  Result := true;
  drvs := ' ';
  pths := '';
  fils := '';
  iSrc := 0;
  SourceName := '';
  DestNameBak := DestName;
  if (Operation = unk) or (Operation = inv)
  then Result := false;
  repeat
    if Result and (iSrc < SourceNames.Count)
    then
      try
        ProcessPath(SourceNames[iSrc], drvs, pths, fils);
        SourceName := drvs + ':' + pths + '\' + fils;
      except
        on e: EInOutError
        do begin
          ReportError(e.ErrorCode, SourceNames[iSrc]);
          Result := false;
        end;
      end;
    if Result
    then begin
      with OperationDialog do begin
        Clear;
        edtSrc.Text := SourceName;
        if DestNameBak <> ''
        then DestName := DestNameBak;
        if operation = del
        then begin
          radDel.Checked := true;
        end
        else if operation = mov
        then begin
          edtDestMov.Text := DestName;
          radMov.Checked := true;
        end
        else begin
          edtDestRen.Text := DestName;
          radRen.Checked := true;
        end;
        if Verbose or (NeedsFile and not Quiet)
        then goOn := (ShowModal = mrOK)
        else if NeedsFile and Quiet
        then goOn := false
        else begin
          ValidateInput;
          ValidateOutput;
          goOn := true;
        end;
        if goOn
        then begin
          SourceName := edtSrc.Text;
          if radDel.Checked
          then operation := del
          else if radRen.Checked
          then begin
            operation := ren;
            DestName := edtDestRen.Text;
          end
          else if radMov.Checked
          then begin
            operation := mov;
            DestName := edtDestMov.Text;
          end;
        end
        else
          Result := false;
      end;
    end;
    if Result
    then begin
      case operation of
        del:
          begin
            Result := MoveFileEx(PChar(SourceName), nil, MOVEFILE_DELAY_UNTIL_REBOOT);
            if not Result
            then nError := GetLastError;
            ReportError(nError, SourceName);
          end;
        ren:
          begin
            Result := MoveFileEx(PChar(SourceName), PChar(DestName), MOVEFILE_DELAY_UNTIL_REBOOT);
            if not Result
            then nError := GetLastError;
            ReportError(nError, SourceName);
          end;
        mov:
          begin
            Result := MoveFileEx(PChar(SourceName), PChar(DestName), MOVEFILE_DELAY_UNTIL_REBOOT);
            if not Result
            then nError := GetLastError;
            ReportError(nError, SourceName);
          end;
      else
        Result := false;
      end;
    end;
    Inc(iSrc);
  until (iSrc >= SourceNames.Count) or not Result;
end;

procedure TOptions.ReportError(nError: Integer; SourceName: String);
var buf, msg: String;
begin
  if nError = 0
  then begin
    if Verbose
    then begin
      case operation of
        del: msg := Format('%s erfolgreich zum Lschen vorgemerkt.', [SourceName]);
        ren: msg := Format('%s erfolgreich zum Umbenennen in %s vorgemerkt.', [SourceName, DestName]);
        mov: msg := Format('%s erfolgreich zum Verschieben nach %s vorgemerkt.', [SourceName, DestName]);
      end;
      Application.MessageBox(PChar(msg), 'Erfolg', MB_OK or MB_ICONINFORMATION)
    end
  end
  else begin
    if not Quiet
    then begin
      SetLength(buf, 255);
      FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, nil, nError, 0, PChar(buf), Length(buf), nil);
      case operation of
        del: msg := Format('%s konnte nicht zum Lschen vorgemerkt werden:'#13'%s', [SourceName, buf]);
        ren: msg := Format('%s konnte nicht zum Umbenennen in %s vorgemerkt werden:'#13'%s', [SourceName, DestName, buf]);
        mov: msg := Format('%s konnte nicht zum Verschieben nach %s vorgemerkt werden:'#13'%s', [SourceName, DestName, buf]);
      end;
      Application.MessageBox(Pchar(msg), nil, MB_OK or MB_ICONSTOP);
    end;
  end;
end;

procedure Usage;
var msg, exe: String;
begin
  exe := ExtractFileName(ParamStr(0));
  msg := exe + ' [Optionen] [Befehl] [Dateinamen [to Dateiname]]'#13#13;

  msg := msg + 'Optionen:'#13;
  msg := msg + '  /i'#9'- Interaktive Oberflche immer anzeigen'#13;
  msg := msg + '  /v'#9'- Sicherheitsabfragen anzeigen'#13;
  msg := msg + '  /q'#9'- Alle Meldungen unterdrcken'#13#13;

  msg := msg + 'Befehle:'#13;
  msg := msg + '  delete, del'#9'- Dateien/Ordner lschen'#13;
  msg := msg + '  rename, ren'#9'- Dateien/Ordner umbenennen'#13;
  msg := msg + '  move, mov'#9'- Dateien/Ordner verschieben'#13#13;

  msg := msg + 'Fr weitere Informationen siehe c''t 6/05, S. 252';
  Application.MessageBox(Pchar(msg), 'Verwendung', MB_OK or MB_ICONINFORMATION);
end;

end.
