require 'vector'

class GameStateRingbuffer

  MAX_DIST = 0x7fffffff
  # research suggests that positions can be exactly predicted after 8 movement steps, so we are using 9 frames in the ringbuffer
  BEST_INDEX = -9
  
  def <<(game)
    tagging_completed = nil
    @semaphore.synchronize do
      @buffer.shift if @buffer.size >= @max_size
      
      track_internal_angle!(game)
      track_shot_count!(game)
      if @sequence_offset
        game[:internal_sequence] = game[:sequence] + @sequence_offset
        game[:frames_till_next_saucer_turn] = 128 - (game[:internal_sequence] % 128)
      end
      @buffer << game
      transponse_coordinates!
      tag_objects!
    end
    @observer.each {|p| p.call }
  end    

  def get
    return_value = nil
    @semaphore.synchronize do
      return_value = @buffer.dup
    end
    return return_value
  end

  def get_last
    return_value = nil
    @semaphore.synchronize do
      return_value = @buffer.last.dup
    end
    return return_value
  end
  
  def attach_controls(controls, controls_packet = nil)
    @semaphore.synchronize do
      @buffer.last[:controls] = controls
      @buffer.last[:controls_packet] = controls_packet
    end
  end
  
  def get_controls_packet
    return_value = nil
    @semaphore.synchronize do
      return_value = @buffer.last[:controls_packet] if @buffer.last
    end
    return return_value
  end
  
  def register_observer(observing_object, method_name)
    @observer << (proc { observing_object.send method_name })
  end
  
  def size
    return_value = nil
    @semaphore.synchronize do
      return_value = @buffer.size
    end
    return return_value
  end

  def initialize(size = 10)
    @semaphore = Mutex.new
    @max_size = size
    @buffer = []
    @observer = []
    @reset_observer = []
  end
  
protected

  # has to be called within the semaphore lock
  # -> transpose all object coordinates to a coordinate system of -512..511/-384..383
  #    with our ship at 0/0
  def transponse_coordinates!
    g = @buffer.last
    if g[:ship_present]
      vx = -g[:ship][:vx]
      vy = -g[:ship][:vy]
      g[:ship][:nvx] = 0
      g[:ship][:nvy] = 0
      
      [g[:asteroids], g[:shots], g[:saucer]].flatten.each do |o|
        next unless o[:vx] && o[:vy]

        dx = o[:vx] + vx
        while dx < -512 do
          dx += 1024 # dx normalize to -512 ... 511
        end
        while dx > 511 do
          dx -= 1024
        end

        dy = o[:vy] + vy
        while dy < -384 do
          dy += 768 # dy normalize to -384 ... 383
        end
        while dy > 383 do
          dy -= 768
        end
        
        o[:nvx], o[:nvy] = dx, dy

      end
    
    else # no ship ... just transpose to -512..511/-384..383
      g[:asteroids].each {|a| a[:nvx] = a[:vx] - 512; a[:nvy] = a[:vy] - 384 }
      g[:shots].each     {|s| s[:nvx] = s[:vx] - 512; s[:nvy] = s[:vy] - 384 }
      if g[:saucer_present]
        g[:saucer][:nvx] = g[:saucer][:vx] - 512
        g[:saucer][:nvy] = g[:saucer][:vy] - 384
      end
    end
  end

  # has to be called within the semaphore lock
  def tag_objects!
    g = @buffer.last
    game_index = @buffer.size-1
    g[:objects_by_tag] = {}
    g[:asteroids].each_with_index do |a,object_index|
      if game_index >= 1 && @buffer[game_index-1][:asteroids].size == g[:asteroids].size
        # tag by index if asteroid count is unchanged
        g[:asteroids][object_index][:tag] = @buffer[game_index-1][:asteroids][object_index][:tag]
      else
        tag!(:asteroids, game_index >= 1 ? @buffer[game_index-1] : nil, g ,object_index)
      end
      g[:objects_by_tag][a[:tag]] = a
      a[:dist], a[:dx], a[:dy] = Vector.distance(0, 0, a[:nvx], a[:nvy], a[:r])
      a[:dist] ||= MAX_DIST
    end

    g[:shots].each_with_index do |s,object_index|
      if game_index >= 1 && @buffer[game_index-1][:shots].size == g[:shots].size
        # tag by index if shout count is unchanged
        g[:shots][object_index][:tag] = @buffer[game_index-1][:shots][object_index][:tag]
      else
        tag!(:shots, game_index >= 1 ? @buffer[game_index-1] : nil, g ,object_index)
      end
      g[:objects_by_tag][s[:tag]] = s
      s[:dist], s[:dx], s[:dy] = Vector.distance(0, 0, s[:nvx], s[:nvy])
      s[:dist] ||= MAX_DIST
    end
    if g[:saucer_present]
      g[:saucer][:dist], g[:saucer][:dx], g[:saucer][:dy] = Vector.distance(0, 0, g[:saucer][:nvx], g[:saucer][:nvy], g[:r])
      g[:saucer][:tag] = :ufo
      g[:objects_by_tag][:ufo] = g[:saucer]
    end
    g[:ship][:tag] = :ship if g[:ship_present]
    g[:tagged] = true
    
    past_index = [-@buffer.size,BEST_INDEX].max
    g[:locked_on_targets] = []
    return unless past_index < -1

    if g[:ship_present]
      past_frame = @buffer[past_index..-2].detect{|f| f[:ship_present]}
      if past_frame
        current_ship = g[:ship]
        past_ship = past_frame[:ship]
        frame_count = g[:sequence] - past_frame[:sequence]
        dist, dx, dy = Vector.distance_normalized(past_ship[:vx], past_ship[:vy], current_ship[:vx], current_ship[:vy])
        current_ship[:mx] = ((dx / frame_count.to_f) * 8.0).round / 8.0
        current_ship[:my] = ((dy / frame_count.to_f) * 8.0).round / 8.0
      end
    end

    g[:shots].each do |current_shot|
      tag = current_shot[:tag]
      past_frame = @buffer[past_index..-2].detect{|f| f[:objects_by_tag].has_key?(tag)}
      if past_frame
        past_shot = past_frame[:objects_by_tag][tag]
        frame_count = g[:sequence] - past_frame[:sequence]
        dist, dx, dy = Vector.distance_normalized(past_shot[:vx], past_shot[:vy], current_shot[:vx], current_shot[:vy])
        current_shot[:mx] = ((dx / frame_count.to_f) * 8.0).round / 8.0
        current_shot[:my] = ((dy / frame_count.to_f) * 8.0).round / 8.0
      end
    end

    shot_lifetime = g[:internal_sequence] ?  72 - ((g[:internal_sequence]+g[:shot_delay]) % 4) : 69
    g[:asteroids].each do |current_asteroid|
      tag = current_asteroid[:tag]
      past_frame = @buffer[past_index..-2].detect{|f| f[:objects_by_tag].has_key?(tag)}
      if past_frame
        past_asteroid = past_frame[:objects_by_tag][tag]
        frame_count = g[:sequence] - past_frame[:sequence]
        dist, dx, dy = Vector.distance_normalized(past_asteroid[:vx], past_asteroid[:vy], current_asteroid[:vx], current_asteroid[:vy])
        current_asteroid[:mx] = ((dx / frame_count.to_f) * 8.0).round / 8.0
        current_asteroid[:my] = ((dy / frame_count.to_f) * 8.0).round / 8.0
        if (current_asteroid[:lock_on] = Vector.shot_will_hit?(g[:ship], g[:shot_angle_index], current_asteroid, shot_lifetime))
          g[:locked_on_targets] << current_asteroid
        end
      end
    end

    if g[:saucer_present]
      past_frame = @buffer[past_index..-2].detect{|f| f[:saucer_present]}
      current_saucer = g[:saucer]
      if past_frame
        past_saucer = @buffer[-2][:saucer]
        frame_count = g[:sequence] - past_frame[:sequence]
        if frame_count == 1 && @sequence_offset.nil? && g[:enabled]
          @sequence_offset = -(g[:sequence] % 128) 
puts "[#{g[:sequence]}] Calibrated internal sequence counter with offset #{@sequence_offset}" if $debug
        end
        dist, dx, dy = Vector.distance_normalized(past_saucer[:vx], past_saucer[:vy], current_saucer[:vx], current_saucer[:vy])
        current_saucer[:mx] = [-2.0, 2.0].closest_to(dx / (g[:sequence] - @buffer[-2][:sequence]).to_f)
        current_saucer[:my] = [-2.0, 0.0, 2.0].closest_to(dy / (g[:sequence] - @buffer[-2][:sequence]).to_f)
        if (current_saucer[:lock_on] = Vector.shot_will_hit?(g[:ship], g[:shot_angle_index], current_saucer, shot_lifetime))
          g[:locked_on_targets] << current_saucer
        end
      end
    end
  end
  
  # object will receive :tag based on proximity to same object(s) in past_game
  # if past_game is nil or no object is found within expected*) range, the object
  # will be given a new tag
  #   *) we expect objects to be close to where they are supposed to be using their
  #      motion vector.
  #      when we have no motion vector yet and the object to be tagged is a shot
  #      we expect it to move no further than shot speed + ship/saucer speed
  def tag!(object_type, past_game, current_game, object_index) 
    # FIXME: expected_range must now be given in squared distances!!!
    object = current_game[object_type][object_index]
    min_index = nil
    min_dist = MAX_DIST
    expected_range = 0
    if past_game
      frame_count = current_game[:sequence] - past_game[:sequence]
      past_game[object_type].each_with_index do |previous_object, index|
        # asteroids must also match size and type!
        next if object_type == :asteroids && (previous_object[:vs] != object[:vs] || previous_object[:type] != object[:type])
        # FIXME: do we need to do wrap-calculations when adding m to v? I don't think so now, but let's see
        if previous_object[:mx] && previous_object[:my]
          vx = previous_object[:nvx] + previous_object[:mx]*frame_count
          vy = previous_object[:nvy] + previous_object[:my]*frame_count
          expected_range = 2.23 # having a motion vector should allow us to nail the position to 2x2 px + 1 for rounding errors
        elsif object_type == :shots && previous_object[:tag]
          # add ship/saucer speed to v
          shooter = ((previous_object[:tag].to_s =~ /^u/ ? past_game[:saucer] : past_game[:ship]) || {})
          vx = previous_object[:nvx] + (shooter[:mx]||0)*frame_count
          vy = previous_object[:nvy] + (shooter[:my]||0)*frame_count
          if shooter[:mx] && shooter[:my]
            expected_range = 10 # when the shooter motion vector is know, the shot should be within its normal 8*8 px + 2px for rounding errors
          else
            expected_range = 8 # without knowing the vector the shot might move at 16px/frame max. -> 16*16
          end
        else
          vx = previous_object[:nvx]
          vy = previous_object[:nvy]
          expected_range = 6*frame_count # normal objects will not move by more than 6px/frame -> 6*6
        end
        dist, dx, dy = Vector.distance(vx, vy, object[:nvx], object[:nvy])
        if dist <= expected_range && dist < min_dist
          min_index = index
          min_dist = dist
        end
      end
    end
    
    object[:tag] = min_index && min_dist <= expected_range ? past_game[object_type][min_index][:tag] : nil

    game = past_game || current_game
    # no tag yet, assign a new tag
    unless object[:tag] # must be new object
      case object_type
      when :asteroids
        object[:tag] = "a#{current_game[:sequence]}-#{object_index}".to_sym
      when :shots
        # u = ufo shot; s = ship shot
        origin = if game[:saucer_present] && game[:ship_present]
          dist_ufo, dx, dy = Vector.distance(game[:saucer][:nvx], game[:saucer][:nvy],
                                             object[:nvx], object[:nvy])
          dist_ship, dx, dy = Vector.distance(0, 0, object[:nvx], object[:nvy])
          dist_ship > dist_ufo ? :u : :s
        elsif game[:saucer_present]
          :u
        elsif game[:ship_present]
          :s
        end
        object[:tag] = "#{origin}#{current_game[:sequence]}-#{object_index}".to_sym
      end
    end
    
    return object[:tag]
  end
  
  def command_to_angle_delta(control_status)
    angle_delta = if Controls.left?(control_status)
                    3
                  elsif Controls.right?(control_status)
                    -3
                  else
                    0
                  end
  end
  
  def track_internal_angle!(game)
    unless game[:lost_frames] == 0
puts "----------------LOST #{game[:lost_frames]} FRAMES------------------" if $debug
      @angle_calibrated = false
      @possible_angles = nil
    end
    
    if game[:latency] > @max_size-2
puts "------------TOO BIG LATENCY: #{game[:latency]} FRAMES--------------" if $debug
      @angle_calibrated = false
      @possible_angles = nil
    end

    return unless game[:ship_present]
    
    # note: game has not been added to the @buffer yet, so @buffer.last is _previous_ game
    #       command_game has effect on game
    #       next_command_game has effect on "game+1" -> shot angle
    previous_game = @buffer.last
    next_commmand_game = previous_game ? @buffer[-1-previous_game[:latency]] : @buffer[-2]
    command_game = previous_game ? @buffer[-2-previous_game[:latency]] : @buffer[-3]
    ship_d = [ game[:ship][:dx], game[:ship][:dy] ]

    # first initialization by ship's d vector (approximation)
    @angle_index ||= Vector::MEDIUM_INDICES_BY_SHIP_DX[ship_d]

    if command_game && next_commmand_game[:ship_present]
      angle_delta = command_to_angle_delta(command_game[:controls])
      @angle_index = (@angle_index + angle_delta*(1+game[:lost_frames])) & 0xff
# puts "[#{game[:sequence]}] latency #{game[:latency]}, ping #{game[:ping]}, angle updated to #{@angle_index} by #{angle_delta} based on cmd [#{command_game[:sequence]}], prev. angle #{next_commmand_game[:angle_index]}" if $debug

      unless Vector::POSSIBLE_INDICES_BY_SHIP_DX[ship_d].include?(@angle_index)
puts "\e[31m[#{game[:sequence]}] Estimated angle at #{ship_d.inspect} as #{Vector::MEDIUM_INDICES_BY_SHIP_DX[ship_d]} (angle was: #{@angle_index} with possible angles #{Vector::POSSIBLE_INDICES_BY_SHIP_DX[ship_d].inspect})\e[0m]" if $debug
        @angle_index = Vector::MEDIUM_INDICES_BY_SHIP_DX[ship_d]
        @angle_calibrated = false
        @possible_angles = nil
      end

      if !@angle_calibrated && next_commmand_game[:lost_frames] == 0 && previous_game[:lost_frames] == 0 && game[:lost_frames] == 0 && next_commmand_game[:latency] == 1 && previous_game[:latency] == 1 && game[:latency] == 1
        @possible_angles ||= Vector::POSSIBLE_INDICES_BY_SHIP_DX[[ next_commmand_game[:ship][:dx], next_commmand_game[:ship][:dy] ]]
        @possible_angles = @possible_angles.map{|a| (a + angle_delta) & 0xff } & Vector::POSSIBLE_INDICES_BY_SHIP_DX[ship_d]
        if @possible_angles.size == 1 # nailed it to one possible angle
# puts "\e[32m[#{game[:sequence]}] Calibrated angle at #{ship_d.inspect} to #{@possible_angles.first} (angle was: #{@angle_index})\e[0m" if $debug
          @angle_index = @possible_angles.first
          @possible_angles = nil
          @angle_calibrated = true
        end
      elsif !@angle_calibrated
        @possible_angles = nil # reset possible angles when latency/lost frames are encountered
      end

    end

  ensure
    game[:angle_calibrated] = @angle_calibrated
    game[:angle_index] = @angle_index
    game[:shot_angle_index] = @angle_index
    game[:shot_delay] = 1
    ((-1-previous_game[:latency])..-1).each do |index|
      game[:shot_angle_index] = (game[:shot_angle_index] + command_to_angle_delta(@buffer[index][:controls])) & 0xff if @buffer[index] && @buffer[index][:controls]
      game[:shot_delay] += 1
    end if previous_game
  end
  
  def track_shot_count!(game)
    # note: game has not been added to the @buffer yet, so @buffer.last is _previous_ game
    #       command_game has effect on game
    previous_game = @buffer.last
    existing_shots = 0
    existing_shots = previous_game[:shots].select{|s| s[:tag].to_s =~ /^s/ }.size if previous_game
    existing_shots += 1 if previous_game && Controls.fire?(previous_game[:controls])
    game[:fired_shot_count] = existing_shots
  end
  
end