package se.jupp.asteroids;

import static se.jupp.asteroids.GameData.*;

import java.awt.Graphics2D;

public class Position {
    final int x, y;

    public Position() {
        this.x = 0;
        this.y = 0;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + x;
        result = prime * result + y;
        return result;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        final Position other = (Position) obj;
        if (x != other.x)
            return false;
        if (y != other.y)
            return false;
        return true;
    }

    public Position(int pixel_x, int pixel_y) {
        this(pixel_x << 3, pixel_y << 3, true);
    }

    Position(int x, int y, boolean internal) {
        this.x = x;
        this.y = y;
    }

    public Position normalize() {
        int minX = (-EXTENT_X / 2) << 3;
        int maxX = (EXTENT_X / 2) << 3;
        int minY = (-EXTENT_Y / 2) << 3;
        int maxY = (EXTENT_Y / 2) << 3;
        int x0 = x, y0 = y;
        // normalize x: minX ... maxX
        while (x0 < minX)
            x0 += maxX - minX;
        while (x0 > maxX)
            x0 -= maxX - minX;
        // normalize y: minY ... maxY
        while (y0 < minY)
            y0 += maxY - minY;
        while (y0 > maxY)
            y0 -= maxY - minY;
        return new Position(x0, y0, true);
    }

    public Position unnormalize() {
        int minX = 0;
        int maxX = EXTENT_X << 3;
        int minY = MIN_Y << 3;
        int maxY = MAX_Y << 3;
        int x0 = x, y0 = y;
        // normalize x: minX ... maxX
        while (x0 < minX)
            x0 += maxX - minX;
        while (x0 > maxX)
            x0 -= maxX - minX;
        // normalize y: minY ... maxY
        while (y0 < minY)
            y0 += maxY - minY;
        while (y0 > maxY)
            y0 -= maxY - minY;
        return new Position(x0, y0, true);
    }

    public int x() {
        return (x >> 3);
    }

    public int y() {
        return (y >> 3);
    }

    public Position subtract(Position b) {
        if (b != null)
            return new Position(x - b.x, y - b.y, true);
        else
            return this;
    }

    public Position add(Position b) {
        if (b != null)
            return new Position(x + b.x, y + b.y, true);
        else
            return this;
    }

    public Position multiply(int b) {
        return new Position(x * b, y * b, true);
    }

    public Position divide(int b) {
        return new Position(x / b, y / b, true);
    }

    public int length() {
        return ((int) Math.sqrt(square()));
    }

    public double getAngle() {
        return Math.atan2(y, x);
    }

    public int distanceSquared(Position other) {
        return subtract(other).normalize().square();
    }

    public Position matrixMultiply(double a, double b, double c, double d) {
        return new Position((int) Math.round(a * x + b * y), (int) Math.round(c
                * x + d * y), true);
    }

    public int dotProduct(Position b) {
        return (x * b.x + y * b.y) >> 6;
    }

    public int square() {
        return dotProduct(this);
    }

    public int crossProduct2D(Position b) {
        return (x * b.y - y * b.x) >> 6;
    }

    public void drawLine(Position b, Graphics2D g) {
        g.drawLine(x(), y(), b.x(), b.y());
        Position b2 = b.unnormalize();
        if (!b2.equals(b)) {
            Position t2 = b2.subtract(b.subtract(this));
            g.drawLine(t2.x(), t2.y(), b2.x(), b2.y());
        }
        //        g.drawLine(b.x(), b.y(), x(), y());
    }

    public void drawLine2(Position b, Graphics2D g) {
        Position b2 = b.subtract(this).add(this).normalize();
        if (!b2.equals(b)) {
            Position t2 = this.subtract(b).add(b);
            g.drawLine(x(), y(), b2.x(), b2.y());
            g.drawLine(t2.x(), t2.y(), b.x(), b.y());
        } else
            g.drawLine(x(), y(), b.x(), b.y());
    }

    public String toString() {
        return "<P " + (x >> 3) + " " + (y >> 3) + ", " + x + " " + y + ">";
    }

    public static int timeToCollision2(Position bp, Position bv, int radius) {
        int d = radius << 3;
        long bx0 = bp.x;
        long by0 = bp.y;
        long bdx = bv.x;
        long bdy = bv.y;
        long term, rootterm;
        double foo, foo2;
        double root;
        term = bdx * bx0 + bdy * by0;
        rootterm =
                4 * term * term - 4 * (bdx * bdx + bdy * bdy)
                        * (bx0 * bx0 + by0 * by0 - d * d);
        root = Math.sqrt(rootterm);
        foo = (-bdx * bx0 - bdy * by0 - root / 2) / (bdx * bdx + bdy * bdy);
        foo2 = (-bdx * bx0 - bdy * by0 + root / 2) / (bdx * bdx + bdy * bdy);
        if (foo < foo2) {
            if (foo >= 0)
                return (int) (foo + 0.5);
            if (foo2 >= 0)
                return (int) (foo2 + 0.5);
            return -1;
        } else {
            if (foo2 >= 0)
                return (int) (foo2 + 0.5);
            if (foo >= 0)
                return (int) (foo + 0.5);
            return -1;
        }
        //        System.out.println("ttc " + foo + " " + foo2);
        //        if (foo2 > 0 && foo2 < 70)
        //            System.out.println("radius = " + radius + " d=" + d + " time="
        //                    + foo2);
    }

    public static int timeToCollision(Position ap, Position av, Position bp,
            Position bv, int radius) {
        int d = radius << 3;
        long ax0 = ap.x;
        long ay0 = ap.y;
        long bx0 = bp.x;
        long by0 = bp.y;
        long adx = av.x;
        long ady = av.y;
        long bdx = bv.x;
        long bdy = bv.y;
        long term, rootterm;
        double foo, foo2;
        double root;
        term =
                -ax0 * bdx - ay0 * bdy + adx * (ax0 - bx0) + bdx * bx0 + ady
                        * (ay0 - by0) + bdy * by0;
        rootterm =
                4
                        * term
                        * term
                        - 4
                        * (adx * adx + ady * ady - 2 * adx * bdx + bdx * bdx
                                - 2 * ady * bdy + bdy * bdy)
                        * (ax0 * ax0 + ay0 * ay0 - 2 * ax0 * bx0 + bx0 * bx0
                                - 2 * ay0 * by0 + by0 * by0 - d * d);
        //        System.out.println("d=" + d + ", ax0=" + ax0 + ", ay0=" + ay0
        //                + ", bx0=" + bx0 + ", by0=" + by0 + ", adx=" + adx + ", ady="
        //                + ady + ", bdx=" + bdx + ", bdy=" + bdy + ", term=" + term
        //                + ", rootterm=" + rootterm);
        root = Math.sqrt(rootterm);
        foo =
                (-adx * ax0 - ady * ay0 + ax0 * bdx + ay0 * bdy + adx * bx0
                        - bdx * bx0 + ady * by0 - bdy * by0 - root / 2)
                        / (adx * adx + ady * ady - 2 * adx * bdx + bdx * bdx
                                - 2 * ady * bdy + bdy * bdy);
        foo2 =
                (-adx * ax0 - ady * ay0 + ax0 * bdx + ay0 * bdy + adx * bx0
                        - bdx * bx0 + ady * by0 - bdy * by0 + root / 2)
                        / (adx * adx + ady * ady - 2 * adx * bdx + bdx * bdx
                                - 2 * ady * bdy + bdy * bdy);
        if (foo < foo2) {
            if (foo >= 0 && foo < 70)
                return (int) (foo + 0.5);
            if (foo2 >= 0 && foo2 < 70)
                return (int) (foo2 + 0.5);
            return -1;
        } else {
            if (foo2 >= 0 && foo2 < 70)
                return (int) (foo2 + 0.5);
            if (foo >= 0 && foo < 70)
                return (int) (foo + 0.5);
            return -1;
        }
        //        System.out.println("ttc " + foo + " " + foo2);
        //        if (foo2 > 0 && foo2 < 70)
        //            System.out.println("radius = " + radius + " d=" + d + " time="
        //                    + foo2);
    }
}
