package se.jupp.asteroids;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;

/**
 *  Generic game object in Asteroid game.
 *
 *  A game object has a position and a size and it can be drawn.
 * 
 *  This is part of a solution for a competition
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">by the German computer magazine c't</a>
 */
public abstract class GameObject implements GameData, Cloneable {
    /** Size of head of velocity arrow. */
    private static final int ARROW_HEAD_SIZE = 8;

    /** The arrow head for velocity drawings. */
    private static final GeneralPath ARROW_HEAD = new GeneralPath();
    static {
        ARROW_HEAD.moveTo(0, 0);
        ARROW_HEAD.lineTo(-ARROW_HEAD_SIZE, ARROW_HEAD_SIZE);
        ARROW_HEAD.lineTo(-ARROW_HEAD_SIZE, -ARROW_HEAD_SIZE);
        ARROW_HEAD.closePath();
    }
    int id;
    private static int counter = 0;
    Position pos, velocity;
    Position bornAt;
    int age, timeBorn;
    boolean dangerous;
    /** The radius of collision detection circle */
    int r;

    /**
     *  Constructor.
     *  @param x  x coordinate
     *  @param y  y coordinate
     */
    protected GameObject(Position pos, int r, int timeBorn) {
        this.pos = pos;
        this.bornAt = pos;
        velocity = new Position();
        age = 0;
        this.timeBorn = timeBorn;
        this.r = r;
        id = counter++;
    }

    @Override
    protected Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

    void update(Position pos) {
        age++;
        //        if (pos != null) {
        //            this.pos = pos;
        //            if (age <= 8)
        //                velocity = pos.subtract(bornAt).divide(age);
        //            else
        //                pos = pos.add(velocity);
        //        }
    }

    /**
     *  Get the size of the object.
     *
     *  The size returned by this method is half the length of a square which contains the object,
     *  so the object's bounding box is between (x - size, y - size) and (x + size, y + size).
     *  @return object size
     */
    public abstract int getSize();

    /**
     *  Draw the object.
     *  @param g graphics context
     */
    public void draw(Graphics2D g) {
        g.setColor(Color.GREEN);
        g.drawOval(pos.x() - r, pos.y() - r, 2 * r, 2 * r);
        drawVelocityVector(g);
    }

    /**
     *  Get the squared size of this object.
     *  @return squared size
     */
    public int getSquaredSize() {
        int size = getSize();
        return size * size;
    }

    public int timeToCollision(GameObject b, int future) {
        Position bpos;
        if (future > 0)
            bpos = b.pos.add(b.velocity.multiply(future)).normalize();
        else
            bpos = b.pos;
        return Position.timeToCollision2(bpos.subtract(pos).normalize(),
            b.velocity.subtract(velocity).normalize(), r + b.r);
    }

    /**
     *  Draw a vector displaying the current velocity.
     *  @param g graphics context
     */
    protected void drawVelocityVector(Graphics2D g) {
        if (velocity.square() != 0) {
            int scale = 16;
            g.setColor(Color.red);
            Position f = pos.add(velocity.multiply(scale));
            pos.unnormalize().drawLine(f, g);
            f = f.unnormalize();
            AffineTransform at =
                    AffineTransform.getTranslateInstance(f.x(), f.y());
            at.concatenate(AffineTransform.getRotateInstance(velocity
                .getAngle()));
            g.fill(at.createTransformedShape(ARROW_HEAD));
        }
    }
}
