package se.jupp.asteroids;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.*;

/**
 *  The info describing one game frame constructed from a received datagram.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>.
 */
public class Frame implements GameData {
    static final int MAME_DATAGRAM_SIZE = VECTORRAM_SIZE + 2;
    private static final int ID_BYTE = VECTORRAM_SIZE;
    private static final int PING_BYTE = ID_BYTE + 1;

    private static final int MASK_OPCODE = 0xF0;
    private static final int OPCODE_LABS = 0xA0;
    private static final int OPCODE_HALT = 0xB0;
    private static final int OPCODE_JSRL = 0xC0;
    private static final int OPCODE_RTSL = 0xD0;
    private static final int OPCODE_JMPL = 0xE0;
    private static final int OPCODE_SVEC = 0xF0;
    private static final int OPCODE_SHIP = 0x60;

    private static final int SUB_ASTEROID_TYPE1 = 0x8F3;
    private static final int SUB_ASTEROID_TYPE2 = 0x8FF;
    private static final int SUB_ASTEROID_TYPE3 = 0x90D;
    private static final int SUB_ASTEROID_TYPE4 = 0x91A;
    private static final int SUB_UFO = 0x929;

    /** Frame ID as sent by MAME. */
    private final byte id;
    /** Ping ID as sent by MAME. */
    private final byte ping;
    final int score;
    UfoPos ufo;
    List<AsteroidPos> asteroids = new LinkedList<AsteroidPos>();
    SpaceShipPos spaceShip;
    List<Position> bullets = new LinkedList<Position>();

    AsteroidPos asteroidAt(int i) {
        return i < asteroids.size() ? asteroids.get(i) : null;
    }

    Position bulletAt(int i) {
        return i < bullets.size() ? bullets.get(i) : null;
    }

    static class SpaceShipPos extends Position {
        final Position direction;

        SpaceShipPos(int x, int y, int dx, int dy) {
            super(x, y);
            this.direction = new Position(dx >> 3, dy >> 3);
        }

    }

    static class UfoPos extends Position {
        final int size;

        UfoPos(int x, int y, int size) {
            super(x, y);
            this.size = size;
        }
    }

    static class AsteroidPos extends Position {
        final int size, type;

        AsteroidPos(int x, int y, int size, int type) {
            super(x, y);
            this.size = size;
            this.type = type;
        }

    }

    /**
     * Interprete 8 bits as unsigned byte.
     * @param b incomoing byte
     * @return unsigned interpretation
     */
    private static int byteToUnsigned(byte b) {
        return b < 0 ? b + 256 : b;
    }

    @Override
    public String toString() {
        return "Frame asteroids=" + asteroids.size() + " bullets="
                + bullets.size();
    }

    /**
     *  Constructor.
     *
     *  Extracts frame info from datagram.
     *  @param bytes      datagram bytes
     *  @param pingTimes  time stamps when pings were sent
     *  @throws IOException on i/o errors
     */
    Frame(ByteBuffer b) throws IOException {
        id = b.get(ID_BYTE);
        ping = b.get(PING_BYTE);
        StringBuilder buf = new StringBuilder();
        decode(b, buf);
        if (buf.length() > 0) {
            score = Integer.parseInt(buf.toString());
        } else
            score = -1;
        //        for (AsteroidPos a : asteroids)
        //            System.out.print(" " + a.x + " " + a.y);
        //        System.out.println();
    }

    private void decode(ByteBuffer b, StringBuilder scoreBuf)
            throws IOException {
        int n = b.remaining();
        if (n != MAME_DATAGRAM_SIZE) {
            throw new IOException("Incorrect datagram with size " + n);
        }
        if ((b.get(1) & MASK_OPCODE) != OPCODE_JMPL) {
            throw new IOException(String.format(
                "Incorrect vector buffer start: %02x%02x", b.get(1), b.get(1)));
        }
        int vx = 0;
        int vy = 0;
        int vs = 0;
        int v1x = 0;
        int v1y = 0;
        int dx = 0;
        int dy = 0;
        boolean possibleShip = false;
        int p = 2; // skip first two
        while (p < VECTORRAM_SIZE) {
            int highbyte = byteToUnsigned(b.get(p + 1));
            int opcode = highbyte & MASK_OPCODE;
            switch (opcode) {
            case OPCODE_LABS:
                vy = (highbyte & 0x03) << 8 | byteToUnsigned(b.get(p));
                p += 2;
                highbyte = byteToUnsigned(b.get(p + 1));
                vx = (highbyte & 0x03) << 8 | byteToUnsigned(b.get(p));
                vs = highbyte >> 4;
                p += 2;
                break;

            case OPCODE_HALT:
                // p += 2;
                return;

            case OPCODE_JSRL:
                switch ((highbyte & 0x0F) << 8 | byteToUnsigned(b.get(p))) {
                case SUB_ASTEROID_TYPE1:
                    asteroids.add(new AsteroidPos(vx, vy, vs, 0));
                    break;
                case SUB_ASTEROID_TYPE2:
                    asteroids.add(new AsteroidPos(vx, vy, vs, 1));
                    break;
                case SUB_ASTEROID_TYPE3:
                    asteroids.add(new AsteroidPos(vx, vy, vs, 2));
                    break;
                case SUB_ASTEROID_TYPE4:
                    asteroids.add(new AsteroidPos(vx, vy, vs, 3));
                    break;
                case SUB_UFO:
                    ufo = new UfoPos(vx, vy, vs);
                    break;
                case 0xADD:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('0');
                    break;
                case 0xB2E:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('1');
                    break;
                case 0xB32:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('2');
                    break;
                case 0xB3A:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('3');
                    break;
                case 0xB41:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('4');
                    break;
                case 0xB48:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('5');
                    break;
                case 0xB4F:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('6');
                    break;
                case 0xB56:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('7');
                    break;
                case 0xB5B:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('8');
                    break;
                case 0xB63:
                    if (vx == 100 && vy == 876 && vs == 1)
                        scoreBuf.append('9');
                    break;
                }
                p += 2;
                break;

            case OPCODE_RTSL:
                // p += 2;
                return;

            case OPCODE_JMPL:
                //p += 2;
                return;

            case OPCODE_SVEC:
                p += 2;
                break;

            default:
                if (spaceShip == null) {
                    dy = (highbyte & 0x03) << 8 | byteToUnsigned(b.get(p));
                    if ((highbyte & 0x04) != 0) {
                        dy = -dy;
                    }
                    p += 2;
                    highbyte = byteToUnsigned(b.get(p + 1));
                    dx = (highbyte & 0x03) << 8 | byteToUnsigned(b.get(p));
                    if ((highbyte & 0x04) != 0) {
                        dx = -dx;
                    }
                    int vz = highbyte >> 4;
                    if (dx == 0 && dy == 0) {
                        if (vz == 15) {
                            bullets.add(new Position(vx, vy));
                        }
                    }
                    if (dx != 0 && dy != 0) {
                        if (opcode == OPCODE_SHIP && vz == 12) {
                            if (possibleShip) {
                                if (spaceShip == null) {
                                    spaceShip =
                                            new SpaceShipPos(vx, vy, v1x - dx,
                                                v1y - dy);
                                }
                                possibleShip = false;
                            } else {
                                v1x = dx;
                                v1y = dy;
                                possibleShip = true;
                            }
                        }
                    } else if (possibleShip) {
                        possibleShip = false;
                    }
                    p += 2;
                } else {
                    p += 4;
                }
                break;
            }
        }
    }

    /**
     *  Get the frame id.
     *
     *  The frame id is a counter which is incremented by mame each time a frame is sent.
     *  @return frame id
     */
    public byte getId() {
        return id;
    }

    /**
     *  Get the ping associated with this frame.
     *  @return a number between <code>1</code> and <code>255</code> if there is a ping associated with this frame,
     *          otherwise <code>0</code>
     */
    public int getPing() {
        return ping;
    }

    /**
     *  Get the ufo.
     *  @return ufo or <code>null</code> if no ufo is present
     */
    public UfoPos getUfo() {
        return ufo;
    }

    /**
     *  Get the asteroids.
     *
     *  @return the asteroids
     */
    public Collection<AsteroidPos> getAsteroids() {
        return Collections.unmodifiableCollection(asteroids);
    }

    /**
     *  Get the space ship.
     *  @return space ship or <code>null</code> if there is no space ship present
     */
    public SpaceShipPos getSpaceShip() {
        return spaceShip;
    }

    /**
     *  Get the bullets.
     *
     *  @return bullets
     */
    public Collection<Position> getBullets() {
        return Collections.unmodifiableCollection(bullets);
    }

}