package se.jupp.asteroids;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.PortUnreachableException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.util.LinkedList;
import java.util.List;

public class Communication {
    private static final int MAME_PORT = 1979;
    private static final byte[] KEY_FRAME =
            new byte[] { 'c', 't', 'm', 'a', 'm', 'e', 0, 0 };

    /** Socket info. */
    private DatagramChannel channel;
    /** Address of MAME. */
    private InetSocketAddress mameAddr;
    ByteBuffer keyBuffer = ByteBuffer.allocate(KEY_FRAME.length);
    private List<FrameListener> frameListeners =
            new LinkedList<FrameListener>();

    public Communication(String hostname) throws IOException {
        int port = MAME_PORT;
        String[] parts = hostname.split(":");
        if (parts.length == 2) {
            try {
                port = Integer.parseInt(parts[1]);
                hostname = parts[0];
            } catch (NumberFormatException e) {
                e.printStackTrace();
            }
        }
        keyBuffer.put(KEY_FRAME);
        mameAddr = new InetSocketAddress(hostname, port);
        channel = DatagramChannel.open();
        channel.connect(mameAddr);
        Thread comThread = new Thread(new Runnable() {
            public void run() {
                loop();
            }
        }, "Communication");
        comThread.start();
    }

    private void loop() {
        boolean first = true;
        while (true) {
            try {
                sendKeys(0, 0);
                ByteBuffer buffer =
                        ByteBuffer.allocate(Frame.MAME_DATAGRAM_SIZE);
                while (true) {
                    buffer.clear();
                    SocketAddress adr = channel.receive(buffer);
                    if (adr == null)
                        System.out.println("adr=" + adr);
                    else if (first)
                        System.out.println("Got first datagram from " + adr);
                    first = false;
                    buffer.flip();
                    Frame f = null;
                    if (buffer.remaining() == 11) {
                        System.out.println("Got game over packet");
                    } else if (buffer.remaining() == Frame.MAME_DATAGRAM_SIZE) {
                        f = new Frame(buffer);
                    } else {
                        System.out.println("Datagram size = "
                                + buffer.remaining());
                        continue;
                    }
                    synchronized (frameListeners) {
                        for (FrameListener l : frameListeners)
                            l.frameReceived(f);
                    }
                    if (f == null)
                        return;
                }
            } catch (PortUnreachableException e) {
                if (!first)
                    e.printStackTrace();
            } catch (IOException e) {
                e.printStackTrace();
            }
            try {
                Thread.sleep(1000L);
            } catch (InterruptedException e) {
            }
        }
    }

    /**
     *  Send the pressed keys.
     *
     *  @param keys combination of
     *         {@link #BUTTON_HYPERSPACE},
     *         {@link #BUTTON_FIRE},
     *         {@link #BUTTON_LEFT},
     *         {@link #BUTTON_RIGHT},
     *         {@link #BUTTON_THRUST}
     *  @throws IOException if sending failed
     */
    public void sendKeys(int keys, int pingid) throws IOException {
        keyBuffer.put(6, (byte) keys);
        keyBuffer.put(7, (byte) pingid);
        keyBuffer.flip();
        channel.write(keyBuffer);
    }

    public void addFrameListener(FrameListener listener) {
        synchronized (frameListeners) {
            frameListeners.add(listener);
        }
    }
}