/**
 * File:    SamplePlayer.java
 * Package: de.heise.asteroid
 * Created: 19.05.2008 22:57:28
 * Author:  Chr. Moellenberg
 *
 * Based on demo program by Harald Boegeholz.
 * Copyright (c) 2008 by Harald Boegeholz, c't Magazin fure Computertechnik
 */

package de.heise.asteroid.player;

import java.util.List;

import de.heise.asteroid.ScreenVector;
import de.heise.asteroid.engine.Engine;
import de.heise.asteroid.model.Asteroid;
import de.heise.asteroid.model.Saucer;
import de.heise.asteroid.model.Ship;

/**
 * The <code>SamplePlayer</code> class implements the Asteroid player strategy 
 * as created by Harald Boegeholz, c't Magazin fuer Computertechnik.
 * 
 * @see AbstractPlayer
 */
public class SamplePlayer extends AbstractPlayer {
   /**
    * Initializes a Player with the given engine.
    * 
    * @param e the <code>Engine</code>
    */
   public SamplePlayer(Engine e) {
      super(e);
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.player.AbstractPlayer#play()
    */
   public void play() {
      if (status != null) {
         // System.out.println(status); // dump game status
         Ship ship = status.getShip();
         Saucer saucer = status.getSaucer();
         
         releaseKeys();
         int minDistance = 0x7fffffff;
         ScreenVector minDifference = null;
         if (ship != null) {
            // search for nearest asteroid
            List<Asteroid> asteroids = status.getAsteroids();
            for (Asteroid ast : asteroids) {
               if (ast.isExploding()) {
                  continue;
               }
               ScreenVector distVector = ship.getDistanceTo(ast);
               distVector.normalize();
               // the squared distance
               int sqDist = distVector.scalarProduct(distVector);
               // adjust the squared distance for the radius of the asteroid
               sqDist -= ast.getRadius() * ast.getRadius();
               if (sqDist < minDistance) {
                  minDistance = sqDist;
                  minDifference = distVector;
               }
            }
            if (saucer != null) {
               ScreenVector distVector = ship.getDistanceTo(saucer);
               distVector.normalize();
               int sqDist = distVector.scalarProduct(distVector);
               // adjust the squared distance for the radius of the flying saucer
               sqDist -= saucer.getSize().getDistanceCorrection();
               if (sqDist < minDistance) {
                  minDistance = sqDist;
                  minDifference = distVector;
               }
            }
         
            if (minDifference != null) {
               // turn the ship towards the nearest object
               if (ship.getOrientation().crossProduct2D(minDifference) > 0) {
                  pressKey(KEY_LEFT);
               } else {
                  pressKey(KEY_RIGHT);
               }
            }
            
            // evade collision with jump to hyperspace
            if (minDistance < 46656) { // screen: 27 * 27
               pressKey(KEY_HYPERSPACE);
            }
         
            // accelerate when no object is near
            if (minDistance > 10240000) { // screen: 400 * 400
               pressKey(KEY_THRUST);
            }
         
            // fire as fast as possible
            if (curFrame % 2 == 0) {
               pressKey(KEY_FIRE);
            }
         }
         sendKeys(0);
      }
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.Player#getInitials()
    */
   public String getInitials() {
      return "HBO"; // Harald Boegeholz
   }
}
