/**
 * File:    HallOfFameEditor.java
 * Package: de.heise.asteroid
 * Created: 24.05.2008 10:41:38
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;


/**
 * @author Chr. Moellenberg
 *
 */
public class HallOfFameEditor extends AbstractProcessor {
   private static final String alphabet = " ABCDEFGHIJKLMNOPQRSTUVWXYZ";
   private static final int SEARCHING = 0;
   private static final int FOUND = 1;

   private String playerInitials;
   private StringBuilder curInitials;
   private int editState;
   private int cursor;
   private int curFrameNo;
   private boolean waiting;
   private int frameToWait;
   private byte lastKey;

   /**
    * Initializes this <code>HallOfFameEditor</code> with the given 
    * <code>Engine</code>.
    * 
    * @param e the <code>Engine</code>
    */
   public HallOfFameEditor(Engine e, String initials) {
      super(e);
      playerInitials = initials;
      curInitials = new StringBuilder();
      initialize();
   }
   
   /* (non-Javadoc)
    * @see de.heise.asteroid.engine.FrameProcessor#initialize()
    */
   public void initialize() {
      cursor = -1;
      waiting = false;
      editState = SEARCHING;
      curFrameNo = 0;
      frameToWait = 0;
      curInitials.setLength(0);
      curInitials.append("   ");
      lastKey = 0;
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.Player#play(de.heise.asteroid.GameStatus, int)
    */
   public void process(GameStatus status, int frameNo) {
      curFrameNo = frameNo;
      if (waiting) {
         if (frameNo < frameToWait) {
            return;
         } else {
            waiting = false;
         }
      }
      String frameIni = status.getInitials();
      if (frameIni == null) {
         System.out.println("No initials found in frame");
         return;
      } else {
         curInitials.replace(0, frameIni.length(), frameIni);
      }
      if (frameIni.length() - 1 > cursor) {
         cursor = frameIni.length() - 1;
         if (cursor < 3) {
            editState = SEARCHING;
            scheduleKey(KEY_NONE);
         } else {
            scheduleKey(KEY_HYPERSPACE);
            System.out.println("Cursor advanced too far");
         }
         return;
      }
      int dir = getLetterDirection(playerInitials.charAt(cursor), curInitials.charAt(cursor));
      if (editState == SEARCHING) {
         switch (dir) {
            case -1:
               scheduleKey(KEY_LEFT);
               break;
            case 0:
               scheduleKey(KEY_NONE);
               editState = FOUND;
               scheduleWaitFor(15);
               break;
            case 1:
               scheduleKey(KEY_RIGHT);
               break;
         }
      } else if (editState == FOUND) {
         if (dir != 0) {
            editState = SEARCHING;
         } else {
            scheduleKey(KEY_HYPERSPACE);
            scheduleWaitFor(5);
         }
      }
   }

   private void scheduleKey(byte key) {
      if (key != lastKey) {
         if (key == KEY_NONE) {
            releaseKeys();
         } else {
            pressKey(key);
         }
         sendKeys(curFrameNo);
         lastKey = key;
      }
   }
   
   private void scheduleWaitFor(int frames) {
      waiting = true;
      frameToWait = curFrameNo + frames;
   }

   private int getLetterDirection(char a, char b) {
      int dist = alphabet.indexOf(b) - alphabet.indexOf(a);
      if (dist == 0) {
         return 0;
      } else if (dist > 0) {
         return (dist > 13) ? -1 : 1;
      } else {
         return (dist < -13) ? 1 : -1;
      }
   }
}
