/**
 * File:    FrameInterpreter.java
 * Package: de.heise.asteroid
 * Created: 18.05.2008 18:47:39
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.engine;

import java.util.LinkedList;
import java.util.List;

import de.heise.asteroid.Position;
import de.heise.asteroid.comm.FramePacket;
import de.heise.asteroid.model.Asteroid;
import de.heise.asteroid.model.Shot;

/**
 * @author Chr. Moellenberg
 *
 */
public class FrameInterpreter {
   List<Asteroid> asteroids;
   List<Shot> shots;
   int state;
   int lifeCount;
   GameStatus status;
   AsteroidSequenceTracker astTracker;
   SaucerTracker ufoTracker;
   ShipTracker shipTracker;
   
   public FrameInterpreter(GameStatus stat) {
      status = stat;
      asteroids = new LinkedList<Asteroid>();
      shots = new LinkedList<Shot>();
      state = Engine.STATE_UNKNOWN;
      lifeCount = 0;
      astTracker = new AsteroidSequenceTracker();
      ufoTracker = new SaucerTracker();
      shipTracker = new ShipTracker();
   }

   public void transitState(int s) {
      state = s;
      astTracker.reset();
      ufoTracker.reset();
      shipTracker.reset();
   }

   /**
    * Fills a <code>GameStatus</code> by interpreting the vector ram contained 
    * in a <code>FramePacket</code>.
    * 
    * @param fp the <code>FramePacket</code> to interpret
    */
   public void interpretFrame(FramePacket fp) {
      int frameNo = fp.getRealFrameNo();
      if (!astTracker.isInSync()) {
         System.out.println("!!!!!!!!!! RESYNCHRONIZING !!!!!!!!!!");
         astTracker.reset();
      }
      astTracker.start(frameNo);
      ufoTracker.start(frameNo);
      shipTracker.start(frameNo);
      status.clear();
      if (state != Engine.STATE_PLAYING) {
         return;
      }
      int dx = 0, dy = 0;
      int px = 0, py = 0;
      int brightness = 0;
      int scale = 0;
      shots.clear();
      lifeCount = 0;

      final int initialJump = fp.getRamAt(0);
      if (initialJump != 0xe001 && initialJump != 0xe201) {
         System.err.println(String.format("Bad instruction in vector RAM[0]: %04x", initialJump));
         return; // should not occur, first instruction is always a jump
      }
      shipTracker.setThrustEnable(initialJump == 0xe001);
      for (int pc = 1; true;) {
         final int currentWord = fp.getRamAt(pc++);
         int op = currentWord >> 12;
         switch (op) {
            case 0xa: { // LABS
               final int nextWord = fp.getRamAt(pc++);
               py = currentWord & 0x3ff;
               px = nextWord & 0x3ff;
               scale = nextWord >> 12;
               break;
            }
            case 0xb: // HALT
               status.clear();
               astTracker.cleanup();
               astTracker.getAsteroids(asteroids, false);
               status.addAsteroids(asteroids);
               status.addShots(shots);
               ufoTracker.cleanup();
               status.setSaucer(ufoTracker.getSaucer());
               shipTracker.cleanup();
               status.setShip(shipTracker.getShip());
               status.setLifeCount(lifeCount);
               return;
            case 0xc: // JSRL
               final int addr = currentWord & 0xfff;
               if (addr >= 0x880 && addr <= 0x929) {
                  int t = getTargetTypeByAddr(addr);
                  if (!ufoTracker.trackTarget(px, py, t, scale)) {
                     astTracker.trackTarget(px, py, t, scale);
                  }
               } else if (addr < 0xa78 || addr > 0xb63) { // ignore any text
                  switch (addr) {
                     case 0x852: // Copyright
                        break;
                     case 0xa6d: // Ship
                        ++lifeCount;
                        break;
                     default:
                        System.err.println(String.format("Unknown vector subroutine %04x", addr));
                  }
               }
               break;
            case 0xd: // RTSL
               System.err.println(String.format("Unexpected RTSL instruction in vector RAM[%03x]", pc - 1));
               return;
            case 0xe: // JMPL
               // pc = currentWord & 0xfff;
               // break;
               System.err.println(String.format("Unexpected JMPL instruction in vector RAM[%03x]", pc - 1));
               return;
            case 0xf: // SVEC
               if (false) {
                  dy = currentWord & 0x300;
                  if ((currentWord & 0x400) != 0)
                     dy = -dy;
                  dx = (currentWord & 3) << 8;
                  if ((currentWord & 4) != 0)
                     dx = -dx;
                  // sf = (((currentWord & 8) >> 2) | ((currentWord & 0x800) >> 11)) + 2;
                  brightness = (currentWord & 0xf0) >> 4;
               }
               break;
            default: { // VCTR
               final int nextWord = fp.getRamAt(pc++);
               dy = currentWord & 0x3ff;
               dx = nextWord & 0x3ff;
               if ((currentWord & 0x400) != 0) {
                  dy = -dy;
               }
               if ((nextWord & 0x400) != 0) {
                  dx = -dx;
               }
               // sf = op;
               brightness = nextWord >> 12;
               if (dx == 0 && dy == 0 && brightness == 15) {
                  Shot shot = new Shot(new Position(px << 3, py << 3));
                  shots.add(shot);
               } else if (op == 6 && brightness == 12 && dx != 0 && dy != 0) {
                  shipTracker.trackShip(px, py, dx, dy, fp.getKeys() & FrameProcessor.KEY_MASK);
               }
               break;
            }
         }
      }
   }
   
   private static int getTargetTypeByAddr(int addr) {
      switch (addr) {
         case 0x880:
            return GameStatus.EXPL_3;
         case 0x896:
            return GameStatus.EXPL_2;
         case 0x8b5:
            return GameStatus.EXPL_1;
         case 0x8d0:
            return GameStatus.EXPL_0;
         case 0x8f3:
            return GameStatus.TYPE_1;
         case 0x8ff:
            return GameStatus.TYPE_2;
         case 0x90d:
            return GameStatus.TYPE_3;
         case 0x91a:
            return GameStatus.TYPE_4;
         case 0x929:
            return GameStatus.SAUCER;
         default:
            System.out.printf("Invalid address 0x%03x for a target", addr);
            return -1;
      }
   }
}
