package de.heise.asteroid.comm;

import java.net.DatagramPacket;

/**
 * The <code>FramePacket</code> class represents the server reply packet. It
 * contains the vector ram, the one-byte frame number and the ping value.
 * 
 * @see ServerConnection
 */
public class FramePacket {
   public static final int SIZE = 1026;

   private byte[] payload;
   private DatagramPacket datagramPacket;
   private int realFrameNo;
   private long timeStamp;
   private byte keys;

   /**
    * Standard constructor
    */
   public FramePacket() {
      payload = new byte[SIZE];
      datagramPacket = new DatagramPacket(payload, payload.length);
   }

   /**
    * Returns the <code>DatagramPacket</code> for receiving the network data
    * 
    * @return the <code>DatagramPacket</code>
    * @see ServerConnection
    */
   public DatagramPacket getDatagramPacket() {
      return datagramPacket;
   }

   /**
    * Returns the frame number. The frame number gets increased for every frame
    * the server sends. It can be used to detect missing frames or wrong frame
    * order.
    * 
    * @return the frame number
    */
   public byte getFrameNo() {
      return payload[SIZE - 2];
   }

   /**
    * Returns the one-byte ping value. The server returns the ping value from last 
    * processed <code>KeyPacket</code>. Use this to detect network latency.
    * 
    * @return the one-byte ping value
    * @see KeyPacket#setPing(byte)
    * @see KeyPacket#getPing()
    */
   public byte getPing() {
      return payload[SIZE - 1];
   }

   /**
    * Returns the real frame number.
    * 
    * @return the real frame number of this packet
    */
   public int getRealFrameNo() {
      return realFrameNo;
   }

   /**
    * Stores the real frame number (i.e. without one-byte wrapping). This number 
    * is tracked by the <code>Communicator</code>.
    * 
    * @param frameNo the real frame number to set
    */
   public void setRealFrameNo(int frameNo) {
      realFrameNo = frameNo;
   }

   /**
    * Returns the time stamp of this packet.
    * @return the time stamp
    */
   public long getTimeStamp() {
      return timeStamp;
   }

   /**
    * Sets a time stamp for this packet. It is set by the <code>ServerConnection</code> 
    * every time after a packet was received at the socket. 
    * 
    * @param stamp the timeStamp to set
    */
   public void setTimeStamp(long stamp) {
      timeStamp = stamp;
   }

   /**
    * Allows access to the 16-bit vector ram words.
    * 
    * @param index the index for the word
    * @return the 16-bit word at index <code>index</code>
    */
   public int getRamAt(int index) {
      return (payload[2 * index] & 0xff) | (payload[2 * index + 1] & 0xff) << 8;
   }

   /**
    * Returns the keys which were processed by the server immediately before this frame.
    * 
    * @return k the keys as defined by the <code>FrameProcessor</code> interface
    * @see FramePacket#setKeys() 
    */
   public byte getKeys() {
      return keys;
   }

   /**
    * Sets the keys which were processed by the server immediately before this frame. 
    * The key value is actually stored in the <code>Communicator</code> for each ping 
    * value transmitted. When the ping value comes back from the server, the 
    * <code>Communicator</code> stores the keys here for further processing by the 
    * player <code>Engine</code>.
    * 
    * @param k the keys as defined by the <code>FrameProcessor</code> interface
    */
   public void setKeys(byte k) {
      keys = k;
   }
}
