/**
 * File:    Communicator.java
 * Package: de.heise.asteroid
 * Created: 09.05.2008 21:07:54
 * Author:  Chr. Moellenberg
 *
 * Copyright (c) 2008 by Chr. Moellenberg
 */

package de.heise.asteroid.comm;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;

import de.heise.asteroid.engine.FrameProcessor;
import de.heise.asteroid.util.WorkerRunnable;

/**
 * @author Chr. Moellenberg
 *
 */
public class Communicator implements WorkerRunnable, FramePacketProvider {
   private ServerConnection srvConn;
   private List<FramePacket> fpQueue;
   private List<KeyPacket> kpQueue;
   private int expFrameNo;
   private int realFrameNo;
   private byte keys;
   private byte ping;
   private int latency;
   private byte[] keyHistory;

   public Communicator(ServerConnection conn) {
      srvConn = conn;
      fpQueue = null;
      kpQueue = null;
      keyHistory = new byte[256];
   }

   /**
    * Sets the packet queues.
    * @param fpq the <code>FramePacket</code> queue to set
    * @param kpq the <code>KeyPacket</code> queue to set
    */
   public void setQueues(List<FramePacket> fpq, List<KeyPacket> kpq) {
      fpQueue = fpq;
      kpQueue = kpq;
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.util.WorkerRunnable#threadInitialize()
    */
   public void workerInitialize() {
      System.out.println("Communicator started.");
      ping = 0;
      keys = '@';
      expFrameNo = -2;
      latency = 0;
      // send initial KeyPacket to trigger the frame transfer
      KeyPacket kp = srvConn.getKeyPacket();
      kp.setKeys(keys);
      kp.setPing(ping);
      keyHistory[0] = keys;
      srvConn.send(kp);
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.util.WorkerRunnable#threadMainLoop()
    */
   public boolean workerMainLoop() {
      FramePacket fp = receiveFramePacket();
      if (fp != null) {
         enqueueFramePacket(fp);
         transmitKeyPacket();
         return true;
      }
      return false;
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.util.WorkerRunnable#threadFinalize()
    */
   public void workerFinalize() {
      System.out.printf("Communicator stopped after %d frames.\n", srvConn.getPacketsRcvd());
      System.out.printf("Max. packets used: %d (F) / %d (K)\n", srvConn.getFramePacketCount(), srvConn.getKeyPacketCount());
      System.out.printf("Packets transferred: %d (F) / %d (K)\n", srvConn.getPacketsRcvd(), srvConn.getPacketsSent());
   }

   /* (non-Javadoc)
    * @see de.heise.asteroid.comm.FramePacketProvider#receiveFramePacket()
    */
   public FramePacket receiveFramePacket() {
      FramePacket fp = null;
      try {
         fp = srvConn.receive();
      } catch (IOException e) {
         System.err.println("Failed to receive a packet, giving up (" + e.getMessage() + ").");
      }
      if (fp != null) {
         checkLatency(fp);
         ++expFrameNo;
         realFrameNo = getRealFrameNo(fp);
         expFrameNo = realFrameNo;
      }
      return fp;
   }
   
   private void enqueueFramePacket(FramePacket fp) {
      if (fpQueue != null) {
         synchronized (fpQueue) {
            fpQueue.add(fp);
            fpQueue.notify();
         }
      } else {
         srvConn.disposeFramePacket(fp);
      }
   }

   private KeyPacket dequeueKeyPacket() {
      KeyPacket kp = null;
      if (kpQueue != null) {
         synchronized (kpQueue) {
            if (!kpQueue.isEmpty() && (realFrameNo + latency) >= kpQueue.get(0).getTargetFameNo()) {
               kp = kpQueue.remove(0);
               keys = kp.getKeys();
            }
         }
      }
      return kp;
   }

   public void transmitKeyPacket() {
      KeyPacket kp = dequeueKeyPacket();
      if (kp == null) {
         kp = srvConn.getKeyPacket();
         // make sure fire and hyperspace keys are released as early as possible
         keys &= ~(FrameProcessor.KEY_FIRE | FrameProcessor.KEY_HYPERSPACE);
         kp.setKeys(keys);
      }
      kp.setPing(++ping);
      keyHistory[(int)ping & 0xff] = keys;
      srvConn.send(kp);
   }

   /**
    * Returns the real frame number (i.e. without byte wrapping) of the given <code>FramePacket</code>
    * @param fp the <code>FramePacket</code>
    * @return the frame number
    */
   private int getRealFrameNo(FramePacket fp) {
      int fpFrameNo = fp.getFrameNo() & 0xff;
      if (expFrameNo < 0) {
         expFrameNo = fpFrameNo;
      } else if ((expFrameNo & 0xff) == fpFrameNo) {
         fpFrameNo = expFrameNo;
      } else {
         int offset = (int)fp.getFrameNo() - (expFrameNo & 0xff);
         if (offset < -128) {
            offset += 256;
         } else if (offset > 127) {
            offset -= 256;
         }
         fpFrameNo = expFrameNo + offset;
         System.out.printf("Expected frame number %d but received %d\n", expFrameNo, fpFrameNo);
      }
      fp.setRealFrameNo(fpFrameNo);
      return fpFrameNo;
   }

   /**
    * Computes the latency between key packets sent and frame packets received.
    * @param fp the <code>FramePacket</code>
    * @see FramePacket#getPing()
    */
   private void checkLatency(FramePacket fp) {
      int fpPing = (int)fp.getPing();
      fp.setKeys(keyHistory[fpPing & 0xff]);
      int lat = (int)ping - fpPing;
      if (lat < -128) {
         lat += 256;
      } else if (lat > 127) {
         lat -= 256;
      }
      if (lat != latency) {
         latency = lat;
//         System.out.printf("Latency %d, %d frames pending.\n", latency, getPendingFrames());
      }
   }

   public int getLatency() {
      return latency;
   }

   public int getPendingFrames() {
      if (fpQueue != null) {
         synchronized (fpQueue) {
            return fpQueue.size();
         }
      } else {
         return -1;
      }
   }

   public void recycle(FramePacket fp) {
      srvConn.disposeFramePacket(fp);
   }

   public KeyPacket getKeyPacket() {
      return srvConn.getKeyPacket();
   }

   public void disposeKeyPacket(KeyPacket kp) {
      srvConn.disposeKeyPacket(kp);
   }
   
   public void flushKeyQueue() {
      if (kpQueue != null) {
         synchronized (kpQueue) {
            while (!kpQueue.isEmpty()) {
               srvConn.disposeKeyPacket(kpQueue.remove(0));
            }
            keys = '@';
         }
      }
   }

   public void dumpKeyQueue() {
      synchronized (kpQueue) {
         System.out.printf("Key queue has %d elements", kpQueue.size());
         Iterator<KeyPacket> it = kpQueue.iterator();
         while (it.hasNext()) {
            KeyPacket kp = it.next();
            System.out.printf("%5d: %02x", kp.getTargetFameNo(), kp.getKeys());
         }
      }
   }
}
