package de.heise.asteroid;

import de.heise.asteroid.util.SimpleMath;

/**
 * The <code>ScreenVector</code> class represents simple 2D vector.
 * <p>
 * It is a value object which, once created, cannot be changed. If a
 * <code>Position</code> becomes invalid, it is simply thrown away. 
 * Some basic vector operations are provided.
 */
public class ScreenVector {
   public static final ScreenVector NULL_VECTOR = new ScreenVector(0, 0);

   protected static int minX = -4096;
   protected static int maxX = 4096;
   protected static int minY = -3072;
   protected static int maxY = 3072;

   protected int x;
	protected int y;

	/**
	 * Initializes the <code>Vector</code> to (dx, dy).
	 * @param dx the x component
	 * @param dy the y component
	 */
	public ScreenVector(int dx, int dy) {
		x = dx;
		y = dy;
	}

	/**
	 * Returns the x component.
	 * @return x component
	 */
	final public int getX() {
		return x;
	}

   /**
    * Returns the y component.
    * @return y component
    */
   final public int getY() {
      return y;
   }

   /**
    * Add vector <code>a</code> to <code>this</code>.
    * @param a the vector to add
    * @return the vector pointing from the origin to (<code>this</code> + <code>a</code>)
    */
   public ScreenVector sum(ScreenVector a) {
      return new ScreenVector(this.x + a.x, this.y + a.y);
   }

	/**
	 * Subtract vector <code>a</code> from <code>this</code>.
	 * @param a the vector to subtract
	 * @return the distance vector pointing from <code>this</code> to <code>a</code>
	 */
	public ScreenVector diff(ScreenVector a) {
		return new ScreenVector(this.x - a.x, this.y - a.y);
	}

	/**
	 * Calculates the scalar product of <code>this</code> and vector <code>a</code>.
	 * @param a vector <code>a</code>
	 * @return the scalar product of <code>this</code> and vector <code>a</code>
	 */
	public int scalarProduct(ScreenVector a) {
		return this.x * a.x + this.y * a.y;
	}

	/**
	 * Calculates the 2D cross product of <code>this</code> and vector <code>a</code>.
	 * @param a vector <code>a</code>
	 * @return the 2D cross product of <code>this</b> and vector <code>a</code>
	 */
	public int crossProduct2D(ScreenVector a) {
		return this.x * a.y - this.y * a.x;
	}
	
	/**
	 * @return the squared absolute value of this vector
	 */
	public int sqAbs() {
	   return x * x + y * y;
	}

	/**
	 * @return the exact absolute value of this vector
	 */
	public double abs() {
	   return Math.sqrt(x * x + y * y);
	}

	/** 
    * Normalizes the vector.
    * The resulting vector holds <code>(minX <= x <= maxX, minY <= y <= maxY)</code>.
    */
   public void normalize() {
      x = SimpleMath.normalize(x, minX, maxX);
      y = SimpleMath.normalize(y, minY, maxY);
   }
   
   public static void setNormalization(int minx, int maxx, int miny, int maxy) {
      minX = minx;
      maxX = maxx;
      minY = miny;
      maxY = maxy;
   }

   /* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
   public String toString() {
      StringBuilder str = new StringBuilder();
      str.append(x).append(",").append(y);
      return str.toString();
   }
}
