package core;

import java.util.ArrayList;

import util.Statistics;
import core.asteroid.Asteroid;
import core.asteroid.AsteroidObject;
import core.asteroid.Decoration;
import core.asteroid.Ship;
import core.asteroid.Shot;
import core.asteroid.Ufo;
import core.opCodes.OpLocation;
import core.opCodes.OpVector;

public class ScreenInterpreter {
	
	public static final int GAME_MODE_DEMO			= 1;
	public static final int GAME_MODE_HIGHSCORE 	= 2;
	public static final int GAME_MODE_ENTER_NAME 	= 3;
	public static final int GAME_MODE_PREPARING		= 4;
	public static final int GAME_MODE_GAME			= 5;
	
	public static final int ADDR_NO_ADDRESS 		= 0;
	public static final int ADDR_ASTEROID_1 		= 0x8F3;
	public static final int ADDR_ASTEROID_2 		= 0x8FF;
	public static final int ADDR_ASTEROID_3 		= 0x90D;
	public static final int ADDR_ASTEROID_4 		= 0x91A;
	public static final int ADDR_UFO        		= 0x929;
	public static final int ADDR_PRESS_START		= 0xAFB;
	public static final int ADDR_HIGHSCORE			= 0xAB3;
	public static final int ADDR_ENTER_NAME			= 0xABA;
	public static final int ADDR_LIFE				= 0xA6D;
	public static final int ADDR_SPACE				= 0xB2C;
	public static final int ADDR_0					= 0xADD;
	public static final int ADDR_1					= 0xB2E;
	public static final int ADDR_2					= 0xB32;
	public static final int ADDR_3					= 0xB3A;
	public static final int ADDR_4					= 0xB41;
	public static final int ADDR_5					= 0xB48;
	public static final int ADDR_6					= 0xB4F;
	public static final int ADDR_7					= 0xB56;
	public static final int ADDR_8					= 0xB5B;
	public static final int ADDR_9					= 0xB63;
	

	private int[] vectorRom;
	private int[] screenMem;
	private int gameMode = GAME_MODE_DEMO;
	private int prevScore = 0;
	private int scoreOffset = 0;
	
	
	public ScreenInterpreter(int[] vectorRom) {
		this.vectorRom = vectorRom;
	}
	
	
	private AsteroidObject createAsteroidObject(OpLocation loc, ArrayList<OpVector> vecs, int romAddr) {
		AsteroidObject o;
		
		loc.y -= 128;
		
		switch(romAddr) {
		case ADDR_ASTEROID_1:
		case ADDR_ASTEROID_2:
		case ADDR_ASTEROID_3:
		case ADDR_ASTEROID_4:
			o = new Asteroid(loc, vecs);
			break;
			
		case ADDR_UFO:
			o = new Ufo(loc, vecs);
			break;
			
		case ADDR_NO_ADDRESS:
			if(vecs.size() == 1) {
				// Nur ein Vector -> Ein Schuss
				o = new Shot(loc, vecs);
			} else {
				int hiddenVecCnt = 0;
				for(int i=0; i<vecs.size() && i<5; i++) {
					OpVector v = vecs.get(i);
					if(v.z == 0) hiddenVecCnt++;
				}
				if(hiddenVecCnt == 1) {
					// Nur ein unsichtbarer Vector -> Das Schiff
					o = new Ship(loc, vecs);
				} else {
					// Mehrere unsichtbare Vectoren -> Schiffsexplosion
					o = new Decoration(loc, vecs);
				}
			}
			break;
			
		default:
			o = new Decoration(loc, vecs);
		}
		
		o.setRomAddress(romAddr);
		
		return o;
	}
	
	
	private int detectGameMode(AsteroidObject obj, int curGameMode) {
		int newGameMode = curGameMode;
		switch(obj.getRomAddress()) {
		case ADDR_ENTER_NAME:
			newGameMode = GAME_MODE_ENTER_NAME;
			break;
			
		case ADDR_HIGHSCORE:
			newGameMode = GAME_MODE_HIGHSCORE;
			break;
		}
		return newGameMode;
	}
	
	
	public ArrayList<AsteroidObject> interpretScreen(int[] screenMem) {
		int pc = 0;
		boolean halt = false;
		int[] stack = new int[4];
		int stackPtr = 0;
		char[] scoreChars = new char[5];
		int scorePtr = 0;
		int score = 0;
		int lifes = 0;
		int newGameMode = GAME_MODE_DEMO;
		boolean lifesPresent = false;
		boolean shipPresent = false;
		
		ArrayList<AsteroidObject> objs = new ArrayList<AsteroidObject>();
		ArrayList<OpVector> vecs = new ArrayList<OpVector>();
		
		OpLocation loc = null;
		int romAddr = 0;
		boolean scoreLABS = false;

		this.screenMem = screenMem;
		for(int i=0; i<5; i++) scoreChars[i] = ' ';
		
		while(!halt) {
			int memOffset = 0;
			int[] mem;
			
			if(pc < screenMem.length) {
				mem = screenMem;
				memOffset = 0;
			} else if(pc-512 < screenMem.length) {
				mem = screenMem;
				memOffset = 512;
			} else {
				mem = vectorRom;
				memOffset = 0x800;
			}
			
			int w1 = mem[pc-memOffset];
			int w2 = 0;
			int op = (w1 & 0xF000) >> 12;
			if(op <= 10) w2 = mem[pc-memOffset+1];
			
			if(op < 10) {
				// VCTR
				OpVector v = MemoryParser.parseVCTR(w1, w2);
				if(v.x != 0 || v.y != 0 || v.z != 0) {
					vecs.add(v);
				}
				pc += 2;
				
			} else if(op == 10) {
				// LABS
				if(loc != null && vecs.size() > 0) {
					AsteroidObject o = createAsteroidObject(loc, vecs, romAddr);
					newGameMode = detectGameMode(o, newGameMode);
					if(o instanceof Ship) shipPresent = true;
					objs.add(o);
					vecs = new ArrayList<OpVector>();
					romAddr = 0;
				}
				loc = MemoryParser.parseLABS(w1, w2);
				pc +=2;

				// Punkte Anzeige
				if(loc.x == 100 && loc.y == 876) {
					scoreLABS = true;
				}
				
			} else if(op == 11) {
				// HALT
				halt = true;
				
			} else if(op == 12) {
				// JSRL
				stack[stackPtr++] = pc+1;
				pc = w1 & 0xFFF;
				
				// Ein Raumschiff Symbol pro Leben - Diese Anzeige existiert nicht im Demo Mode...
				if(pc == 0xA6D) {
					lifes++;
					lifesPresent = true;
				}
				// Punkteanzeige parsen
				if(scoreLABS) {
					switch(pc) {
					case ADDR_SPACE:	scoreChars[scorePtr++] = ' ';	break;
					case ADDR_0:		scoreChars[scorePtr++] = '0';	break;
					case ADDR_1:		scoreChars[scorePtr++] = '1';	break;
					case ADDR_2:		scoreChars[scorePtr++] = '2';	break;
					case ADDR_3:		scoreChars[scorePtr++] = '3';	break;
					case ADDR_4:		scoreChars[scorePtr++] = '4';	break;
					case ADDR_5:		scoreChars[scorePtr++] = '5';	break;
					case ADDR_6:		scoreChars[scorePtr++] = '6';	break;
					case ADDR_7:		scoreChars[scorePtr++] = '7';	break;
					case ADDR_8:		scoreChars[scorePtr++] = '8';	break;
					case ADDR_9:		scoreChars[scorePtr++] = '9';	break;
					}
					if(scorePtr == 5) {
						scorePtr = 0;
						scoreLABS = false;
						if(scoreChars[4] != ' ') {
							String s = new String(scoreChars);
							score = Integer.parseInt(s.trim());
						} else {
							score = 0;
						}
					}
				}
				// Copyright Meldung Springt ohne vorheriges LABS
				if(pc == 0x852 && loc != null && vecs.size() > 0) {
					objs.add(createAsteroidObject(loc, vecs, romAddr));
					vecs = new ArrayList<OpVector>();
					romAddr = 0;
				}
				if(romAddr == 0) romAddr = pc;
				
			} else if(op == 13) {
				// RTSL
				pc = stack[--stackPtr];
				
			} else if(op == 14) {
				// JMPL
				pc = w1 & 0xFFF;
				
			} else if(op == 15) {
				// SVEC
				OpVector v = MemoryParser.parseSVEC(w1);
				if(v.x != 0 || v.y != 0 || v.z != 0) {
					vecs.add(v);
				}
				pc++;
				
			}
		}
		// Letztes Objekt hinzufgen
		if(loc != null && vecs.size() > 0) {
			AsteroidObject o = createAsteroidObject(loc, vecs, romAddr);
			newGameMode = detectGameMode(o, newGameMode);
			if(o instanceof Ship) shipPresent = true;
			objs.add(o);
		}
		
		if(lifesPresent && !shipPresent && newGameMode == GAME_MODE_DEMO && gameMode != GAME_MODE_GAME)
			newGameMode = GAME_MODE_PREPARING;
		else if(lifesPresent && newGameMode == GAME_MODE_DEMO && (gameMode == GAME_MODE_GAME || shipPresent))
			newGameMode = GAME_MODE_GAME;
		gameMode = newGameMode;
		
		Statistics.putFloat(Statistics.KEY_LIFES, lifes);

		if(gameMode == GAME_MODE_GAME) {
			if(prevScore > score) scoreOffset += 100000;
			prevScore = score;
			Statistics.putFloat(Statistics.KEY_SCORE, score+scoreOffset);
		} else {
			prevScore = 0;
			scoreOffset = 0;
		}

		String gm = "";
		switch(gameMode) {
		case ScreenInterpreter.GAME_MODE_DEMO: 			gm = "Demo";			break;
		case ScreenInterpreter.GAME_MODE_HIGHSCORE:		gm = "Highscore";		break;
		case ScreenInterpreter.GAME_MODE_ENTER_NAME:	gm = "Name";			break;
		case ScreenInterpreter.GAME_MODE_PREPARING:		gm = "Vorbereitung";	break;
		case ScreenInterpreter.GAME_MODE_GAME:			gm = "Spiel";			break;
		}
		Statistics.put(Statistics.KEY_GAME_MODE, gm);
		
		return objs;
	}
	
	
	public int getGameMode() {
		return gameMode;
	}
	
	
	public void dumpScreenMem() {
		MemoryParser.dumpMemory(screenMem, 0);
	}
}
