﻿/*
 * Asteroids Player Client 
 * 2008 ct-Wettbewerb
 * 
 * by Stefc (stefan.boether@gmail.com) */

using System;
using System.Collections.Generic;
using System.Text;

namespace stefc.asteroids
{
    public class Vector
    {
        private double length;
        private double angle;

        private Vector(double length, double angle)
        {
            this.length = length;
            this.angle = angle;
        }

        public override string ToString()
        {
            return String.Format("(x={0};y={1})  (angle={2};len={3})",
                X,Y,Angle,Length);
        }

        public static Vector CreateCartesian(int x, int y)
        {
            return new Vector(Math.Sqrt(x * x + y * y), MathUtils.Rad2Deg(Math.Atan2(y, x)));
        }

        public static Vector CreateCartesian(double x, double y)
        {
            return new Vector(Math.Sqrt(x * x + y * y), MathUtils.Rad2Deg(Math.Atan2(y, x)));
        }

        public static Vector CreatePolar(double value, double angle)
        {
            return new Vector(value, angle);
        }

        
        public double Length
        {
            get { return Math.Round(length); }
        }

        public double Angle
        {
            get { return Math.Round(angle); }
        }

        public int X
        {
            get { return (int)Math.Round(length * Math.Cos(MathUtils.Deg2Rad(angle))); }
        }

        public int Y
        {
            get { return (int)Math.Round(length * Math.Sin(MathUtils.Deg2Rad(angle))); }
        }

        public static Vector operator +(Vector v1, Vector v2)
        {
            double x =
                ((v1.length * Math.Cos(MathUtils.Deg2Rad(v1.angle))) +
                 (v2.length * Math.Cos(MathUtils.Deg2Rad(v2.angle))));
            double y =
                ((v1.length * Math.Sin(MathUtils.Deg2Rad(v1.angle))) +
                 (v2.length * Math.Sin(MathUtils.Deg2Rad(v2.angle))));

            double length = Math.Sqrt(x * x + y * y);
            double angle = MathUtils.Rad2Deg(Math.Atan2(y, x));

            return new Vector(length, angle);
        }
    }

    public static class MathUtils
    {
        public static double Rad2Deg(double radians)
        {
            return ((180.0 / Math.PI) * radians);
        }

        public static double Deg2Rad(double degree)
        {
            return (Math.PI / 180) * degree;
        }

        public static int InnerCircleAngle(int segments)
        {
            return (360 / segments);
        }

        public static int OuterCircleAngle(int segments)
        {
            return (180 - (360 / segments)) / 2;
        }

        public static int Sekante(int radius, int alpha)
        {
            return (int)(2 * radius * Math.Sin(Deg2Rad(alpha) / 2));
        }

        public static double AngleDelta(double a, double b)
        {
            if (a < 0.0)
                a = 180.0 + (180.0 - Math.Abs(a));
            if (b < 0.0)
                b = 180.0 + (180.0 - Math.Abs(b));

            double delta = a - b;

            if (delta < -180)
                return 360 + delta;
            else if (delta > 180)
                return delta - 360;

            return delta;
        }

        public static double NormalizeAngle(double angle)
        {
            while ((angle < -180) || (angle > +180))
            {
                if (angle < -180)
                    angle += 360;
                if (angle > +180)
                    angle -= 360;
            }
            return angle;
        }


        /// <summary>
        /// Berechnet den besten Abschusswinkel für einen Treffer
        /// </summary>
        /// <param name="a">Vector vom Raumschiff zum Ziel</param>
        /// <param name="b">Vector Ziel-Bewegung</param>
        /// <param name="speed">Betrag Schussgeschwindigkeit</param>
        /// <returns>Schussvector</returns>
        public static Vector GetDeltaAngle(Vector a, Vector b, double speed)
        {
            double a1 = a.X;
            double a2 = a.Y;
            double b1 = b.X;
            double b2 = b.Y;

            double n = (b1 * b1) + (b2 * b2);
            double m = (a1 * a1) + (a2 * a2);
            double k = (a1 * b1) + (a2 * b2);

            double ff = (speed*speed) - 1.0;  
            double ffff = ff * ff;  

            double H = (m/((ff)*n))+((k*k)/(ffff*(n*n)));

            if (H <= 0.0000001)   // Es gibt keine Lösung wenn H negativ oder 0 ist!!!
                return null;

            double t1 = (k / (ff * n)) + Math.Sqrt(H);
            double t2 = (k / (ff * n)) - Math.Sqrt(H);

            double t;
            if (t1 > 0 && t2 > 0)
                t = Math.Min(t1, t2);
            else
                t = Math.Max(t1, t2);

            double s1 = a1 + (t * b1);
            double s2 = a2 + (t * b2);
            
            return Vector.CreateCartesian((int)(s1+0.5), (int)(s2+0.5));
        }

        /// <summary>
        /// Berechnet den Minimalwinkel um ein Objekt zu treffen
        /// </summary>
        /// <param name="target">Zielvektor</param>
        /// <param name="radiusA">Radius von sich selbst</param>
        /// <param name="radiusB">Radius von dem Zielobjekt</param>
        /// <returns></returns>
        internal static double GetHitAngle(Vector target, int radiusA, int radiusB)
        {
            return (int)Rad2Deg(Math.Atan((double)(radiusA + radiusB) / target.Length));
        }

        /// <summary>
        /// Minimalwinkel des Ziels
        /// </summary>
        /// <param name="target"></param>
        /// <param name="radius"></param>
        /// <returns></returns>
        internal static double GetHitAngle(Vector target, double radius)
        {
            return (int)Rad2Deg(Math.Atan(radius / target.Length));
        }
    }

}

