/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.client;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import de.hfaber.asteroids.bot.IAsteroidsBot;
import de.hfaber.asteroids.game.state.GameStatus;
import de.hfaber.asteroids.game.state.InvalidVectorRamException;

/**
 * The mame client that communicates with the mame server.
 * 
 * @author Henning Faber
 */
public class MameClient {

    /**
     * The LOGGER instance. 
     */
    private static final Logger LOGGER = Logger.getLogger(MameClient.class);
    
    /**
     *  
     */
    public static final int DYNAMIC_LATENCY_CORRECTION = -1;
    
    /**
     * The size of a valid data packet for receiving data from the mame 
     * server.
     */
    private static final int RECEIVE_DATA_SIZE = 1026;
    
    /**
     * The maximum size of a data packet that may be sent to the 
     * mame server.
     */
    private static final int MAX_SEND_DATA_SIZE = 40;
    
    /**
     * The length of the player name as expected by the mame server 
     * when sending a name packet.
     */
    private static final int PLAYER_NAME_LENGTH = 32;

    /**
     * The signature that precedes each data packet sent to the mame server
     */
    private static final byte[] KEY_PACKET_SIGNATURE 
        = new byte[] {'c', 't', 'm', 'a', 'm', 'e'};

    /**
     * The signature that precedes a data packet for sending the player
     * name to the server.
     */
    private static final byte[] NAME_PACKET_SIGNATURE 
        = new byte[] {'c', 't', 'n', 'a', 'm', 'e'};

    /**
     * The message that is send by the server, if the game is over. 
     */
    private static final String GAME_OVER_MESSAGE = "game over\r\n";
    
    /**
     * The message that is send by the server, if the server is busy.  
     */
    private static final String SERVER_BUSY_MESSAGE = "busy";
    
    /**
     * The number of frames that a received game status is projected 
     * into the future, before it is passed to the bot. A value of zero 
     * disables latency correction completely, setting it to 
     * {@link #DYNAMIC_LATENCY_CORRECTION} activates dynamic latency 
     * correction. 
     */
    private final int m_latencyCorrection;
    
    /**
     * the socket used to send and receive data 
     */
    private final DatagramSocket m_socket;

    /**
     * The address of the mame server this client is talking to. 
     */
    private final InetSocketAddress m_serverAddress;

    /** 
     * Reusable datagram for sending packets to the mame server
     */
    private final DatagramPacket m_sendDatagram;
    
    /** 
     * Reusable datagram for receiving data from the mame server. 
     */
    private final DatagramPacket m_receiveDatagram;
    
    /**
     * The asteroids bot that is executed by this client.
     */
    private final IAsteroidsBot m_bot;
    
    /**
     * The list of registered game status listeners. 
     */
    private final List<IGameStatusListener> m_gameStatusListenerList;

    /**
     * Creates a mame client that talks to a mame application on the
     * given server host.
     *  
     * @param serverAddress the address of the mame server, to which this 
     *  client should connect
     * @param bot the bot implementation that will be queried for
     *  commands
     * @param latencyCorrection The number of frames the client should
     *  project each game status into the future, before passing it to
     *  the bot. Zero deactivates any latency correction, a positive
     *  value activates constant latency correction, setting 
     *  {@link #DYNAMIC_LATENCY_CORRECTION} activates dynamic latency 
     *  correction.
     * @throws IOException if there is a problem with the network 
     */
    public MameClient(InetSocketAddress serverAddress, IAsteroidsBot bot, 
            int latencyCorrection) throws IOException {
        super();
        
        // set latency correction
        m_latencyCorrection = latencyCorrection;
        
        // create the game status listener list
        m_gameStatusListenerList = new ArrayList<IGameStatusListener>();

        // set the bot and register it 
        m_bot = bot;
        addGameStatusListener(bot);
        
        // create datagrams
        m_receiveDatagram = new DatagramPacket(new byte[RECEIVE_DATA_SIZE], 
                RECEIVE_DATA_SIZE);
        m_sendDatagram = new DatagramPacket(new byte[MAX_SEND_DATA_SIZE],
                MAX_SEND_DATA_SIZE);
        
        // create socket and connect it
        m_socket = new DatagramSocket();
        m_serverAddress = serverAddress;
        m_socket.connect(m_serverAddress);
    }

    /**
     * Starts the mame client.
     * 
     * @throws IOException if there is a problem with the network
     * @throws ServerBusyException if the server is busy 
     */
    public final void run() throws IOException, ServerBusyException {
        Map<Byte, Commands> commandMap = new HashMap<Byte, Commands>(342);  // (int)256 / 0.75
        byte ping = 0;
        GameStatus prevGameStatus = null;
        
        // send initial packet
        Commands commands = new Commands();
        sendCommands(commands, ping, commandMap);
        
        // receive initial game status
        // if the server is busy, ServerBusyException will be thrown here
        GameStatus gs = getGameStatus(prevGameStatus, commandMap);
        
        // send player name
        sendPlayerName("Sitting Duck");
        
        // continue, until game is over
        while (gs != null) {
            
            GameStatus latencyCorrectedGameStatus = gs;
            
            // check for lost frames and high latency
            if ((prevGameStatus != null)) {
                
                // frames lost?
                byte prevServerFrame = prevGameStatus.getServerFrame();
                int frameLoss = gs.getServerFrame() - prevServerFrame - 1;
                if ((frameLoss > 0) && LOGGER.isDebugEnabled()) {
                    StringBuilder sb = new StringBuilder();
                    sb.append("Frame loss detected: ");
                    sb.append(frameLoss);
                    sb.append(frameLoss == 1 ? " frame" : " frames");
                    sb.append(" lost");
                    LOGGER.debug(sb.toString());
                }
                
                // latency > 0?
                int latency = ping - gs.getPing();
                if (latency > 0) {
                    if (LOGGER.isDebugEnabled()) {
                        StringBuilder sb = new StringBuilder();
                        sb.append("Latency detected: ");
                        sb.append(latency);
                        sb.append(latency == 1 ? " frame" : " frames");
                        LOGGER.debug(sb.toString());
                    }

                    // perform dynamic latency correction, if activated
                    if (m_latencyCorrection == DYNAMIC_LATENCY_CORRECTION) {
                        latencyCorrectedGameStatus = new GameStatus(gs, 
                                commandMap, ping, latency);
                    }
                }
            }

            // perform constant latency correction, if activated
            if (m_latencyCorrection > 0) {
                latencyCorrectedGameStatus = new GameStatus(gs, commandMap, 
                        ping, m_latencyCorrection);
            }
            
            // fire game status update event
            fireGameStatusListeners(latencyCorrectedGameStatus);
            
            // determine next move and send it to the mame server
            ping++;
            commands = m_bot.compute();
            sendCommands(commands, ping, commandMap);
            
            // remember last game status and receive the next one
            prevGameStatus = gs;
            gs = getGameStatus(prevGameStatus, commandMap);
        } 
    }

    /**
     * Adds a game status listener to the client.
     * 
     * @param listener the listener to add
     */
    public final void addGameStatusListener(IGameStatusListener listener) {
        if (!m_gameStatusListenerList.contains(listener)) {
            m_gameStatusListenerList.add(listener);
        }
    }
    
    /**
     * Removes a game status listener from the clients listener list.
     * 
     * @param listener the listener to remove
     */
    public final void removeGameStatusListener(IGameStatusListener listener) {
        m_gameStatusListenerList.remove(listener);
    }
    
    /**
     * Creates a new GameStatusEvent for the given game status and
     * fires update events to all registered listeners. 
     * 
     * @param gs the game status
     */
    private void fireGameStatusListeners(GameStatus gs) {
        GameStatusEvent gameStatusEvent = new GameStatusEvent(this, gs);
        for (IGameStatusListener listener : m_gameStatusListenerList) {
            listener.gameStatusUpdated(gameStatusEvent);
        }
    }
    
    /**
     * Receives the next available frame from the mame server,
     * interprets it into a game status and returns this status.
     * 
     * @param prevGameStatus the previous game status that is used
     *   when creating the successor
     * @param commandMap maps ping bytes to the command objects that
     *  have been sent to the mame server with the corresponding ping byte  
     * @return the game status or <code>null</code>, if the game is over
     * @throws IOException if there is a problem with the network
     * @throws ServerBusyException if the server is busy
     */
    private GameStatus getGameStatus(GameStatus prevGameStatus,
            Map<Byte, Commands> commandMap) throws IOException,
            ServerBusyException {

        // initialize
        boolean gameOver = false;
        GameStatus gameStatus = null;
        
        do {
            // receive a datagram
            m_socket.receive(m_receiveDatagram);
            
            // datagram with vector ram contents? 
            if (m_receiveDatagram.getLength() == RECEIVE_DATA_SIZE) {
                
                // extract received data
                ByteBuffer data = ByteBuffer.wrap(m_receiveDatagram.getData());
                data.order(ByteOrder.LITTLE_ENDIAN);
                
                // create game status from received data
                try {
                    gameStatus = new GameStatus(data, prevGameStatus,
                            commandMap);
                } catch (InvalidVectorRamException e) {
                    LOGGER.debug(e.getMessage());
                }
            } else {
                
                // create string from received data
                String text = new String(m_receiveDatagram.getData());
                
                // game over message received?
                if (text.startsWith(GAME_OVER_MESSAGE)) {
                    gameOver = true;
                    
                // server busy message received?
                } else if (text.startsWith(SERVER_BUSY_MESSAGE)) { 
                    String[] messageParts = text.split("\r\n");
                    throw new ServerBusyException(messageParts[0]);
                }
            }
            
        // continue, until a valid frame has been received, or the server
        // has signaled that game is over
        } while ((gameStatus == null) && (!gameOver));
        
        // return the game status
        return gameStatus;
    }
    
    /**
     * Sends a packet with the given commands to the mame server.
     * Uses the given ping byte and updates the given command map.  
     * 
     * @param commands the commands that should be send to the mame server
     * @param ping the ping byte that should be send with the commands
     * @param commandMap maps ping bytes to the command objects 
     * @throws IOException if there is a problem with the network
     */
    private void sendCommands(Commands commands, byte ping,
            Map<Byte, Commands> commandMap) throws IOException {
        // update command map
        commandMap.put(Byte.valueOf(ping), commands);
        
        // prepare datagram to send
        byte[] data = new byte[KEY_PACKET_SIGNATURE.length + 2];
        System.arraycopy(KEY_PACKET_SIGNATURE, 0, data, 0,
                KEY_PACKET_SIGNATURE.length);
        data[KEY_PACKET_SIGNATURE.length] = commands.getCommands();
        data[KEY_PACKET_SIGNATURE.length + 1] = ping;
        m_sendDatagram.setData(data);
        
        // send the datagram
        m_socket.send(m_sendDatagram);
    }

    /**
     * Sends a name packet with the given player name to the mame server.
     * 
     * @param name the name to send
     * @throws IOException if there is a problem with the network
     */
    private void sendPlayerName(String name) throws IOException {
        // prepare datagram to send
        byte[] data = new byte[NAME_PACKET_SIGNATURE.length
                + PLAYER_NAME_LENGTH];
        System.arraycopy(NAME_PACKET_SIGNATURE, 0, data, 0,
                NAME_PACKET_SIGNATURE.length);
        System.arraycopy(name.getBytes(), 0, data,
                NAME_PACKET_SIGNATURE.length, name.length());
        m_sendDatagram.setData(data);
        
        // send the datagram
        m_socket.send(m_sendDatagram);
    }
}
