/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */

package de.hfaber.asteroids.bot.sittingduck;

import de.hfaber.asteroids.game.objects.AngleByteRange;

/**
 * A look up table that pre-calculates the reachability of moves for
 * a given maximum number of frames. This is used to improve performance.
 * 
 * @see Move#isReachableFrom(int, int, int, int)
 * @author Henning Faber
 */
public class ReachabilityTable {

    /**
     * The size of the look up table.
     */
    private final int m_tableSize; 
    
    /**
     * The look up table.
     */
    private final boolean m_table[][][];
    
    /**
     * @param tableSize
     */
    public ReachabilityTable(int tableSize) {
        super();
        m_tableSize = tableSize;
        m_table = createTable(m_tableSize,
                AngleByteRange.ANGLE_BYTE_VALUE_COUNT);
    }

    /**
     * Creates a fully populated reachability table of the given extent.
     * 
     * @param maxFrames the number of frames
     * @param the number of angles
     */
    private boolean[][][] createTable(int maxFrames, int maxAngle) {
        // create table
        boolean[][][] table = new boolean[maxFrames * 2 + 1][maxAngle][maxAngle];
        
        // fill the table
        for (int frame = -maxFrames; frame < maxFrames; frame++) {
            for (int firstAngle = 0; firstAngle < maxAngle; firstAngle++) {
                for (int secondAngle = 0; secondAngle < maxAngle; secondAngle++) {
                    table[maxFrames + frame][firstAngle][secondAngle] = Move
                            .isReachableFrom(0, firstAngle, frame, secondAngle);
                }
            }
        }
        
        // return the table
        return table;
    }
    
    /**
     * Determines if a move that shoots a target at the given frame number
     * and shot angle is still a valid move after the game has
     * advanced forward to the given frame and the ship has rotated to
     * the given angle.
     * 
     * @param parentFrameNo the shot frame number of the 'move' from where
     *  the child 'move' should be tested
     * @param parentAngle the ship's shot angle of the 'move' from where
     *  the child 'move' should be tested 
     * @param childFrameNo the shot frame number of the 'move' that should be
     *  tested for reachability 
     * @param childAngle the ship's shot angle of the 'move' that should be
     *  tested for reachability
     * @return <code>true</code>, if the child 'move' is reachable from the 
     *  given parent 'move', <code>false</code> if not
     * @see Move#isReachableFrom(int, int, int, int)
     */
    public final boolean isReachableFrom(int parentFrameNo, 
            int parentAngle, int childFrameNo, int childAngle) {
        int frameDist = m_tableSize + childFrameNo - parentFrameNo;
        return m_table[frameDist][parentAngle][childAngle]; 
    }

    /**
     * Determines if a move that shoots a target at the given frame number
     * and shot angle is still a valid move after the game has
     * advanced forward to the given frame and the ship has rotated to
     * the given angle.
     * 
     * @param parentMove the move from where the child move should be 
     *  reachable
     * @param childMove the move that should be tested for reachability 
     * @return
     */
    public boolean isReachableFrom(Move parentMove, Move childMove) {
        int frameDist = m_tableSize + childMove.getShotFrameNo()
                - parentMove.getShotFrameNo();
        return m_table[frameDist][parentMove.getAngle()][childMove.getAngle()]; 
    }
}
