/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.bot.sittingduck;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.log4j.Logger;

import de.hfaber.asteroids.client.GameStatusEvent;
import de.hfaber.asteroids.client.IGameStatusListener;
import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.objects.Asteroid;
import de.hfaber.asteroids.game.objects.Bullet;
import de.hfaber.asteroids.game.objects.Explosion;
import de.hfaber.asteroids.game.objects.GameObject;
import de.hfaber.asteroids.game.objects.Saucer;
import de.hfaber.asteroids.game.objects.ScaleableGameObject;
import de.hfaber.asteroids.game.objects.Ship;
import de.hfaber.asteroids.game.state.GameStatus;

/**
 * @author Henning Faber
 */
public class MoveListManager implements Runnable, IGameStatusListener {

    /**
     * The LOGGER instance.
     */
    private static final Logger LOGGER = Logger.getLogger(
            MoveListManager.class);

    /**
     * The maximum number of frames to calculate moves ahead.
     */
    private static final int MAX_LOOK_AHEAD = 48;

    /**
     * The default size that is used to initialize a list of moves. 
     */
    private static final int DEFAULT_MOVE_LIST_SIZE = 512;

    /**
     * Number of frames a successive shot from a series of volley shots
     * is expected to impact later than the actual calculation, since the
     * calculation considered a large object, but the successive shot
     * is expected to hit a smaller debris part.  
     */
    private static final int IMPACT_FRAME_CORRECTION = 2;

    /**
     * A re-usable comparator for moves that sorts moves by impact frame 
     * number.
     */
    private static final Comparator<Move> IMPACT_FRAME_COMPARATOR =
        new Comparator<Move>() {

            /* (non-Javadoc)
             * @see java.util.Comparator#compare(T, T)
             */
            public int compare(Move o1, Move o2) {
                // sort tracked shots with lower impact frame number first
                return o1.getImpactFrameNo() - o2.getImpactFrameNo();
            }
        };

    /**
     * The last game status received from the mame server.
     */
    private GameStatus m_gameStatus;
    
    /**
     * The move table.
     */
    private final MoveTable m_moveTable;
    
    /**
     * Look up table for testing if a move is reachable from another one.
     */
    private final ReachabilityTable m_reachabilityTable;
    
    /**
     * Sorted list of all available moves. This list does not consider
     * possible collisions.
     */
    private final List<Move> m_availableMoves;
    
    /**
     * Sorted list of moves that allow to prevent all known collisions.
     */
    private final List<Move> m_collisionPreventingMoves;
    
    /**
     * List of moves that have been executed.
     */
    private final Set<TrackedShot> m_trackedShots;

    /**
     * The frame number, when the fire button was last pressed.
     */
    private int m_firePressed;

    /**
     * Number of frames that were not processed, because a new frame
     * arrived before processing started.
     */
    private int m_droppedFrames;
    
    /**
     * Creates the bot.
     */
    public MoveListManager() {
        super();
        
        // initialize move table
        m_moveTable = new MoveTable(MAX_LOOK_AHEAD);
        
        // pre-calculate reachability look up table
        m_reachabilityTable = new ReachabilityTable(MAX_LOOK_AHEAD);
        
        // initialize move lists
        m_availableMoves = new ArrayList<Move>(DEFAULT_MOVE_LIST_SIZE);
        m_collisionPreventingMoves = new ArrayList<Move>(
                DEFAULT_MOVE_LIST_SIZE);

        // initialize shot tracking
        m_trackedShots = new HashSet<TrackedShot>();
        m_firePressed = -1;
    }
    
    /* (non-Javadoc)
     * @see java.lang.Runnable#run()
     */
    public final void run() {
        GameStatus gs = null;
        GameStatus prevGs = null;
        while (true) {
            // get the next game status
            synchronized (this) {
                while (m_gameStatus == null) {
                    try {
                        wait();
                    } catch (InterruptedException e) {
                        // next game status is now available!
                    }
                }
                prevGs = gs;
                gs = m_gameStatus;
                m_gameStatus = null;
                
                // write log message, if a game status was not processed
                if (m_droppedFrames > 0) {
                    if (LOGGER.isDebugEnabled()) {
                        StringBuilder sb = new StringBuilder(60);
                        sb.append("Internal frame drop detected: ");
                        sb.append(m_droppedFrames);
                        sb.append(m_droppedFrames > 1 ? " frames" : " frame");
                        LOGGER.debug(sb.toString());
                    }            
                    m_droppedFrames = 0;
                }
            }

            // clean up tracked shots and get frame number, where next
            // shot is available
            int nextAvailableShot;
            synchronized (m_trackedShots) {
                updateTrackedShots(gs);
                validateSaucerShotLock(gs, prevGs);
                nextAvailableShot = getFramesTillNextAvailableShot(gs);
            }
            
            List<Move> availableMoves;
            List<Move> collisionPreventingMoves;
            if (gs.getShip() != null) {
                
                // calculate the possible moves for this game status
                availableMoves = determineMoveList(gs, nextAvailableShot);
                
                // calculate the collision preventing moves
                collisionPreventingMoves = preventCollisions(gs, 
                        availableMoves);
            } else {
                
                // the data in the move table is calculated for a fixed ship
                // position; if the ship is no longer present, the table must
                // be reset
                m_moveTable.reset();
                availableMoves = new ArrayList<Move>();
                collisionPreventingMoves = availableMoves;
            }
            
            synchronized (m_availableMoves) {
                // update the list of available moves 
                m_availableMoves.clear();
                m_availableMoves.addAll(availableMoves);
            
                // update the list of collision preventing moves
                m_collisionPreventingMoves.clear();
                m_collisionPreventingMoves.addAll(collisionPreventingMoves);
            }
        }
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.client.IGameStatusListener#gameStatusUpdated(de.hfaber.asteroids.asteroids.client.GameStatusEvent)
     */
    public final synchronized void gameStatusUpdated(GameStatusEvent e) {
        if (m_gameStatus != null) {
            m_droppedFrames++;
        }
        m_gameStatus = e.getGameStatus();
        notifyAll();
    }
    
    /**
     * Adds a shot to the list of tracked shots.
     * 
     * @param move the move, whose execution should be tracked
     * @param frameNo the frame number, when the shot was fired
     */
    public final void trackShot(Move move, int frameNo) {
        synchronized (m_trackedShots) {
            // check, if this is the first shot that is fired at
            // the target, or if there are already other shots
            // on the way
            boolean firstShot = true;
            for (TrackedShot trackedShot : m_trackedShots) {
                if (trackedShot.getTargetId() == move.getTargetId()) {
                    firstShot = false;
                    break;
                }
            }
            
            // If this is not the first shot at the target, consider the
            // possibility that the it will not hit any of the debris
            // left by the previous shot(s). Thus, correct the impact frame
            // to the frame when the shot will disappear due to maximum
            // bullet life time reached.
            int impactFrameCorrection = 0;
            if (firstShot) {
                impactFrameCorrection = IMPACT_FRAME_CORRECTION;
            }
            
            // create a new tracked shot for the given move
            TrackedShot newTrackedShot = new TrackedShot(move, 
                    impactFrameCorrection);

            // add the new tracked shot
            m_trackedShots.add(newTrackedShot);
            m_firePressed = frameNo;
        }
    }
    
    /**
     * Returns the list of moves that are available from the given frame
     * number and shot angle.
     * 
     * @param frameNo the frame number
     * @param angle the shot angle
     * @return the moves list of moves
     */
    public final List<Move> getMoves(int frameNo, int angle) {
        
        // determine list of moves
        List<Move> moveList;
        synchronized (m_availableMoves) {
        
            // first, get all reachable moves that would allow to prevent
            // all known collisions
            moveList = getReachableMoves(m_collisionPreventingMoves, 
                    frameNo, angle);
            
            // if no not a single collision preventing move is reachable,
            // get all reachable moves without considering collisions
            if (moveList.isEmpty()) {
                moveList = getReachableMoves(m_availableMoves, frameNo, 
                        angle);
                if (LOGGER.isDebugEnabled() && (!moveList.isEmpty())) {
                    LOGGER.debug("Unable to reach collision preventing moves!");
                }
            }
        }
        
        // return the move list
        return moveList;
    }
    
    /**
     * <p>Examines a given list of moves and returns a sub list that only
     * contains those moves that are reachable from a given frame number
     * and shot angle.</p>
     * 
     * @param fullMoveList the list of moves to process
     * @param frameNo the frame number, from where the moves must be
     *  reachable
     * @param angle the shot angle, from where the moves must be reachable
     * @return a sub list of the given list of moves that only contains
     *  those moves that are reachable from the given frame number and 
     *  shot angle
     */
    private List<Move> getReachableMoves(List<Move> fullMoveList,
            int frameNo, int angle) {
        
        // initialize the result list
        List<Move> resultList = new ArrayList<Move>(DEFAULT_MOVE_LIST_SIZE);
        
        // iterate the given list of moves and add all moves to the
        // result list, that could be executed from the given situation
        for (Move move : fullMoveList) {
            boolean isReachable = m_reachabilityTable.isReachableFrom(frameNo,
                    angle, move.getShotFrameNo(), move.getAngle());
            if (isReachable && (move.getShotFrameNo() - m_firePressed > 1)) {
                resultList.add(move);
            }
        }
        
        // return the result
        return resultList;
    }

    /**
     * <p>Updates the list of tracked shots.</p>
     * 
     * <p> NOTE: This method has to be called from inside a synchronization
     * block that synchronizes on {@link #m_trackedShots}.</p>
     * 
     * @param gs the currently processed game status
     */
    private void updateTrackedShots(GameStatus gs) {
        if (gs.isGameStarting()) {
            
            // reset shot tracking, for new games
            m_trackedShots.clear();
            m_firePressed = -1;
        } else {
            
            // create list with ids of available friendly bullets
            int untrackedBullets = 0;
            TreeSet<Integer> bulletIds = new TreeSet<Integer>();
            for (Bullet b : gs.getBullets()) {
                int id = b.getId();
                if (b.getSource() == Bullet.FRIENDLY) {
                    if (id != GameObject.UNTRACKED) {
                        bulletIds.add(id);
                    } else {
                        untrackedBullets++;                        
                    }
                }
            }
            
            // remove all tracked shots that had a bullet assigned, which 
            // is now no longer available in the game status
            removeDisappearedBullets(bulletIds);
            
            // as long as there are bullets that are not yet assigned
            // to any tracked shot, assign them to tracked shots that
            // do not have a bullet set
            assignNewBullets(gs, untrackedBullets, bulletIds);
            
            // All bullets that are at this point still in the list of
            // bullets, have appeared in the game status but do not have
            // a corresponding tracked shot. This should not happen.
            // However, create a dummy tracked shot for these bullets,
            // in order to document that the bullet is not available for
            // firing another shot
            for (Integer bulletId : bulletIds) {
                LOGGER.debug("Found bullet without a tracked shot that caused it!");
                int eolFrameNo = gs.getFrameNo()
                        + gs.getBulletById(bulletId).getTtl();
                TrackedShot dummyTrackedShot = new TrackedShot(bulletId, 
                        eolFrameNo);
                m_trackedShots.add(dummyTrackedShot);
            }
            
            // remove target ids of tracked shots that have reached their
            // impact frame
            removeImpactedTargets(gs);
        }
    }

    /**
     * <p>Cleans up the tracked shot list by removing all tracked shots that 
     * had a bullet assigned, which is now no longer available in the game 
     * status.</p>
     * 
     * <p>NOTE: This method has to be called from inside a synchronization
     * block that synchronizes on {@link #m_trackedShots}.</p>
     * 
     * @param bulletIds list of bullet ids that are available in the current
     *  game status; all ids that are already assigned to a tracked shot
     *  will be removed from this list
     */
    private void removeDisappearedBullets(TreeSet<Integer> bulletIds) {
        
        // iterate the list of tracked shots
        Iterator<TrackedShot> it = m_trackedShots.iterator();
        while (it.hasNext()) {
            
            // get the next tracked shot
            TrackedShot trackedShot = it.next();
            
            // check, if the shot has bullet assigned
            int bulletId = trackedShot.getBulletId();
            if (bulletId != GameObject.UNTRACKED) {
                
                // check, if the bullet is still in the game
                if (bulletIds.contains(bulletId)) {
                    
                    // bullet is still available -> remove it from the id list
                    bulletIds.remove(bulletId);
                } else {
                    
                    // bullet has disappeared -> remove the tracked shot
                    it.remove();
                }
            }
        }
    }

    /**
     * <p>Assigns new bullets to tracked shots that do not have a bullet
     * assigned yet.</p>   
     * 
     * <p>NOTE: This method has to be called from inside a synchronization
     * block that synchronizes on {@link #m_trackedShots}.</p>
     * 
     * @param gs the current game status
     * @param untrackedBullets the number of bullets that are available
     *  in the game status, but do not yet have an object id assigned
     * @param bulletIds list of bullet ids that are available in the current
     *  game status and are not yet assigned to any tracked shot; all ids
     *  that are successfully assigned to a tracked shot are removed from
     *  the list
     */
    private void assignNewBullets(GameStatus gs, int untrackedBullets,
            TreeSet<Integer> bulletIds) {
        
        // initialize number of remaining untracked bullets
        int remainingUntrackedBullets = untrackedBullets;
        
        // when searching for tracked shots that do not yet have a bullet
        // assigned, the tracked shots with a lower shot frame number have
        // to be considered first -> thus, sort the tracked shot list
        List<TrackedShot> sortedTrackedShotList 
            = new ArrayList<TrackedShot>(m_trackedShots);
        Collections.sort(sortedTrackedShotList);
        
        // loop through the sorted list of tracked shots 
        for (TrackedShot trackedShot : sortedTrackedShotList) {
            
            // check, if the tracked shot has no bullet assigned yet
            if (trackedShot.getBulletId() == GameObject.UNTRACKED) {
                
                // assign a bullet, if available 
                if (!bulletIds.isEmpty()) {
                    
                    // retrieve and remove the bullet from the id list
                    Integer bulletId = bulletIds.pollFirst();
                    
                    // create an updated tracked shot, that has the bullet id
                    // assigned
                    TrackedShot updatedTrackedShot = new TrackedShot(
                            trackedShot, bulletId, 0);
                    
                    // remove the old tracked shot from the list
                    // and add the updated one
                    m_trackedShots.remove(trackedShot);
                    m_trackedShots.add(updatedTrackedShot);
                    
                } else {
                    
                    // determine the number of frames that have passed,
                    // since the shot was fired
                    int elapsedFrames = gs.getFrameNo()
                            - trackedShot.getShotFrameNo();
                    
                    // A bullet should appear directly one frame after the
                    // shot was fired. If it does not, this is usually a
                    // sign that no shot was available (i.e. the maximum
                    // number of bullets were already in the game). Note
                    // however, that this may not be correct, if a _dynamic_
                    // latency correction is used!
                    if (elapsedFrames > 1 + gs.getLatencyCorrection()) {
                        
                        // if there are no untracked bullets in the game, 
                        // (which could be assigned as soon as they have 
                        // obtained an id) the tracked shot did not entail
                        // in a bullet and must thus be removed
                        if (remainingUntrackedBullets == 0) {
                            if (LOGGER.isDebugEnabled()) {
                                StringBuilder sb = new StringBuilder(80);
                                sb.append("Removing tracked shot for which ");
                                sb.append("no bullet appeared [frameNo=");
                                sb.append(gs.getFrameNo());
                                sb.append("]");
                                LOGGER.debug(sb.toString());
                            }
                            m_trackedShots.remove(trackedShot);
                        } else {
                            remainingUntrackedBullets--;
                        }
                    }
                }
            }
        }
    }

    /**
     * <p>Removes the target ids from all tracked shots, whose impact frame
     * number has been reached. Usually, the corresponding bullet will have
     * disappeared by then and the tracked shot has been removed during the
     * last update of the tracked shot list. However, if the shot should
     * have missed the target, the tracked shot is still available and the
     * target may be shot again, even while the bullet that missed it is 
     * still in the game.</p>
     * 
     * <p>NOTE: This method has to be called from inside a synchronization
     * block that synchronizes on {@link #m_trackedShots}.</p>
     * 
     * @param gs the game status
     */
    private void removeImpactedTargets(GameStatus gs) {
        // create list of updated tracked shots
        List<TrackedShot> updatedTrackedShots = new ArrayList<TrackedShot>();
        
        // Remove the target id for all tracked shots that should have
        // impacted by now, but are still available in the game status.
        // This effectively removes the shot lock for the target.
        
        // iterate the list of tracked shots
        Iterator<TrackedShot> it = m_trackedShots.iterator();
        while (it.hasNext()) {
            
            // get the next tracked shot
            TrackedShot trackedShot = it.next();
            
            // the shot has reached its target, if the impact frame number
            // is reached and the assigned target is not a fake one
            if ((gs.getFrameNo() >= trackedShot.getImpactFrameNo())
                    && (!trackedShot.hasFakeTarget())) {
                
                // if the tracked shot has a bullet assigned, it has to be
                // kept, until the bullet disappears -> thus, replace the
                // target with a fake one
                int bulletId = trackedShot.getBulletId();
                if (bulletId != GameObject.UNTRACKED) {
                    
                    // determine the expected end of lifetime of the bullet
                    int eolFrameNo = gs.getFrameNo()
                            + gs.getBulletById(bulletId).getTtl();
                    
                    // create an updated tracked shot, that still has the
                    // bullet assigned, but no longer locks the target
                    TrackedShot updatedTrackedShot = new TrackedShot(
                            bulletId, eolFrameNo);
                    updatedTrackedShots.add(updatedTrackedShot);
                }
                
                // remove the (old) tracked shot
                it.remove();
            }
        }
        
        // add the updated tracked shots to the list of tracked shots
        m_trackedShots.addAll(updatedTrackedShots);
    }

    /**
     * <p>If a saucer is present and it changes its moving direction after 
     * a shot has been fired at it, the saucer's target id is removed
     * from the corresponding tracked shot. This effectively removes the
     * shot lock that the tracked shot enforces on the saucer.</p>
     * 
     * <p>NOTE: This method has to be called from inside a synchronization
     * block that synchronizes on {@link #m_trackedShots}.</p>
     * 
     * @param gs the game status
     * @param prevGs the previous game status; may be <code>null</code>, if
     *  no previous game status is available 
     */
    private void validateSaucerShotLock(GameStatus gs, GameStatus prevGs) {
        
        // a valid previous game status is required in order to be able
        // to detect a direction change of the saucer
        if (prevGs != null) {
            
            // get the saucer objects from the previous and the current frame
            Saucer prevSaucer = prevGs.getSaucer();
            Saucer saucer = gs.getSaucer();
            
            // both frames have to have a valid saucer object, in order to
            // be able to detect a direction change
            if ((prevSaucer != null) && (saucer != null)) {
                
                // get the direction vectors
                Point prevDirection = prevSaucer.getDirection();
                Point direction = saucer.getDirection();
                
                // detect, if the saucer has changed its moving direction
                if (!prevDirection.equals(direction)) {
                    
                    // create list of updated tracked shots
                    List<TrackedShot> updatedTrackedShots 
                        = new ArrayList<TrackedShot>();
                    
                    // iterate the list of tracked shots
                    Iterator<TrackedShot> it = m_trackedShots.iterator();
                    while (it.hasNext()) {
                        
                        // get the next tracked shot
                        TrackedShot trackedShot = it.next();
                        
                        // determine, if the tracked shot was fired at
                        // the saucer
                        if (trackedShot.getTargetId() == saucer.getId()) {
                            
                            // create an updated tracked shot, that still has 
                            // the bullet assigned, but no longer locks the 
                            // saucer
                            TrackedShot updatedTrackedShot = new TrackedShot(
                                    trackedShot.getBulletId(), 
                                    trackedShot.getImpactFrameNo());
                            updatedTrackedShots.add(updatedTrackedShot);
                            
                            // remove the (old) tracked shot
                            it.remove();
                        }
                    }
                    
                    // add the updated tracked shots to the list of tracked 
                    // shots
                    m_trackedShots.addAll(updatedTrackedShots);
                }
            }
        }
    }

    /**
     * <p>Returns the number frames that have to pass, before the next shot is
     * available. A shot is available, as soon as less than the maximum number
     * of shots are emitted and the fire button is <em>not</em> pressed.</p>
     * 
     * <p>NOTE: This method has to be called from inside a synchronization
     * block that synchronizes on {@link #m_trackedShots}.</p>
     * 
     * @param gs the game status
     * @return the number of frames to wait, before the next shot is available
     */
    private int getFramesTillNextAvailableShot(GameStatus gs) {
        // start with high result value
        int framesTillNextShot = Integer.MAX_VALUE;
    
        // iterate through the list of tracked shots
        int bulletCount = 0;
        for (Move trackedShot : m_trackedShots) {
            int timeToImpact = trackedShot.getImpactFrameNo() - gs.getFrameNo();
            
            // it is possible to press the fire button one frame
            // before the actual impact frame, because the mame 
            // server will consider a received key packet not until
            // the next frame. -> thus, reduce the time to impact by 
            // one.
            timeToImpact--;
            
            // consider only shots, that have not yet reached their
            // target
            if (timeToImpact > 1) {
                bulletCount++;
                
                // remember the frame of the earliest impact
                if (timeToImpact < framesTillNextShot) {
                    framesTillNextShot = timeToImpact;
                }
            }
        }
    
        // not all available shots have been fired, ...
        if (bulletCount < Ship.MAX_BULLETS) {
            // ...thus, next shot is available, as soon as the fire
            // button is released
            if (gs.getFrameNo() > m_firePressed + 1) {
                framesTillNextShot = 0;    
            } else {
                framesTillNextShot = 1;    
            }
        }
        
        return framesTillNextShot;
    }

    /**
     * Calculates the moves that are available from the given game status.
     * Updates the move table to reflect changes that have happened between
     * the current and the previous game status.
     * 
     * @param gs the game status, for which the new move list should be
     *  calculated
     * @param nextAvailableShot the frame number, when the next shot will
     *  be available
     * @return list of moves
     */
    private List<Move> determineMoveList(GameStatus gs, int nextAvailableShot) {
        
        // update move table
        m_moveTable.update(gs, nextAvailableShot);
        
        // get target list
        List<ScaleableGameObject> targetList = gs.getTargetList();
        
        // initialize move list
        List<Move> moves = new ArrayList<Move>(DEFAULT_MOVE_LIST_SIZE);
        
        // get shot count map
        Map<Integer, ShotCount> shotCountMap = getShotCountMap();
        
        for (int frameNo = nextAvailableShot; frameNo 
                < m_moveTable.getTableLength(); frameNo++) {
            
            int absoluteFrameNo = gs.getFrameNo() + frameNo;
            for (int angle = -frameNo; angle <= frameNo; angle++) {
                TreeSet<Move> queue = m_moveTable.getMoveQueue(gs, frameNo, 
                        angle);

                Move move = null;
                if (!queue.isEmpty()) {
                    move = queue.first();
                }
                while ((move != null) 
                        && ((move.getShotFrameNo() != absoluteFrameNo)
                        || isMoveLocked(shotCountMap, targetList.size(), move))) {

                    if (move.getShotFrameNo() != absoluteFrameNo) {
                        if (LOGGER.isInfoEnabled()) {
                            StringBuilder sb = new StringBuilder(100);
                            sb.append("Invalid frame in move table! [frameNo=");
                            sb.append(gs.getFrameNo());
                            sb.append(" relativeFrameNo=");
                            sb.append(frameNo);
                            sb.append(" expectedFrameNo=");
                            sb.append(gs.getFrameNo() + frameNo);
                            sb.append(" actualFrameNo=");
                            sb.append(move.getShotFrameNo());
                            sb.append("]");
                            LOGGER.info(sb.toString());
                        }
                        queue.remove(move);
                    }
                    move = queue.higher(move);
                }
                
                if (move != null) {
                    moves.add(move);
                }
            }
        }
        
        // take care not to fire at anything else but small objects,
        // if the maximum number of possible objects would be exceeded
//        moves = considerTwentySixObjectLimit(gs, moves);

        // sort the move list
        sortMoveList(gs, moves);
        
        // return the move list
        return moves;
    }
    
    /**
     * Determines the number of shots that have been fired at each target
     * and returns it as map that maps each non zero shot count to its
     * target id.  
     * 
     * @return the shot count map
     */
    private Map<Integer, ShotCount> getShotCountMap() {
        
        // create the shot map
        Map<Integer, ShotCount> shotCountMap 
            = new HashMap<Integer, ShotCount>();
        
        // fill the map
        synchronized (m_trackedShots) {
            
            // iterate all tracked shots
            for (TrackedShot trackedShot : m_trackedShots) {
                
                // get the id of the target that this shot aims at
                int targetId = trackedShot.getTargetId();
                
                // get the shot count object for this target
                ShotCount shotCount = shotCountMap.get(targetId);
                
                if (shotCount == null) {
                    
                    // create and add a new shot count object, if not yet 
                    // available
                    shotCount = new ShotCount(1, trackedShot.getShotFrameNo());
                    shotCountMap.put(targetId, shotCount);
                } else {
                    
                    // update existing shot count object
                    shotCount.m_shotCount++;
                    int shotFrameNo = trackedShot.getShotFrameNo();
                    if (shotFrameNo < shotCount.m_firstShotFrameNo) {
                        shotCount.m_firstShotFrameNo = shotFrameNo;
                    }
                }
            }
        }
        
        // return the shot map
        return shotCountMap;
    }
    
    /**
     * Checks if the given move may be executed or if it is locked because
     * the focused target has already been shot at.
     * 
     * @param shotCountMap 
     * @param targetCount the number of remaining targets
     * @param move the move to check
     * @return <code>true</code>, if the move is locked, <code>false</code>,
     *  if not
     */
    private boolean isMoveLocked(Map<Integer, ShotCount> shotCountMap,
            int targetCount, Move move) {
        
        boolean isLocked = false;
        ShotCount shotCount = shotCountMap.get(move.getTargetId());
        if ((shotCount != null) && (shotCount.m_shotCount > 0)) {
            switch (move.getTargetSize()) {
                case Asteroid.LARGE_ASTEROID_SQUARE_RADIUS:
                    isLocked = (shotCount.m_shotCount > 3)
                        || (move.getShotFrameNo() 
                                - shotCount.m_firstShotFrameNo > 6)
                        || (move.getShotDistance() 
                                > Asteroid.MIDDLE_ASTEROID_SQUARE_RADIUS);
                    break;
                case Asteroid.MIDDLE_ASTEROID_SQUARE_RADIUS:
                    isLocked = (shotCount.m_shotCount > 1)
                        || (move.getShotFrameNo() 
                                - shotCount.m_firstShotFrameNo > 2)
                        || (move.getShotDistance() 
                                > Asteroid.SMALL_ASTEROID_SQUARE_RADIUS);
                    break;
                default:
                    isLocked = true;
                    break;
            }
            
            // allow two shots, if only a single target remains 
            if ((targetCount == 1) && (shotCount.m_shotCount < 2)) {
                isLocked = false;
            }
        }
        
        // return the result
        return isLocked;
    }

    /**
     * Examines the given list of moves to determine, which moves would
     * cause an exceedance of the 26 object limit. Returns a modified
     * list of moves, where all invalid moves have been removed unless
     * they are necessary to avoid a collision.
     * 
     * @param gs the game status
     * @param moves the move list to examine
     * @return the modified list of moves
     */
    private List<Move> considerTwentySixObjectLimit(GameStatus gs, 
            List<Move> moves) {

        // Collision targets may be shot, even if the 26 object limit is
        // exceeded. However, only consider collisions for which all moves 
        // from the current frame to the collision frame are available.
        int lastFrameWithKnownMoves = gs.getFrameNo()
                + m_moveTable.getTableLength();
        
        // calculate the set of frame numbers, during which the 
        // number of objects will reach the limit
        Set<Integer> limitExceedingFrames = calculateLimitExceedingFrames(gs);
        
        // initialize the result list
        List<Move> reasonableMoves = new ArrayList<Move>(
                DEFAULT_MOVE_LIST_SIZE);

        // iterate the list of moves
        for (Move move : moves) {
            
            // add the move to the result list, if..
            //  * it aims at a small asteroid
            //  * it may prevent a collision
            //  * it will not exceed the 26 object limit
            int collisionFrameNo = move.getCollisionFrameNo();
            if ((move.getTargetSize() == Asteroid.SMALL_ASTEROID_SQUARE_RADIUS) 
                    || ((collisionFrameNo != Move.NO_COLLISION)
                            && (collisionFrameNo < lastFrameWithKnownMoves))
                    || (!limitExceedingFrames.contains(
                            move.getImpactFrameNo()))) {
                reasonableMoves.add(move);
            }
        }
        
        // return the updated move list
        return reasonableMoves;
    }

    /**
     * Returns a set of frame numbers, during which the number of 
     * objects will reach the technically possible limit of 
     * {@link GameStatus#MAX_ASTEROIDS}. The calculation considers
     * the given game status and the contents of the shots that have
     * been fired so far.
     * 
     * @param gs the game status
     * @return the set of limit exceeding frames
     */
    private Set<Integer> calculateLimitExceedingFrames(GameStatus gs) {
        // initialize the result set
        Set<Integer> limitExceedingFrames = new HashSet<Integer>(128);

        // get event lists
        List<Integer> explosionFrames = createExplosionVanishingList(gs);
        List<TrackedShot> sortedTrackedShots = getSortedTrackedShots();
        
        // initialize the object counter with the number of currently
        // present asteroids and explosions
        int objectCount = gs.getAsteroidCount() + gs.getExplosionCount();
        int framePtr = gs.getFrameNo();

        // iterate the two event lists, until there are no events left
        int explosionPtr = 0;
        int trackedShotPtr = 0;
        while ((explosionPtr < explosionFrames.size())
            || (trackedShotPtr < sortedTrackedShots.size())) {
            
            // get the next explosion disappearance frame, if any left
            Integer lastExplosionFrame = null;
            if (explosionPtr < explosionFrames.size()) {
                lastExplosionFrame = explosionFrames.get(explosionPtr);
            }
            
            // get the next tracked shot, if any left
            TrackedShot trackedShot = null;
            if (trackedShotPtr < sortedTrackedShots.size()) {
                trackedShot = sortedTrackedShots.get(trackedShotPtr);
            }

            // determine the next event
            if ((lastExplosionFrame != null) && ((trackedShot == null) 
                    || (lastExplosionFrame <= trackedShot.getImpactFrameNo()))) {
                
                // -> next event is a disappearing explosion
                
                // update result set, if the limit was reached during the
                // preceding block of frames
                if (objectCount >= GameStatus.MAX_ASTEROIDS - 1) {
                    addLimitExceedingFrames(limitExceedingFrames, framePtr,
                            lastExplosionFrame);
                }
                framePtr = lastExplosionFrame;
                
                // the explosion is gone -> reduce object counter by one
                objectCount--;
                
                // the disappearing explosion has been processed -> move on 
                // to the next one
                explosionPtr++;
            } else {
                
                // -> next event is an impacting shot
                framePtr = trackedShot.getImpactFrameNo();
                
                // the target is destroyed and leaves an explosion behind
                // -> object count remains the same, because the target
                //    is replaced by the explosion
                explosionFrames.add(trackedShot.getImpactFrameNo()
                        + Explosion.EXPLOSION_LIFE_TIME);
                
                // depending on its size, the target leaves debris behind
                int targetSize = trackedShot.getTargetSize();
                if ((targetSize == Asteroid.MIDDLE_ASTEROID_SQUARE_RADIUS)
                        || (targetSize == Asteroid.LARGE_ASTEROID_SQUARE_RADIUS)) {
                    objectCount = objectCount + 2;
                    if (objectCount > GameStatus.MAX_ASTEROIDS) {
                        objectCount = GameStatus.MAX_ASTEROIDS;
                    }
                }
                
                // update result set, if the limit is reached
                if (objectCount == GameStatus.MAX_ASTEROIDS) {
                    addLimitExceedingFrames(limitExceedingFrames, 
                            gs.getFrameNo(), trackedShot.getImpactFrameNo());
                } else if (objectCount == GameStatus.MAX_ASTEROIDS - 1) {
                    addLimitExceedingFrames(limitExceedingFrames, 
                            trackedShot.getImpactFrameNo() 
                                - Explosion.EXPLOSION_LIFE_TIME, 
                            trackedShot.getImpactFrameNo());
                }

                // the impacting shot has been processed -> move on to the
                // next tracked shot
                trackedShotPtr++;
            }
        }

        //  return the result
        return limitExceedingFrames;
    }

    /**
     * Creates a sorted list of the frame numbers, when the explosions 
     * from the given game status will have disappeared.
     * 
     * @param gs the game status
     * @return the sorted list of frame number
     */
    private List<Integer> createExplosionVanishingList(GameStatus gs) {
        
        // initialize the result list
        List<Integer> explosionFrames = new ArrayList<Integer>();
        
        // for each explosion, add the number of the first frame
        // when it will have disappeared
        for (Explosion explosion : gs.getExplosions()) {
            int lastExplosionFrame = gs.getFrameNo() + explosion.getTtl();
            explosionFrames.add(lastExplosionFrame);
        }
        
        // sort the result list
        Collections.sort(explosionFrames);
        
        // return the result
        return explosionFrames;
    }

    /**
     * Creates a sorted list of tracked shots that has elements with
     * a low impact frame at the beginning and only contains elements 
     * with a valid target.
     *  
     * @return the sorted list of tracked shots
     */
    private List<TrackedShot> getSortedTrackedShots() {
        
        // initialize the result list
        List<TrackedShot> sortedTrackedShots = new ArrayList<TrackedShot>();
        
        // add all tracked shots with a valid target
        synchronized (m_trackedShots) {
            for (TrackedShot trackedShot : m_trackedShots) {
                if (!trackedShot.hasFakeTarget()) {
                    sortedTrackedShots.add(trackedShot);
                }
            }
        }
        
        // sort the result list
        Collections.sort(sortedTrackedShots, IMPACT_FRAME_COMPARATOR);
        
        // return the result
        return sortedTrackedShots;
    }

    /**
     * Updates the given set by adding all frame numbers to the set, which
     * are part of the given interval.
     * 
     * @param limitExceedingFrames the set to update
     * @param fromFrame the lower boundary of the interval; the lower boundary
     *  <em>is</em> part of the interval
     * @param toFrame the upper boundary of the interval; the boundary itself
     *  is <em>not</em> part of the interval
     */
    private void addLimitExceedingFrames(Set<Integer> limitExceedingFrames, 
            int fromFrame, int toFrame) {
        for (int frame = fromFrame; frame < toFrame; frame++) {
            limitExceedingFrames.add(frame);
        }
    }

    /**
     * Checks for each move in the given list of moves, if all imminent
     * collisions can still be prevented if the move would be executed.
     * 
     * @param gs the game status
     * @param moves the list of moves
     * @return a new list that only contains those moves from the given
     *  move list that have been found to be save 
     */
    private List<Move> preventCollisions(GameStatus gs, List<Move> moves) {
        
        // create list with the ids of the targets that have already
        // been fired at
        int nextAvailableShot = 0;
        Set<Integer> targetedIds = new HashSet<Integer>();
        synchronized (m_trackedShots) {
            List<TrackedShot> sortedTrackedShots = new ArrayList<TrackedShot>(
                    m_trackedShots);
            Collections.sort(sortedTrackedShots, IMPACT_FRAME_COMPARATOR);

            // determine frame, when next shot is available,
            // while filling the list of target ids
            for (int i = 0; i < sortedTrackedShots.size(); i++) {
                TrackedShot trackedShot = sortedTrackedShots.get(i);
                if (sortedTrackedShots.size() - i == Ship.MAX_BULLETS - 1) {
                    nextAvailableShot = trackedShot.getImpactFrameNo(); 
                }
                
                targetedIds.add(trackedShot.getTargetId());
            }
        }
        
        // initialize list that will hold the save moves
        List<Move> safeMoves = new ArrayList<Move>(DEFAULT_MOVE_LIST_SIZE); 

        // Get the list of collision lists. Each collision list contains
        // all possible moves to prevent the collision of a specific
        // target
        Collection<List<Move>> collisionLists = getCollisionLists(gs, moves);
        
        // check each move in the given move list, if it leaves at least
        // one collision preventing move for each imminent collision
        for (Move currentMove : moves) {

            // check, if move is save; add it to the move list, if so 
            if (isMoveSafe(nextAvailableShot, targetedIds, collisionLists, 
                    currentMove)) {
                safeMoves.add(currentMove);
            }
        }
    
        // if all moves are unsafe, keep the original list, in order to make
        // some score before being hit or having to jump through hyperspace
        if (!safeMoves.isEmpty()) {
            sortMoveList(gs, safeMoves);
            return safeMoves;
        } else {
            if (LOGGER.isDebugEnabled() && (!moves.isEmpty())) {
                LOGGER.debug("Unable to prevent all imminent collisions.");
            }
            return moves;
        }
    }

    /**
     * @param nextAvailableShot the frame number, when the next shot
     *  will be available
     * @param targetedIds set with the ids of all targets for which a
     *  shot has already been fired
     * @param collisionLists the list of collision lists that each contain
     *  the moves that may prevent the collision with a specific target 
     * @param move the move to check for safety
     * @return <code>true</code>, if the move is safe, <code>false</code>,
     *  if not
     */
    private boolean isMoveSafe(int nextAvailableShot, Set<Integer> targetedIds,
            Collection<List<Move>> collisionLists, Move move) {
        // determine impact frame no for the given move
        int currentMoveImpactFrameNo;
        if (targetedIds.contains(move.getTargetId())) {
            currentMoveImpactFrameNo = move.getShotFrameNo()
                    + Bullet.BULLET_LIFE_TIME;
        } else {
            currentMoveImpactFrameNo = move.getImpactFrameNo();
        }
        
        // check all objects that are on collision course with the ship
        boolean allCollisionsPreventable = true;
        for (List<Move> collisionList : collisionLists) {
            
            // if the current object has already been fired at, 
            // consider the collision as prevented 
            if (!targetedIds.contains(collisionList.get(0).getTargetId())) {
                
                // try to find at least one move that could prevent the
                // collision with the current object
                // NOTE: iterate backwards through the list, because
                //       moves with higher frame numbers are at the
                //       end of the list and have a higher probability
                //       to be reachable
                boolean isPreventable = false;
                for (int i = collisionList.size() - 1; i >= 0; i--) {
                    // get the next move that could prevent the collision
                    Move preventionMove = collisionList.get(i);
                    
                    // determine,  if this move would still be available
                    boolean isReachable = m_reachabilityTable
                            .isReachableFrom(move, preventionMove);
                    
                    // determine, if a shot would be available
                    boolean isShotAvailable 
                        = (preventionMove.getShotFrameNo() 
                                - move.getShotFrameNo() > 1)
                            && ((preventionMove.getShotFrameNo() 
                                    >= nextAvailableShot) 
                                || (currentMoveImpactFrameNo 
                                    < preventionMove.getShotFrameNo()));
                    
                    if (isReachable && isShotAvailable) {
                        isPreventable = true;
                        break;
                    }
                }
                
                // if the collision with the current object cannot be
                // prevent, the current move is not save
                if (!isPreventable) {
                    allCollisionsPreventable = false;
                    break;
                }
            }
        }
        
        // return the result
        return allCollisionsPreventable;
    }

    /**
     * Creates a list of lists that each contain all moves from the
     * given move list that may prevent the collision with a specific
     * target.
     * 
     * @param gs the game status
     * @param moves the list of moves, for which the collision lists
     *  should be determined
     * @return the list of collision lists
     */
    private Collection<List<Move>> getCollisionLists(GameStatus gs,
            List<Move> moves) {
        // create a map the id of a collision target to a list of
        // moves that may prevent this collision
        Map<Integer, List<Move>> collisionMap 
            = new HashMap<Integer, List<Move>>();
        
        // Only consider collisions, for which all moves from the current
        // frame to the collision frame are available.
        int lastFrameWithKnownMoves = gs.getFrameNo()
                + m_moveTable.getTableLength();
        
        // iterate all moves
        for (Move move : moves) {
            
            // determine, if this moves prevents a collision
            int collisionFrameNo = move.getCollisionFrameNo();
            if ((collisionFrameNo != Move.NO_COLLISION)
                    && (collisionFrameNo < lastFrameWithKnownMoves)) {
                
                // add the move to the target's collision list and 
                // lazily create the list, if it does not yet exist
                int targetId = move.getTargetId();
                List<Move> collisionList = collisionMap.get(targetId);
                if (collisionList == null) {
                    collisionList = new ArrayList<Move>();
                    collisionMap.put(targetId, collisionList);
                }
                collisionList.add(move);
            }
        }
        
        // return the result
        Collection<List<Move>> collisionLists = collisionMap.values();
        return collisionLists;
    }

    /**
     * Sorts the given list of moves. Uses a different comparator depending
     * on the number of remaining targets. By default, the natural comparator
     * is used. If however, only a single target is left, this target should
     * be destroyed as soon as possible and the list is thus sorted by
     * impact frame. 
     *
     * @param gs the game status for which the given list of moves
     *  has been calculated
     * @param moves the list of moves to sort
     */
    private void sortMoveList(GameStatus gs, List<Move> moves) {
        
        // create list with the ids of the targets that have already
        // been fired at
        Set<Integer> targetedIds = new HashSet<Integer>();
        synchronized (m_trackedShots) {
            for (TrackedShot trackedShot : m_trackedShots) {
                targetedIds.add(trackedShot.getTargetId());
            }
        }
        
        // determine the number of remaining targets
        Set<Integer> remainingTargetIds = new HashSet<Integer>();
        List<ScaleableGameObject> targetList = gs.getTargetList();
        for (ScaleableGameObject target : targetList) {
            int targetId = target.getId();
            if (!targetedIds.contains(targetId)) {
                remainingTargetIds.add(targetId);
            }
        }
        
        if (remainingTargetIds.size() <= 1) {
            
            // if there is only one target left, sort move list by impact
            // frame instead of the usual shot frame
            Collections.sort(moves, IMPACT_FRAME_COMPARATOR);
        } else {
            Collections.sort(moves);
        }
    }

    /**
     * Wrapper class used as value objects for the shot map, which stores 
     * the number of shots and the first shot frame for a target.
     * 
     * @author Henning Faber
     */
    private static class ShotCount {
        
        /**
         * The number of shots fired.
         */
        private int m_shotCount;
        
        /**
         * The frame number when the first shot was fired.
         */
        private int m_firstShotFrameNo;

        /**
         * @param shotCount
         * @param firstShotFrame
         */
        public ShotCount(int shotCount, int firstShotFrame) {
            super();
            m_shotCount = shotCount;
            m_firstShotFrameNo = firstShotFrame;
        }

        /**
         * @return the shotCount
         */
        public final int getShotCount() {
            return m_shotCount;
        }

        /**
         * @param shotCount the shotCount to set
         */
        public final void setShotCount(int shotCount) {
            m_shotCount = shotCount;
        }

        /**
         * @return the firstShotFrame
         */
        public final int getFirstShotFrame() {
            return m_firstShotFrameNo;
        }

        /**
         * @param firstShotFrame the firstShotFrame to set
         */
        public final void setFirstShotFrame(int firstShotFrame) {
            m_firstShotFrameNo = firstShotFrame;
        }
    }
}
