/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.bot.simplebot;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import de.hfaber.asteroids.bot.IAsteroidsBot;
import de.hfaber.asteroids.client.Commands;
import de.hfaber.asteroids.client.GameStatusEvent;
import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.objects.Asteroid;
import de.hfaber.asteroids.game.objects.GameObject;
import de.hfaber.asteroids.game.objects.Saucer;
import de.hfaber.asteroids.game.objects.ScaleableGameObject;
import de.hfaber.asteroids.game.objects.Ship;
import de.hfaber.asteroids.game.objects.Bullet;
import de.hfaber.asteroids.game.state.GameStatus;

/**
 * <p>A simple bot that uses the following strategy:</p>
 * <ul>
 *  <li>Assigns a score for each available target.
 *  <li>Picks the target that has the best score.
 *  <li>Calculates the optimal shot direction for that target.
 *  <li>Fires, if the target can be hit from the current shot angle of the ship.
 *  <li>Rotates the ship into the direction of the optimal shot vector.
 * </ul>
 * 
 * @author Henning Faber
 */
public class SimpleBot implements IAsteroidsBot {
    
    /**
     * Maximum number of frames, a target stays in the target map. 
     */
    private static final int MAX_TARGET_LIFETIME = 90;

    /**
     * The score factor that is applied for objects that have
     * are on collision course with the ship. 
     */
    private static final int COLLISION_SCORE_FACTOR = 10000;

    /**
     * The score factor that is applied for saucer objects 
     */
    private static final double SAUCER_SCORE_FACTOR = 75;
    
    /**
     * The score factor that is applied for objects that are
     * near the ship. 
     */
    private static final double LOCATION_SCORE_FACTOR = 25;

    /**
     * The score factor that is applied for objects that can be
     * hit without having to rotate much. 
     */
    private static final double ROTATION_SCORE_FACTOR = 100;

    /**
     * The score factor that is applied for objects that have already
     * been fired at in order to avoid firing unnecessary shots.
     */
    private static final double MULTIPLE_SHOT_SCORE_FACTOR = 200;

    /**
     * The score factor that is applied for objects that are out of
     * firing range. 
     */
    private static final double DISTANCE_SCORE_FACTOR = 200;
    
    /**
     * The lastest game status. 
     */
    private GameStatus m_gameStatus;
    
    /**
     * Maps frame numbers to object ids. Whenever fire button is pressed,
     * an entry with the current frame number and the indended target
     * is added. 
     */
    private Map<Integer, Integer> m_targetMap;
    
    /**
     * Frame counter used as key when adding entries to the target map. 
     */
    private int m_frameCounter;
    
    /**
     * Flag that indicates, if fire button was pressed in the last frame.
     */
    private boolean m_firePressed = false;
    
    /**
     * 
     */
    public SimpleBot() {
        super();
        m_targetMap = new HashMap<Integer, Integer>();
        m_frameCounter =  0;
        m_firePressed = false;
    }
    
    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.client.IGameStatusListener#gameStatusUpdated(de.hfaber.asteroids.asteroids.client.GameStatusEvent)
     */
    public final void gameStatusUpdated(GameStatusEvent e) {
        m_gameStatus = e.getGameStatus();
        m_frameCounter++;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.bot.IAsteroidsBot#compute(de.hfaber.asteroids.asteroids.client.Commands)
     */
    public final Commands compute() {
        cleanUpTargetMap();

        // create new command object
        Commands commands = new Commands();

        // do not press buttons, if ship is not present
        Ship ship = m_gameStatus.getShip();
        if (ship != null) {

            // find the best target and fire, if shot will hit
            GameObject bestTarget = findBestTarget(commands);

            // 
            if (bestTarget != null) {
                rotateShip(bestTarget, commands);
            }

            // use hyperspace in case of inevitable collisions
            considerHyperSpaceEscape(commands);
        }
            
        // remember fire pressed status
        m_firePressed = commands.isFirePressed();

        // return the commands
        return commands;
    }

    /**
     * Removes outdated targets from the target map.
     */
    private void cleanUpTargetMap() {
        Set<Integer> frameNumbers = m_targetMap.keySet();
        Iterator<Integer> it = frameNumbers.iterator();
        while (it.hasNext()) {
            Integer frameNumber = it.next();
            if (frameNumber + MAX_TARGET_LIFETIME < m_frameCounter) {
                it.remove();
            }
        }
    }

    /**
     * Evaluates all available targets and returns the target with the
     * best score. If a direct hit on a target is possible, the fire
     * button on the given command object is pressed.
     * 
     * @param commands the commands object, where the fire button may
     *  be pressed
     * @return the best target or <code>null</code>, if no target is 
     *  available
     */
    private GameObject findBestTarget(Commands commands) {
        // initialize search for best target
        double bestScore = Integer.MIN_VALUE;
        GameObject bestTarget = null;
    
        // create list of game objects to evaluate
        List<ScaleableGameObject> objectList = m_gameStatus.getTargetList();
        
        // evaluate each game object
        Ship ship = m_gameStatus.getShip();
        Point currentShotDirection = ship.getShotDirection();
        for (GameObject o : objectList) {
            
            // check, if direct hit is possible
            MinDist dist = DistCalculator.calculateMinimumDistance(
                    ship.getLocation(), currentShotDirection, o.getLocation(), 
                    o.getDirection());
            if (dist.getMinDist() < o.getSquareRadius()
                    && (dist.getT() > 0) 
                    && (dist.getT() < Bullet.BULLET_LIFE_TIME)
                    && (!m_firePressed)
                    && (moreShotsAllowed(o) || (objectList.size() == 1))) {
                
                // direct hit is possible!
                // -> press fire key
                commands.pressFire(true);
                m_targetMap.put(m_frameCounter, o.getId());

                // allow this target as best target, if no other targets
                // are available; thus, do not set the best score variable,
                // so any other target will be prefered 
                if (bestTarget == null) {
                    bestTarget = o;
                }
            } else {                
    
                // direct hit is not possible
                // -> consider object as potential target
                double score = getScore(o);
                if (score > bestScore) {
                    bestScore = score;
                    bestTarget = o;
                }
            }
        }
        
        // return the best target found
        return bestTarget;
    }

    /**
     * Rotates the ship into the direction of the optimal shot vector
     * on the given target. 
     * 
     * @param target the target, to whose optimal shot vector the ship 
     *  should be rotated
     * @param commands the command object, where the left/right buttons
     *  may be pressed
     */
    private void rotateShip(GameObject target, Commands commands) {
        // determine optimal shot direction
        Ship ship = m_gameStatus.getShip();
        Point optimalShotDirection = aim(target, ship.getLocation(), 
                ship.getBulletSpeed());
        
        if (optimalShotDirection != null) {
            // rotate ship into optimal shot direction
            int direction = optimalShotDirection.surfaceArea(
                    ship.getShotDirection());
            if (direction > 0) {
                commands.pressRight(true);
            } else {
                commands.pressLeft(true);
            }
        } 
    }

    /**
     * Checks the last received game status for inevitable collisions
     * and presses the hyperspace button on the given command object, 
     * if necessary.
     * 
     * @param commands the command object, where the hyperspace button
     *  may be pressed
     */
    private void considerHyperSpaceEscape(Commands commands) {

        // create list of objects to consider
        List<GameObject> objectList = m_gameStatus.getObstacleList();
        
        // check all objects on the list for inevitable collisions
        Ship ship = m_gameStatus.getShip();
        for (GameObject o : objectList) {
            Point projectedObjectLocation = o.project(1);
            Point projectedShipLocation = ship.project(1);
            int dist = projectedObjectLocation.distance(projectedShipLocation);
            if (dist < o.getSquareRadius() + ship.getSquareRadius()) {
                commands.pressHyperspace(true);
                break;
            }   
        }
    }

    /**
     * Calculates the overall score for the given game object.
     * 
     * @param o the game object, for which the score should be calculated
     * @return the score
     */
    private double getScore(GameObject o) {
        // initialize score value
        double score = 0;

        // add up scores of the different criterias
        score += getCollisionCriteriaScore(o);
        score += getSaucerCriteriaScore(o);
        score += getLocationCriteriaScore(o);
        score += getRotationCriteriaScore(o);
        score += getMultipleShotCriteriaScore(o);
        score += getDistanceCriteriaScore(o);
        
        // return the score value
        return score;
    }
    
    
    /**
     * Calculates the score for the <em>collision criteria</em> for 
     * the given game object. The <em>collision criteria</em> assigns
     * scores for objects that are on collision course with the ship.
     * 
     * @param o the game object, for which the score should be calculated
     * @return the score
     */
    private double getCollisionCriteriaScore(GameObject o) {
        // initialize score value for this criteria
        double score = 0.0;

        // check, if object is on collision course with ship
        Ship ship = m_gameStatus.getShip();
        MinDist dist = DistCalculator.calculateMinimumDistance(ship, o);
        if ((dist.getMinDist() < o.getSquareRadius() + ship.getSquareRadius())
                && (dist.getT() > 0)) {
            
            // calculate factor between zero and one: 
            // an early impact induces a higher score
            double factor = (Bullet.BULLET_LIFE_TIME - dist.getT())
                    / Bullet.BULLET_LIFE_TIME;
            
            // if object is out of shot range, handle it as if it would 
            // not be on collision course
            if (factor < 0) {
                factor = 0;
            }
            
            // calculate score according to scorefactor
            score = factor * COLLISION_SCORE_FACTOR;
        }

        // retrun the score
        return score;
    }

    /**
     * Calculates the score for the <em>saucer criteria</em> for the 
     * given game object. The <em>saucer criteria</em> grants an additional
     * score to saucers, since they are potentially dangerous.
     * 
     * @param o the game object, for which the score should be calculated
     * @return the score
     */
    private double getSaucerCriteriaScore(GameObject o) {
        // initialize score value for this criteria
        double score = 0.0;
    
        // check, if object is a saucer
        if (o instanceof Saucer) {
            
            // calculate distance between saucer and ship
            Ship ship = m_gameStatus.getShip();
            int dist = o.getLocation().distance(ship.getLocation());
            
            // calculate factor between zero and one:
            // a close saucer induces a higher score
            final int maxSquareDistance = 1000 * 1000;
            double factor = (maxSquareDistance - dist) / maxSquareDistance; 
            
            // if saucer is to far away to be dangerous, handle it as
            // if it were a usual game object
            if (factor < 0) {
                factor = 0;
            }
            
            // calculate score according to scorefactor
            score = factor * SAUCER_SCORE_FACTOR;
        }
        
        // retrun the score
        return score;
    }

    /**
     * Calculates the score for the <em>location criteria</em> for the
     * given game object. The <em>location criteria</em> assigns scores
     * for objects that are close to the ship.
     * 
     * @param o the game object, for which the score should be calculated
     * @return the score
     */
    private double getLocationCriteriaScore(GameObject o) {
        // determine the distance between the game object and the ship
        Ship ship = m_gameStatus.getShip();
        double dist = o.getLocation().distance(ship.getLocation());

        // calculate factor between zero and one:
        // a close target induces a higher score
        final int maxSquareDistance = 500 * 500;
        double factor = (maxSquareDistance - dist) / maxSquareDistance; 

        // do not apply negative scores for objects that are very far away
        if (factor < 0) {
            factor = 0;
        }
        
        // calculate score according to scorefactor
        double score = factor * LOCATION_SCORE_FACTOR;
        
        // retrun the score
        return score;
    }
    
    /**
     * Calculates the score for the <em>rotation criteria</em> for the
     * given game object. The <em>rotation criteria</em> assigns scores
     * for objects that can be hit without rotating the ship too much.
     * 
     * @param o the game object, for which the score should be calculated
     * @return the score
     */
    private double getRotationCriteriaScore(GameObject o) {
        // determine angle between current and optimal shot direction
        Ship ship = m_gameStatus.getShip();
        Point currentShotDirection = ship.getShotDirection();
        Point optimalShotDirection = aim(o, ship.getLocation(),
                ship.getBulletSpeed());
        double angle = currentShotDirection.angle(optimalShotDirection);

        // calculate factor between zero and one:
        // a small angle induces a higher score
        final double maxAngle = Math.PI / 2;
        double factor = (maxAngle - angle) / maxAngle; 

        // do not apply negative scores for objects that need a very
        // long rotation
        if (factor < 0) {
            factor = 0;
        }
        // calculate score according to scorefactor
        double score = factor * ROTATION_SCORE_FACTOR;
        
        // retrun the score
        return score;
    }
    
    /**
     * Calculates the score for the <em>multiple shot criteria</em> for 
     * the given game object. The <em>multiple shot criteria</em> assigns 
     * negative scores (penalties) for objects that have already been
     * fired at.
     * 
     * @param o the game object, for which the score should be calculated
     * @return the score
     */
    private double getMultipleShotCriteriaScore(GameObject o) {
        if (!moreShotsAllowed(o)) {
            return -1 * MULTIPLE_SHOT_SCORE_FACTOR;
        } else {
            return 0;
        }
    }
    
    /**
     * Calculates the score for the <em>distance criteria</em> for the
     * given game object. The <em>distance criteria</em> assigns negative 
     * scores (penalties) for objects that are out of firing range.
     * 
     * @param o the game object, for which the score should be calculated
     * @return the score
     */
    private double getDistanceCriteriaScore(GameObject o) {
        // initialize score value for this criteria
        double score = 0.0;

        // determine, if object would be out of shot range, assumed that
        // the shot would be fired along the optimal shot vector
        Ship ship = m_gameStatus.getShip();
        Point optimalShotDirection = aim(o, ship.getLocation(),
                ship.getBulletSpeed());
        MinDist dist = DistCalculator.calculateMinimumDistance(ship.getLocation(),
                optimalShotDirection, o.getLocation(), o.getDirection());

        // calculate score according to scorefactor
        if (dist.getT() > Bullet.BULLET_LIFE_TIME) {
            score = -1 * DISTANCE_SCORE_FACTOR;
        }

        // retrun the score
        return score;
    }

    /**
     * Determines, if the maximum number of shots for a given target
     * has been reached. 
     * 
     * @param o the game object, for which the information should be
     *  determined
     * @return <code>true</code>, if the game object has not been fired at
     *  more than the allowed number of times within the last 
     *  {@link #MAX_TARGET_LIFETIME} frames, <code>false</code>, if no
     *  more shots allowed for now
     */
    private boolean moreShotsAllowed(GameObject o) {
        // get the number of times this object has been fired at
        // during the last MAX_TAREGET_LIFETIME frames
        Collection<Integer> firedAtObjects = m_targetMap.values();
        int firedAtCount = Collections.frequency(firedAtObjects, o.getId());
        
        if (firedAtCount > 0) {
            // default is to fire just one shot per object
            int allowedShots = 1;
            
            // large and middle size asteroids may be fired at more
            // than once, because of the chance to hit the fragments
            if (o instanceof Asteroid) {
                Asteroid a = (Asteroid)o;
                switch (a.getSquareRadius()) {
                    case Asteroid.LARGE_ASTEROID_SQUARE_RADIUS:
                        allowedShots = 4;
                        break;
                    case Asteroid.MIDDLE_ASTEROID_SQUARE_RADIUS:
                        allowedShots = 3;
                        break;
                    default:
                        allowedShots = 1;
                        break;
                }
            }
            
            return firedAtCount < allowedShots;
        }
        
        return true;
    }
    
    /**
     * Aims at a given target by calculating the optimal shot direction.
     * 
     * @param target The target to aim at. This object provides information
     *  about the location, direction and speed of the target.
     * @param shotOrigin the location from where the shot is fired
     * @param shotSpeed speed of the shot
     * @return the optimal shot direction or <code>null</code>, if the
     *  target cannot be hit with the given target speed and from the given 
     *  shot origin  
     */
    private Point aim(GameObject target, Point shotOrigin, double shotSpeed) {
        
        // determine direct shot vector
        Point delta = shotOrigin.delta(target.getLocation());
        double dx = delta.getX();
        double dy = delta.getY();
        
        // determine angle of direct shot
        double shotAngle = Math.atan2(dy, dx);
        
        // determine target speed 
        Point targetDirection = target.getDirection();
        double targetSpeed = Math.sqrt(targetDirection.getX()
                * targetDirection.getX() + targetDirection.getY()
                * targetDirection.getY());
    
        // calculate correction angle, if target is moving
        if (targetSpeed > 0) {
        
            // determine distance between shot origin and target
            double targetDistance = Math.sqrt(dx * dx + dy * dy);
            
            // determine angle between direct shot vector and motion vector of
            // target
            double cosbeta = (dx * targetDirection.getX() + dy
                    * targetDirection.getY()) / (targetDistance * targetSpeed);
            double beta = Math.acos(cosbeta);
    
            // calculate factor between shot speed and target speed
            // -> the shot is f times faster than the target
            double f = shotSpeed / targetSpeed;
    
            // calculate correction angle with law of sines
            double sinbeta = Math.sin(beta);
            double singamma = sinbeta / f;
            double correctionAngle = Math.asin(singamma); 
            
            // Determine, if correction angle has to be added or subtracted.
            // This is done by building the cross product between the shot 
            // vector and the target's motion vector
            double direction = dx * targetDirection.getY() - dy
                    * targetDirection.getX();
            
            // modify direct shot angle with correction angle
            if (direction < 0) {
                shotAngle -= correctionAngle;
            } else {
                shotAngle += correctionAngle;
            }
        }
        
        // check, if shot angle is valid, i.e. if it is possible
        // to hit the target
        if (shotAngle != Double.NaN) {
        
            // calculate corrected shot vector
            int sx = (int)Math.round(Math.cos(shotAngle) * shotSpeed);
            int sy = (int)Math.round(Math.sin(shotAngle) * shotSpeed);
            Point shotDirection = new Point(sx, sy);
            
            // return the calculated shot direction
            return shotDirection;
        } else {
            
            // corrected angle is invalid -> target cannot be hit
            // FIXME: it seems this branch is never reached :/
            return null;
        }
    }
}
