/*
 * Copyright (C) 2008 Henning Faber
 * 
 * This file is part of Sitting Duck Asteroids Bot project.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>. 
 */
package de.hfaber.asteroids.bot.research;

import java.text.MessageFormat;

import org.apache.log4j.Logger;

import de.hfaber.asteroids.bot.IAsteroidsBot;
import de.hfaber.asteroids.client.Commands;
import de.hfaber.asteroids.client.GameStatusEvent;
import de.hfaber.asteroids.game.field.Point;
import de.hfaber.asteroids.game.objects.Ship;
import de.hfaber.asteroids.game.objects.Bullet;
import de.hfaber.asteroids.game.state.GameStatus;

/**
 * @author Henning Faber
 */
public class ShotAngleCorrectnessTester implements IAsteroidsBot {

    /**
     * The LOGGER instance. 
     */
    private static final Logger LOGGER = Logger.getLogger(
            ShotAngleCorrectnessTester.class);
    
    /**
     * The last game status received from the mame server. 
     */
    private GameStatus m_gameStatus;
    
    /**
     * Bullet will be fired when counter reaches zero. Ship rotates left
     * while the counter has a negative value. Set random negative
     * number here, in order to test different angles. 
     */
    private int m_counter = -13;

    /**
     * The shot direction vector that the ship reports when the
     * shot is fired. This is the direction vector that is to be
     * verified.
     */
    private Point m_shotDirection;
    
    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.bot.IAsteroidsBot#compute()
     */
    public Commands compute() {
        Commands c = new Commands();
        
        // wait until the ship is present
        if (m_gameStatus.getShip() != null) {
            
            // increase counter
            m_counter++;
            if (m_counter < 0) {
                
                // rotation phase: 
                //  -> rotate left
                c.pressRight(true);
            } else if (m_counter == 0) {
                
                // shot phase: 
                //  -> stop rotating, fire a shot, remember the shot angle
                Ship ship = m_gameStatus.getShip();
                m_shotDirection = ship.getShotDirection();
                c.pressFire(true);
            } else {
                
                // verification phase: 
                //  -> track the shot, calculate and output the difference 
                //     between the actual shot angle and the previously assumed 
                //     angle
                if (m_gameStatus.getBulletCount() > 0) {
                    Bullet shot = m_gameStatus.getBullet(0);
                    double angle = m_shotDirection.angle(shot.getDirection());
                    String msg = MessageFormat.format(
                            "counter: {0}, location: {1}, ttl: {2}, "
                                + "angle: {3,number,#.#####}",
                            new Object[] { m_counter, shot.getLocation(),
                                        shot.getTtl(), angle });
                    LOGGER.info(msg);
                }
            }
        }
        
        return c;
    }

    /* (non-Javadoc)
     * @see de.hfaber.asteroids.asteroids.client.IGameStatusListener#gameStatusUpdated(de.hfaber.asteroids.asteroids.client.GameStatusEvent)
     */
    public void gameStatusUpdated(GameStatusEvent e) {
        m_gameStatus = e.getGameStatus();
    }
}
