// ============================================================
// Signaturen:
// Z 	boolean
// B 	byte
// C 	char
// S 	short
// I 	int
// J 	long
// F 	float
// D 	double
// V 	void
// LvollQualifizierterName; 	Objekttyp
// [Typ 	Typ[]
//
//	GetObjectField() 	jobject 	Object
//	GetBooleanField() jboolean 	boolean
//	GetByteField() 		jbyte 		byte
//	GetCharField() 		jchar 		char
//	GetShortField() 	jshort 		short
//	GetIntField() 		jint 			int
//	GetLongField() 		jlong 		long
//	GetFloatField() 	jfloat 		float
//	GetDoubleField() 	jdouble 	double
// ============================================================

#include <jni.h>
#include "ship.h"
#include "ship_consts.h"
#include <stdio.h>

#define DEBUG						 0

#define SHOT_MAX_LIFE		72			// vgl. Java Shot

#define MAX_X					1023
#define EXTENT_X			1024

#define MAX_Y					 767
#define EXTENT_Y			 768

class Target {
	public:
		Target()
		{
			leftMinHitInFrames			= 0;
			leftMinHitInFramesKeys	= 0;
			rightMinHitInFrames			= 0;
			rightMinHitInFramesKeys	= 0;
		};

		int			shotWillHit( const int shot_x, const int shot_y, int frameHorizon );
		
		/** u.a. Attribute aus dem Java-Objekt bernehmen */
		void		readJObject( JNIEnv *env, jobject jTarget );
		void		writeJObject( JNIEnv *env );

		void		setJInt( JNIEnv *env, jclass clazz, const char *field, const int value );
		int			Target::getJInt( JNIEnv *env, jclass clazz, jobject obj, char *field );
		double	Target::getJDouble( JNIEnv *env, jclass clazz, jobject obj, char *field );

		jobject jTarget;
		
		int			x;
		int			y;
		
		double	dx;
		double	dy;
		
		int			shots_fired_at;
		int			squaredPixelSize;
		int			squaredPixelSizeForDistCheck;	// 1/4 bei Shot 1, 1/10 bei Shots 2,3

		int			leftMinHitInFrames;
		int			leftMinHitInFramesKeys;
		int			rightMinHitInFrames;
		int			rightMinHitInFramesKeys;
};

Target *targets[27];		// max. 26 Asteroiden + 1 UFO

static int initFlag = 0;

// Schiffskoordinaten, -1 erzwingt beim ersten Aufruf einen Aufbau der Tabellen shot_offset_norm_x|y
static int ship_x		= -1;
static int ship_y		= -1;

/** Positionen eines Schusses in #n Frames bei Winkelbyte wb */
void getShotPrediction( int &px, int &py, const int frameHorizon, const int wb )
{
	// In Abstand von ca. 20 Abstand vom Schiffsmittelpunkt treten die Schsse aus:
	const int x0 = ship_x + shot_origin[wb][0];
	const int y0 = ship_y + shot_origin[wb][1];

	// Korrigiert auf -2: Im nchsten Frame wird erst der Bildschirminhalt zurckgegeben und dann
	// der Key "FIRE" ausgewertet, erst ab dem dritten Frame entfernt sich der Schuss
	px = x0 + (int)( ( frameHorizon - 2 ) * shot_delta[wb][0] );
	py = y0 + (int)( ( frameHorizon - 2 ) * shot_delta[wb][1] );
}

void updateShotOffsets()
{
#if DEBUG
	fprintf( stderr, "c++ updateShotOffsets x,y = (%4d,%4d)\n", ship_x, ship_y );
#endif

	for ( int wb = 0; wb < 256; wb++ ) {
		for ( int f = 2; f < 74; f++ ) {
			int sx = ship_x + shot_offset_x[wb][f];
			int sy = ship_y + shot_offset_y[wb][f];

			// Koordinaten des Schusses in das Spielfeld zurcknormalisieren:
			if ( sx < 0 ) {
				sx += EXTENT_X;
			}
			else if ( sx > MAX_X ) {
				sx -= EXTENT_X;
			}
			if ( sy < 0 ) {
				sy += EXTENT_Y;
			}
			else if ( sy > MAX_Y ) {
				sy -= EXTENT_Y;
			}
			shot_offset_norm_x[wb][f] = (short)sx;
			shot_offset_norm_y[wb][f] = (short)sy;
		}
	}
}

void init( const int sx, const int sy )
{
	ship_x = sx;
	ship_y = sy;

	for ( int i = 0; i < 27; i++ ) {
		targets[i] = new Target();
	}
	for ( int wb=0; wb < 256; wb++ ) {
		shot_delta[wb][0]		= shot_delta_tmp[wb][0];
		shot_delta[wb][1]		= shot_delta_tmp[wb][1];

		shot_origin[wb][0]	= (int)shot_delta_tmp[wb][4];
		shot_origin[wb][1]	= (int)shot_delta_tmp[wb][5];

#if DEBUG
		fprintf( stdout, "%3d:", wb );
#endif
		// Schussoffsets bei stehendem Schiff, pro Winkelbyte, pro Frame
		for ( int f = 2; f < 74; f++ ) {
			int px, py;
			getShotPrediction( px, py, f, wb );

			shot_offset_x[wb][f] = px;
			shot_offset_y[wb][f] = py;
#if DEBUG
			fprintf( stdout, "[%4d,%4d]", px, py );
#endif
		}
#if DEBUG
		fprintf( stdout, "\n" );
#endif
	}
	updateShotOffsets();
}

JNIEXPORT jobject JNICALL Java_de_wens02_Ship_findBestTargetJNI(
		JNIEnv *env, jclass clazzShip,
		jint _ship_x, jint _ship_y,
		jobject jTargets,
		jint _wbStart,
		jint _minKeyPress, jint _maxKeyPress,
		jint _currentShotLifeSpan, jint _frameHorizon )
{
	if ( !initFlag ) {
		// Einmalige Initialisierung der Hilfstabellen
		init( _ship_x, _ship_y );
		initFlag = 1;
	}
	if ( ship_x != _ship_x || ship_y != _ship_y ) {
		// Bei jeder nderung der Schiffsposition einmalig die Schussoffsets
		// neu berechnen und in das Spielfeld normalisieren:
		ship_x = _ship_x;
		ship_y = _ship_y;

		updateShotOffsets();
	}
	jclass		clazz			= env->GetObjectClass( jTargets );
	jmethodID	vsizemid	= env->GetMethodID( clazz, "size",			"()I" );
	jmethodID vgetmid		= env->GetMethodID( clazz, "elementAt",	"(I)Ljava/lang/Object;" );

	if ( NULL == vsizemid ) {
		fprintf( stderr, "vsizemid == NULL\n" );
		return 0;
	}
	if ( NULL == vgetmid ) {
		fprintf( stderr, "vgetmid == NULL\n" );
		return 0;
	}
	const int nTargets = (int)env->CallIntMethod( jTargets, vsizemid );

	for ( int i = 0; i < nTargets; i++ ) {
		jobject jTarget = env->CallObjectMethod( jTargets, vgetmid, (jint)i );
		Target	*target	= targets[i];

		target->readJObject( env, jTarget );

		target->leftMinHitInFrames			= 1000;
		target->leftMinHitInFramesKeys	= 1000;

		target->rightMinHitInFrames			= 1000;
		target->rightMinHitInFramesKeys	= 1000;

#if DEBUG
		fprintf( stdout, "[%2d] x,y=%4d,%4d %5f,%5f; %d/%d (%d|%d %d|%d)\n",
				i,
				targets->x, targets[i]->y,
				targets->dx, targets[i]->dy,
				targets->squaredPixelSize,
				targets->shots_fired_at,
				targets->leftMinHitInFrames,
				targets->leftMinHitInFramesKeys,
				targets->rightMinHitInFrames,
				targets->rightMinHitInFramesKeys );
#endif
	}
	// Mit welchem Winkelbyte msste man schiessen, um zu treffen?
	// Nur max. eine halbe Drehung bercksichtigen
	unsigned char wb_left		= (unsigned char)_wbStart;
	unsigned char wb_right	= (unsigned char)_wbStart;

	const int lifeSpanOffset	= SHOT_MAX_LIFE - (int)_currentShotLifeSpan;
	const int minKeyPress			= (int)_minKeyPress;
	const int maxKeyPress			= (int)_maxKeyPress;
	const int frameHorizon		= (int)_frameHorizon;

	Target *tBest		= NULL;
	int			tFrames	= 1000;

#if DEBUG
	fprintf( stdout, "nTargets = %2d, wb %3d, %2d->%2d, #f=%d, lifeSpan=%d\n",
			nTargets, wb_left, minKeyPress, maxKeyPress, frameHorizon, (int)_currentShotLifeSpan );
#endif
	
	for ( int keypress = minKeyPress; keypress <= maxKeyPress; keypress++ ) {
		const int shotLifeSpan	= SHOT_MAX_LIFE - ( ( frameHorizon + keypress + lifeSpanOffset ) & 3 );
		// Edit 6.6.08: -1 wegen der Tatsache, dass der Schuss im letzten Frame nichts mehr trifft
		const int maxShotFrame	= keypress + 2 + ( shotLifeSpan - 1 );

		for ( int f = keypress + 2; f < maxShotFrame; f++ ) {
			if ( f > tFrames ) {
				// Ein schneller erreichbares Ziel ist schon gefunden.
				break;
			}
			const int fk = f - keypress;
	
			const int lShotx = shot_offset_norm_x[wb_left][fk];
			const int lShoty = shot_offset_norm_y[wb_left][fk];
	
			const int rShotx = shot_offset_norm_x[wb_right][fk];
			const int rShoty = shot_offset_norm_y[wb_right][fk];

			for ( int i = 0; i < nTargets; i++ ) {
				Target *target = targets[i];

				if ( target->shotWillHit( lShotx, lShoty, frameHorizon + f ) ) {
#if DEBUG
					fprintf( stdout, "L[%2d] [%4d,%4d] ->(K%2d,F%2d)-> [%4d,%4d]\n",
							i, target->x, target->y, keypress, frameHorizon + f, lShotx, lShoty );
#endif
					// Minimale Frameanzahl, bis target abgeschossen ist, incl. Drehung
					if ( f < target->leftMinHitInFrames ) {
						target->leftMinHitInFrames			= f;
						target->leftMinHitInFramesKeys	= keypress;
					}
					if ( NULL == tBest || target->leftMinHitInFrames < tFrames ) {
						tBest		= target;
						tFrames	= target->leftMinHitInFrames;
					}
				}
				if ( target->shotWillHit( rShotx, rShoty, frameHorizon + f ) ) {
#if DEBUG
					fprintf( stdout, "R[%2d] [%4d,%4d] ->(K%2d,F%2d)-> [%4d,%4d]\n",
							i, target->x, target->y, keypress, frameHorizon + f, lShotx, lShoty );
#endif
					if ( f < target->rightMinHitInFrames ) {
						target->rightMinHitInFrames			= f;
						target->rightMinHitInFramesKeys	= keypress;
					}
					if ( NULL == tBest || target->rightMinHitInFrames < tFrames ) {
						tBest		= target;
						tFrames	= target->rightMinHitInFrames;
					}
				}
			}
		}
		wb_left++;
		wb_right--;
	}
	// Ergebnisse zurckschreiben:
	for ( int i = 0; i < nTargets; i++ ) {
		Target *target = targets[i];
		target->writeJObject( env );
	}
	if ( NULL != tBest ) {
		return tBest->jTarget;
	}
	return NULL;
}

int Target::shotWillHit( const int shot_x, const int shot_y, int frameHorizon )
{
	if ( shots_fired_at >= 1 ) {
		// Siehe Java Asteroid.shotWillHit, es wird mit frameHorizon + 1 gerechnet
		frameHorizon++;
	}
	int tx = x + (int)( frameHorizon * dx );
	int ty = y + (int)( frameHorizon * dy );

	if ( tx < 0 ) tx += EXTENT_X; else if ( tx > MAX_X ) tx -= EXTENT_X;
	if ( ty < 0 ) ty += EXTENT_Y; else if ( ty > MAX_Y ) ty -= EXTENT_Y;

	const int sdx			= shot_x - tx;
	const int sdy			= shot_y - ty;

	const int distSq	= sdx * sdx + sdy * sdy;

	return distSq < squaredPixelSizeForDistCheck;
}

void Target::writeJObject( JNIEnv *env )
{
	jclass clazz = env->GetObjectClass( jTarget );

	setJInt( env, clazz, "leftMinHitInFrames",			leftMinHitInFrames );
	setJInt( env, clazz, "leftMinHitInFramesKeys",	leftMinHitInFramesKeys );

	setJInt( env, clazz, "rightMinHitInFrames",			rightMinHitInFrames );
	setJInt( env, clazz, "rightMinHitInFramesKeys",	rightMinHitInFramesKeys );
}

void Target::setJInt( JNIEnv *env, jclass clazz, const char *field, const int value )
{
	jfieldID fid = env->GetFieldID( clazz, field, "I" );
	env->SetIntField( jTarget, fid, value );
}

void Target::readJObject( JNIEnv *env, jobject _jTarget )
{
	jTarget = _jTarget;

	jclass clazz	= env->GetObjectClass( jTarget );

	x		= getJInt( env, clazz, jTarget, "x" );
	y		= getJInt( env, clazz, jTarget, "y" );

	dx	= getJDouble( env, clazz, jTarget, "dx" );
	dy	= getJDouble( env, clazz, jTarget, "dy" );

	shots_fired_at		= getJInt( env, clazz, jTarget, "shots_fired_at" );
	squaredPixelSize	= getJInt( env, clazz, jTarget, "squaredPixelSize" );

	// vgl. Java Asteroid.shotWillHit( Point p, int frameHorizon )
	squaredPixelSizeForDistCheck	= shots_fired_at >= 1	? squaredPixelSize / 10
																											: squaredPixelSize /  4;
}

int Target::getJInt( JNIEnv *env, jclass clazz, jobject obj, char *field )
{
	jfieldID fid = env->GetFieldID( clazz, field, "I" );
	return (int)env->GetIntField( obj, fid );
}

double Target::getJDouble( JNIEnv *env, jclass clazz, jobject obj, char *field )
{
	jfieldID fid = env->GetFieldID( clazz, field, "D" );
	return (double)env->GetDoubleField( obj, fid );
}
