package io;


import helpers.CombatHeuristics;
import helpers.CombatMessages;
import helpers.CombatStatistics;

import java.util.LinkedList;
import java.util.ListIterator;

import objects.Asteroid;
import objects.CombatObject;
import objects.Saucer;
import objects.Ship;
import objects.Shot;
import objects.TargetObject;
import utils.Utils;


public class CombatState {	
	
	public static final int MAX_ASTEROIDS = 26;

	public static final int MAX_SHOTS = 6;
	
	private	Ship ship;
	
	private	Saucer saucer;
	
	private TargetObject nextTarget;
	
	private	LinkedList<Asteroid> asteroids;
	
	private	LinkedList<Shot> shots;
	
	private	LinkedList<Shot> pendingShots;
	
	private int packetCount;
	
	private int packetDif;
	
	private int ships;
	
	private boolean isIngame;
	
	private boolean isValid;
	
	private boolean isMovementPhase;
	
	private boolean areAsteroidsVisible;

	public CombatState() {
		asteroids = new LinkedList<Asteroid>();
		shots = new LinkedList<Shot>();
		pendingShots = new LinkedList<Shot>();
		ship = new Ship(0, 0, 0, 0);
		reset();
	}
	
	private Shot createShot(int x, int y)
	{
		double shipDist = Double.MAX_VALUE;
		double saucerDist = Double.MAX_VALUE;
		if(isShipAlive())
		{
			shipDist = Utils.getDistSquare(x, y, ship.getPosX() , ship.getPosY());
		}
		if(isSaucerAlive())
		{
			saucerDist = Utils.getDistSquare(x, y, saucer.getPosX() , saucer.getPosY());
		}
		Shot newShot = null;		
		if(shipDist < saucerDist)
		{
			newShot = new Shot(x, y, ship, nextTarget, packetCount);			
			++CombatStatistics.SHOTS_FIRED;
			nextTarget = null;
		} else {
			newShot =  new Shot(x, y, saucer, null, packetCount);
		}
		return newShot;
	}
	
	private int createWord(byte lo, byte hi)
	{
		return (((hi & 0xFF) << 8) | (lo & 0xFF));
	}
	
	public void fireFlag(TargetObject target, int hitTime)
	{
		nextTarget = target;
		ship.onTorpedoTriggered(hitTime);
		nextTarget.onTargeted();
	}
	
	public LinkedList<Asteroid> getAsteroids()
	{
		return asteroids;
	}
	
	public int getAsteroidNumber()
	{
		return asteroids.size();
	}
	
	public boolean areTargetsLeft()
	{
		for(Asteroid curAsteroid : asteroids)
		{
			if(curAsteroid.getTargetedStatus() == 0)
			{
				return true;
			}
		}
		return false;
	}
	
	public int getPacketCount()
	{
		return packetCount;
	}
	
	public int getPacketDif()
	{
		return packetDif;
	}
	
	public Saucer getSaucer()
	{
		return saucer;
	}
	
	public Ship getShip()
	{
		return ship;
	}
	
	public int getShipsLeft()
	{
		return ships;
	}
	
	public LinkedList<Shot> getShots()
	{
		return shots;
	}
	
	public boolean isSaucerAlive() {
		return saucer != null;
	}
	
	public boolean isShipAlive() {
		return ship.isAlive();
	}
	
	public boolean isIngame() {
		return isIngame;
	}
	
	public boolean isValid() {
		return isValid;
	}
	
	public boolean isMovementPhase() {
		return isMovementPhase;
	}
	
	public boolean isSectorEmpty() {
		return asteroids.isEmpty() && saucer == null;
	}
	
	private void matchAsteroid(ListIterator<Asteroid> curAsteroid, int x, int y, int type, int scale)
	{
		if(curAsteroid.hasNext())
		{
			Asteroid required = curAsteroid.next();
			if(required.matches(x, y, type, scale, packetDif)) {
				required.update(x, y, packetDif);
				return;
			} else {
				int delCount = 0;
				while(curAsteroid.hasNext() && delCount < Utils.MAX_SIMULTANEOUS_SHOTS)
				{
					Asteroid sufficient = curAsteroid.next();
					if(sufficient.matches(x, y, type, scale, packetDif))
					{
						sufficient.update(x, y, packetDif);
						Asteroid delAsteroid = curAsteroid.previous();
						while(delAsteroid != required)
						{
							delAsteroid = curAsteroid.previous();
							delAsteroid.destroy();
							freeShots(delAsteroid);
							curAsteroid.remove();							
						}				
						curAsteroid.next();
						return;
					}
					++delCount;
				}
				while(curAsteroid.previous() != required) {	}
				Asteroid newAsteroid = new Asteroid(x, y, type, scale);
				curAsteroid.add(newAsteroid);
			}
		} else {
			Asteroid newAsteroid = new Asteroid(x, y, type, scale);
			curAsteroid.add(newAsteroid);
		}
	}
	
	private void matchPacketCount(int id)
	{
		if(isValid)
		{
			if(packetCount > id)
			{
				packetCount -= 256;
			}
			packetDif = id - packetCount;
			if(packetDif > 1)
			{
				CombatStatistics.PACKET_LOSS += packetDif - 1;
			}
		}
		packetCount = id;
		isValid= true;
	}
	
	private void matchSaucer(int x, int y, int scale)
	{
		if(!isSaucerAlive())
		{
			CombatMessages.verbose(CombatMessages.SAUCER_DETECT);
			saucer = new Saucer(x, y, scale);
		} else {
			if(saucer.matches(x, y, packetDif))
			{
				saucer.update(x, y, packetDif);
			} else {
				saucer.reMatch(x, y, packetDif);
			}
		}
	}
	
	private void matchShip(int x, int y, int dirX, int dirY)
	{
		if(!isShipAlive())
		{
			ship.spawn(x, y);
		} else {
			ship.update(x, y, packetDif);
		}
		ship.setDirection(dirX, dirY);
	}
	
	private void matchShot(ListIterator<Shot> curShot, int x, int y)
	{
		if(curShot.hasNext())
		{
			Shot required = curShot.next();
			if(required.matches(x, y, packetDif)) {
				required.update(x, y, packetDif);
				return;
			} else {
				while(curShot.hasNext())
				{
					Shot sufficient = curShot.next();
					if(sufficient.matches(x, y, packetDif))
					{
						sufficient.update(x, y, packetDif);
						Shot delShot = curShot.previous();
						while(delShot != required)
						{
							delShot = curShot.previous();
							delShot.destroy();
							curShot.remove();							
						}				
						curShot.next();
						return;
					}
				}
				while(curShot.previous() != required) {	}
				curShot.add(createShot(x, y));
			}
		} else {
			curShot.add(createShot(x, y));
		}
	}
	
	private void freeShots(TargetObject target)
	{
		pendingShots.addAll(target.getShotsFiredUpon());
	}
	
	public boolean isLastTargetLeft()
	{
		int numTargets = 0;
		for(Asteroid curAsteroid : asteroids)
		{
			if(curAsteroid.getTargetedStatus() == 0)
			{
				++numTargets;
				if(numTargets > 1)
				{
					break;
				}
			}
		}
		return numTargets == 1;
	}

	private void reAssignShots()
	{
		for(Shot curShot : pendingShots)
		{
			if(curShot.isAlive())
			{
				curShot.setTimeToDestination(-1);
				for(Asteroid curAsteroid : asteroids)
				{
					double moveAngle = curShot.getMovementAngle();
					int xDif = Utils.getXDif(curShot.getPosX(), curAsteroid.getPosX());
					int yDif = Utils.getYDif(curShot.getPosY(), curAsteroid.getPosY());
					double dist = Utils.TABLES.dist(xDif, yDif);
					double time;
					switch(curAsteroid.getScale())
					{
					case CombatObject.LARGE_SCALE: 
						time = dist / curShot.getSpeed() - 3; 
						break;
					case CombatObject.MEDIUM_SCALE: 
						time = dist / curShot.getSpeed() - 1; 
						break;
					default: 
						time = dist / curShot.getSpeed(); 
						break;
					}
					if(time < curShot.getTimeToDestination())
					{
						double angleToTarget = Utils.TABLES.atan2(yDif, xDif);
						double angleDif = Math.abs(Utils.getAngleDif(angleToTarget, moveAngle));
						if(angleDif < Math.PI / 4.0)
						{
							if(Utils.TABLES.sin(angleDif) * dist < CombatHeuristics.getHitZone(curAsteroid))
							{
								curShot.reAssignTarget(curAsteroid, (int)Math.round(time));
							}
						}
					}
				}
			}
		}
	}
	
	public String toString()
	{
		StringBuffer result = new StringBuffer();
		result.append(" #");
		result.append(packetCount);
		result.append(" Skipped: ");
		result.append(packetDif);
		result.append("\n\nShip: ");
		result.append(ship);
		result.append("\nShips: ");
		result.append(ships);
		result.append("\nSaucer: ");
		result.append(saucer);
		result.append("\n\nShots: ");
		for(Shot curShot : shots)
		{
			result.append("\n" + curShot);
		}
		result.append("\n\nAsteroids: ");
		for(Asteroid curAsteroid : asteroids)
		{
			result.append("\n" + curAsteroid);
		}
		return result.toString();
	}
	
	void reset()
	{
		if(isSaucerAlive())
		{
			saucer.destroy();
		}
		saucer = null;
		nextTarget = null;
		for(Asteroid curAsteroid : asteroids)
		{
			curAsteroid.destroy();
		}
		asteroids.clear();
		shots.clear();
		pendingShots.clear();
		packetCount = 0;
		packetDif = 0;
		ships = 3;
		isIngame = false;
		isValid = false;
		isMovementPhase = false;
		areAsteroidsVisible = false;
		ship.prepareTorpedos();
		ship.destroy();		
	}
	
	void update(byte[] data) 
	{
		ListIterator<Asteroid> curAsteroid = asteroids.listIterator();
		ListIterator<Shot> curShot = shots.listIterator();
		int curY = 0;
		int curX = 0;
		int curScale = 0;
		int dirX = 0;
		int dirY = 0;
		int vz = 0;
		int v1x = 0;
		int v1y = 0;
		int shipdetect = 0;		
		boolean saucerDetected = false;
		boolean doExit = false;
		boolean readScore = true;
		boolean isExplosionExisting = false;
		boolean ingameDetected = true;
		String scoreStr = "";
		int shipsLeft = 0;
		int i = 2;
		pendingShots.clear();
		switch(data[1])
		{
		case (byte)0xE2: 
			isMovementPhase = false;
			break;
		case (byte)0xE0:
			isMovementPhase = true;
			break;
		default: 
			// JMPL should be 1st command always
			isIngame = false;
			isValid = false;
			return; 	
		}
		matchPacketCount(data[1024] & 0xFF);
		
		// interpret data (according to example code)
		while(!doExit) 
		{
			// retrieve word an operation code
			int word = createWord(data[i], data[i + 1]);
			int opCode =  word >> 12;
			
			// handle code
			switch (opCode)
			{
			case 0xA:	// LABS (position ray) -> double word
				curY = word & 0x3FF;	
				i += 2;
				word = createWord(data[i], data[i + 1]);			
				curX = word & 0x3FF;
				curScale = word >> 12;
				break;
			case 0xB:	// HALT (exit)
				doExit = true;
				break;
			case 0xC:	// JSRL (subroutine start)
				switch (word & 0xfff)
				{
				case 0x880:
				case 0x896:
				case 0x8B5:
				case 0x8D0:
					isExplosionExisting = true;
					break;
				case 0x8F3:
					matchAsteroid(curAsteroid, curX, curY, 1, curScale);					
					break;
				case 0x8FF:
					matchAsteroid(curAsteroid, curX, curY, 2, curScale);
					break;
				case 0x90D:
					matchAsteroid(curAsteroid, curX, curY, 3, curScale);
					break;
				case 0x91A:
					matchAsteroid(curAsteroid, curX, curY, 4, curScale);
					break;
				case 0x929:
					saucerDetected = true;
					matchSaucer(curX, curY, curScale);					
					break;
				case 0xA6D:
					readScore = false;
					if(!scoreStr.isEmpty()) {						
						int newScore = Integer.parseInt(scoreStr);
						CombatStatistics.setScore(newScore);
					}
					++shipsLeft;					
					break;
				case 0xADD:
					if(readScore) scoreStr += "0";
					break;
				case 0xB2E:
					if(readScore) scoreStr += "1";
					break;
				case 0xB32:
					if(readScore) scoreStr += "2";
					break;
				case 0xB3A:
					if(readScore) scoreStr += "3";
					break;
				case 0xB41:
					if(readScore) scoreStr += "4";
					break;
				case 0xB48:
					if(readScore) scoreStr += "5";
					break;
				case 0xB4F:
					if(readScore) scoreStr += "6";
					break;
				case 0xB56:
					if(readScore) scoreStr += "7";
					break;
				case 0xB5B:
					if(readScore) scoreStr += "8";
					break;
				case 0xB63:
					if(readScore) scoreStr += "9";
					break;
				case 0xA9B: // 'C' of 'Druecken'
					ingameDetected = false;
					readScore = false;
					break;
				case 0xAFB:	// 'S' of 'Spieler 1'
					ingameDetected = false;
					readScore = false;
					break;
				default:
					break;
				}
				break;
			case 0xD:	// RTSL (subroutine return)
				doExit = true;
				break;
			case 0xE:	// JMPL (jump out)
				doExit = true;
				break;
			case 0xF:	// SVEC (draw short vector)
				break;
			default:	// VCTR (draw long vector) -> double word
				dirY = word & 0x3ff;
				if ((word & 0x400) != 0)
				{
					dirY = -dirY;
				}
				i += 2;
				word = createWord(data[i], data[i + 1]);			
				dirX = word & 0x3ff;
				if ((word & 0x400) != 0)
				{
					dirX = -dirX;
				}
				vz = word >> 12;
				if (dirX == 0 && dirY == 0 && vz == 15)
				{				
					matchShot(curShot, curX, curY);
				} else if (opCode == 6 && vz == 12 && dirX != 0 && dirY != 0)
				{
					switch (shipdetect)
					{
					case 0:
						v1x = dirX;
						v1y = dirY;
						++shipdetect;
						break;
					case 1:
						matchShip(curX, curY, v1x - dirX, v1y - dirY);						
						++shipdetect;
						break;
					}
				}
				break;
			}
			
			// increase data index if possible
			i += 2;
			if(i > 1022)
			{
				doExit = true;
			}
		}
		
		// check if ship is dead
		if(shipdetect != 2)
		{
			if(isShipAlive())
			{
				ship.destroy();
			}
			ship.updateAway(packetDif);
		}
		
		// check if saucer is dead
		if(!saucerDetected && isSaucerAlive())
		{
			CombatMessages.verbose(CombatMessages.SAUCER_DESTROYED);
			CombatStatistics.setSaucerDestroyed();
			if(!areAsteroidsVisible)
			{
				CombatStatistics.setSectorFinished();				
				CombatMessages.verbose(CombatMessages.SECTOR_CLEAN);
			}
			saucer.destroy();
			saucer = null;
		}
		
		// remove non-matched asteroids
		while(curAsteroid.hasNext())
		{
			Asteroid a = curAsteroid.next();
			a.destroy();
			freeShots(a);
			curAsteroid.remove();
		}
		
		// remove non-matched shots
		while(curShot.hasNext())
		{			
			Shot s = curShot.next();
			s.destroy();
			curShot.remove();
		}
		
		reAssignShots();
		
		// update sector based on existing asteroids / saucer
		if(areAsteroidsVisible)
		{
			if(!isExplosionExisting && asteroids.isEmpty())
			{
				areAsteroidsVisible = false;
				if(saucer == null)
				{
					CombatStatistics.setSectorFinished();
					CombatMessages.verbose(CombatMessages.SECTOR_CLEAN);
				}
			}
		} else {
			if(!asteroids.isEmpty())
			{
				areAsteroidsVisible = true;
				CombatStatistics.SECTOR_START_TIME = CombatStatistics.TIME;				
				CombatMessages.verbose(CombatMessages.SECTOR_NEW);
			}
		}
		
		// update lives
		ships = shipsLeft;
		if(shipsLeft == 0)
		{
			ingameDetected = false;
		}
		
		// update ingame status
		if(ingameDetected)
		{
			if(!isIngame)
			{
				isIngame = true;
				CombatStatistics.setNewGameStarted();
			}
		} else {
			isIngame = false;
		}
	}
	
	void update(CombatControl control)
	{
		if(isShipAlive()) 
		{
			for(int i = 0; i < packetDif; i++)
			{
				if(control.hasTurnedLeft())			
				{
					ship.performTurnLeft();
				} else if(control.hasTurnedRight())
				{
					ship.performTurnRight();
				}
				if(isMovementPhase)
				{
					if(control.hasAccelerated())
					{
						ship.performAcceleration();
					} else {
						ship.performDeceleration();
					}
				} else {
					ship.updateMovement();
				}
			}
		}
	}
}
