package io;
import gui.StatusPanel;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.UnknownHostException;


public class CombatConnection {
	
	public static final String DEFAULT_HOST = "127.0.0.1";
	
	public static final int DEFAULT_PORT = 1979;
	
	public static final int RECEIVE_TIMEOUT = 5000;
	
	public static final int RETRY_DELAY = 2000;
	
	private byte[] inBuffer = new byte[1026];
	
	private byte[] outBuffer = {'c','t','m', 'a', 'm', 'e', 0, 0};
	
	private String host = DEFAULT_HOST;
	
	private int port = DEFAULT_PORT;
	
	private DatagramSocket socket = null;
	
	private boolean isConnected = false;
	
	private boolean isAlive = false;
	
	private CombatState state = new CombatState();
	
	private CombatControl control = new CombatControl();
	
	private CombatAI listener;
	
	private boolean stop = false;
	
	private Runnable communicationCycle = new Runnable()
	{
		public void run()
		{
			while(!stop)
			{
				// Connection cycle
				if(!stop)
				{
					connect();
					while(!isConnected && !isAlive && !stop)
					{				
						try {
							Thread.sleep(RETRY_DELAY);
						} catch (InterruptedException e) {}
						connect();
					}
					updateStatus();
				}
				// Initialization cycle
				if(!stop)
				{
					sendControl(control);				
					receiveState(state);
					while(isConnected && !isAlive && !stop)
					{				
						try {
							Thread.sleep(RETRY_DELAY);
						} catch (InterruptedException e) {}
						sendControl(control);				
						receiveState(state);
						updateStatus();
					} 
				}
				// Working cycle
				if(!stop)
				{
					while(isConnected && isAlive && !stop)
					{
						if(receiveState(state))
						{
							if(listener.isActivated())
							{
								listener.handleState(state, control);
							} else {
								listener.watchState(state, control);
							}
							sendControl(control);
						}
					}
					state.reset();
					control.reset();
					listener.reset();
				}
			}
			disconnect();
		}
	};
	
	private Thread communicationThread = new Thread(communicationCycle);
	
	private StatusPanel guiOut = null;
	
	public CombatConnection(String host, int port, CombatAI listener)
	{
		this.host = host;
		this.port = port;
		this.listener = listener;	
	}
	
	private void connect()
	{		
		try{	
			disconnect();
			InetAddress adress = InetAddress.getByName(host);
			socket = new DatagramSocket();	
			socket.setSoTimeout(RECEIVE_TIMEOUT);
			socket.connect(adress, port);
			isConnected = true;
			System.out.println("connected to "+host+" at port "+port);
		}catch(UnknownHostException uhe){
			isConnected = false;
			System.out.println("unable to connect: "+uhe);
		}catch(IOException ioe){			
			isConnected = false;
			System.out.println("unable to connect: "+ioe);
		}
	}
	
	private void disconnect()
	{
		if(socket != null)
		{
			socket.close();
		}
	}
	
	public String getHost()
	{
		return host;
	}
	
	public int getPort()
	{
		return port;
	}
	
	public boolean isAlive()
	{
		return isAlive;
	}
	
	public boolean isConnected()
	{
		return isConnected;
	}
	
	private boolean receiveState(CombatState state)
	{		
		try {
			DatagramPacket packet = new DatagramPacket(inBuffer, inBuffer.length);
			socket.receive(packet);
			state.update(inBuffer);
			if(state.isValid())
			{
				control.onApproval(inBuffer[1025] & 0xFF);
				state.update(control);
				isAlive = true;
			} else {
				isAlive = false;
			}
		} catch (IOException e) {
			isAlive = false;
		}
		return isAlive;
	}
	
	public void selectHost(String server)
	{
		String[] values = server.split(":");
		if(values.length < 3)
		{
			if(values.length > 0)
			{
				host = values[0];
			}
			if(values.length > 1)
			{
				port = Integer.parseInt(values[1]);
			} else {
				port = DEFAULT_PORT;
			}
			disconnect();
			updateConnection();
		}
	}
	
	private void sendControl(CombatControl control)
	{		
		try {	
			outBuffer[6] = control.getControl();
			control.onSend();
			outBuffer[7] = control.getPacketCount();
			DatagramPacket packet = new DatagramPacket(outBuffer, outBuffer.length);
			socket.send(packet);
			isConnected = true;
		} catch (IOException e) {
			isConnected = false;
		}
	}
	
	public void setGUIOutput(StatusPanel guiOut)
	{
		this.guiOut = guiOut;
		updateConnection();
		updateStatus();
	}
	
	public void startCommunication()
	{
		communicationThread.setName("Connection Cycle Thread");
		communicationThread.setPriority(Thread.MAX_PRIORITY);
		communicationThread.start();	
	}
	
	public void stopCommunication()
	{
		stop = true;
		try {
			communicationThread.interrupt();
			communicationThread.join();
		} catch (InterruptedException e) {}
	}
	
	public String toString()
	{
		String serverFormat = "%s:%d";
		return String.format(serverFormat, host, port);
	}
	
	public void updateConnection()
	{
		if(guiOut != null)
		{
			guiOut.updateConnection(getHost(), getPort());
		}
	}
	
	public void updateStatus()
	{
		if(guiOut != null)
		{
			if(isConnected)
			{
				if(isAlive)
				{
					if(listener != null && listener.isActivated())
					{
						guiOut.updateState(StatusPanel.STATE_RUNNING);
					} else {
						guiOut.updateState(StatusPanel.STATE_CONNECTED);
					}
				} else {
					guiOut.updateState(StatusPanel.STATE_WAITING);
				}
			} else {
				guiOut.updateState(StatusPanel.STATE_ERROR);
			}			
		} 
	}
}
