// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   (c) 2008  Rammi (rammi@caff.de)
//                     This code is in the public domain.
//                     Use at own risk.
//                     No guarantees given.
//
// Latest change:      $Date$
//
// History:	       $Log$
//=============================================================================
package de.caff.asteroid;

import java.awt.*;
import java.awt.font.FontRenderContext;
import java.util.Collection;

/**
 *  A text (or number) displayed on screen.
 *
 *  This class is part of a solution for a
 *  <a href="http://www.heise.de/ct/creativ/08/02/details/">competition by the German computer magazine c't</a>
 */
public class Text
        extends GameObject
{
  /** Need font render context to calculate text bounds. */
  private static FontRenderContext fontRenderContext = null;

  /** Font to use for scaling 1. */
  public static final Font FONT = new Font(Font.MONOSPACED, Font.PLAIN, 14);
  /** The text. */
  private final String text;
  /** The scale. */
  private final int scale;

  /**
   *  Constructor.
   *  @param text text
   *  @param x     positon x
   *  @param y     postion y
   *  @param scale text scaling
   */
  public Text(String text, int x, int y, int scale)
  {
    super(x, y);
    this.text = text;
    this.scale = (int)Math.pow(2.0, scale);
  }

  /**
   *  Get the displayed text.
   *  Take care that all text are only uppercase and use 0 (zero) instead of O (oh).
   *  @return text
   */
  public String getText()
  {
    return text;
  }

  /**
   * Returns a string representation of the object.
   *
   * @return a string representation of the object.
   */
  @Override
  public String toString()
  {
    return String.format("Text(%s)@(%d,%d)", text, x, y);
  }

  /**
   *  Has this text a given location?
   *  @param px x coordinate
   *  @param py y coordinate
   *  @return <code>true</code> if this text is at the given location,
   *          <code>false</code> otherwise
   */
  public boolean hasLocation(int px, int py)
  {
    return px == x  &&  py == y;
  }

  /**
   *  Has this text a given location?
   *  @param p location
   *  @return <code>true</code> if this text is at the given location,
   *          <code>false</code> otherwise
   */
  public boolean hasLocation(Point p)
  {
    return hasLocation(p.x, p.y);
  }

  /**
   * Get the bounding box of this rectangle.
   *
   * @return the bounding box
   */
  public Rectangle getBounds()
  {
    if (fontRenderContext != null) {
      Rectangle bounds = getFont().getStringBounds(getText(), fontRenderContext).getBounds();
      bounds.y = -bounds.height - bounds.y; // NOTE: y points up, but getStringBounds() assumes down.
      bounds.translate(getX(), getY());
      return bounds;
    }
    else {
      // w/o contexrt we just return a fix rectangle
      return new Rectangle(getLocation(),
                           new Dimension(100, 20));
    }
  }

  /**
   *  Get the scale.
   *  @return scaling
   */
  public int getScale()
  {
    return scale;
  }

  /**
   *  Get a font which can be used to display the text.
   *  @return font
   */
  public Font getFont()
  {
    return FONT.deriveFont((float)(FONT.getSize() * scale));
  }

  /**
   * Get the properties of this object.
   *
   * @return collection of properties
   */
  @Override
  public Collection<Property> getProperties()
  {
    Collection<Property> props = super.getProperties();
    props.add(new Property<String>("Text", getText()));
    props.add(new Property<Integer>("Scale", getScale()));
    return props;
  }

  /**
   * Get the type of game object.
   *
   * @return game object type
   */
  public String getObjectType()
  {
    return "Text";
  }

  /**
   *  Workaround to allow bounds calculation.
   *  Call this at least once.
   *  Sorry for the inconvenience!
   *  @param ctx font render context, usually gotten from Graphics object
   */
  public static void setFontRenderContext(FontRenderContext ctx)
  {
    fontRenderContext = ctx;
  }
}
