import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;

import javax.swing.JPanel;


public class VisuPanel extends JPanel 
{
	private static final long serialVersionUID = 1L;

	State currentState = null;
	
	VisuPanel()
	{
		setBackground(Color.BLACK);
		setForeground(Color.WHITE);
		setOpaque(true);
	}
	
	@Override
	public Dimension getPreferredSize() 
	{
		return new Dimension(1024, 768);
	}

	public void setState(State state)
	{
		currentState = state;
	}
	
	@Override
	public void paint(Graphics g) 
	{
		Graphics2D g2d = (Graphics2D) g;

		g.setColor(Color.BLACK);
		g.fillRect(0, 0, 1024, 768);
		
		if(currentState == null)
		{
			return;
		}
		
		paintShip(g2d);
		paintAsteroids(g2d);
		paintShotsLaunched(g2d);
		paintShots(g2d);
		paintUfo(g2d);
		
		g.setColor(Color.WHITE);
		g.drawString("Score: " + currentState.score, 20, 20);

		g.drawString("Missed: " + currentState.shotsMissed, 
				150, 20);

		long millisPlayed = currentState.currentTime - currentState.timeGameStarted;
		
		String millis = "000" + millisPlayed % 1000;
		g.drawString("Time: " + millisPlayed / 60000 + ":" + millisPlayed % 60000 / 1000 
						+ "," + millis.substring(millis.length() - 3), 
					300, 20);
		
		g.drawString("Frame: " + currentState.frame, 
				500, 20);

		g.drawString("Keys: " 
					+ ((currentState.keys & 0x01) > 0 ? "H" : "")
					+ ((currentState.keys & 0x02) > 0 ? "F" : "")
					+ ((currentState.keys & 0x04) > 0 ? "S" : "")
					+ ((currentState.keys & 0x08) > 0 ? "R" : "")
					+ ((currentState.keys & 0x10) > 0 ? "L" : ""),
				700, 20);

		g.drawString("Ships: " + currentState.ships,
				800, 20);
		
		if(currentState.ship != null)
		{
			g.drawString("Angle: " + currentState.ship.angle,
					900, 20);
		}
	}

	private void paintShip(Graphics2D g) 
	{
		g.setColor(Color.green);
		paintBounds(g, currentState.ship);
		
		State.Ship s = currentState.ship;
		if(s != null)
		{
			g.drawLine((int) (s.x + 0.5), (int) (896.5 - s.y), 
					(int) (s.x + s.dx / 100 + 0.5), (int) (896.5 - (s.y + s.dy / 100)));
		}
	}

	private void paintAsteroids(Graphics2D g) 
	{
		for(State.Asteroid a : currentState.asteroids)
		{
			g.setColor(Color.GRAY);
			paintBounds(g, a);
		}
	}

	private void paintShots(Graphics2D g) 
	{
		for(State.Shot s : currentState.shots)
		{
			int framesLeft = s.sinceFrame + 70 - currentState.frame;

			Color c = (s.myShot ? Color.GREEN : Color.ORANGE);
			if(framesLeft > 0)
			{
				g.setColor(c.darker().darker().darker());
				
				for(int x = -1; x <= 1; x++)
				{
					for(int y = -1; y <= 1; y++)
					{
						int ax = (int) (s.x + x * 1024 + 0.5);
						int ay = (int) (896.5 - s.y + y * 768);
						int bx = (int) (ax + framesLeft * s.vx + 0.5);
						int by = (int) (ay - framesLeft * s.vy + 0.5);
						g.drawLine(ax, ay, bx, by);
					}
				}
			}
			
			g.setColor(c);
			g.drawLine((int) (s.x + 0.5), (int) (896.5 - s.y), 
					(int) (s.x + s.vx + 0.5), 
					(int) (896.5 - (s.y + s.vy)));
		}
	}

	private void paintShotsLaunched(Graphics2D g)
	{
		for(State.Shot s : currentState.shotsLaunched)
		{
			double frames = s.hitAtFrame - s.sinceFrame;

			if(frames > 0)
			{
				g.setColor(Color.RED.darker().darker().darker());
				
				for(int x = -1; x <= 1; x++)
				{
					for(int y = -1; y <= 1; y++)
					{
						int ax = (int) (s.shipX + x * 1024 + 0.5);
						int ay = (int) (896.5 - s.shipY + y * 768);
						int bx = (int) (ax + frames * s.vx + 0.5);
						int by = (int) (ay - frames * s.vy + 0.5);
						g.drawLine(ax, ay, bx, by);
						
						int cx = (int) (s.x + x * 1024 + 0.5);
						int cy = (int) (896.5 - s.y + y * 768);
						
						int dx = (int) (s.vx / 2 + 0.5);
						int dy = (int) (s.vy / 2 + 0.5);
						
						g.drawLine(cx - dy, cy - dx, cx + dy, cy + dx);
					}
				}
			}
		}
	}

	private void paintUfo(Graphics2D g) 
	{
		g.setColor(Color.ORANGE);
		paintBounds(g, currentState.ufo);
	}
	
	private void paintBounds(Graphics2D g, State.GameObject gObj)
	{
		if(gObj == null)
		{
			return;
		}
		
		Color c = g.getColor();
		
		Color colorInner = Color.BLACK;
		Color colorOuter = Color.BLACK;
		if(gObj.nextAttacked)
		{
			colorInner = Color.RED;
		}
		else if(gObj.attacked)
		{
			colorInner = Color.YELLOW;
		}
		else if(gObj.willBeHit)
		{
			colorInner = Color.GREEN;
		}
		
		if(gObj.collidesNext)
		{
			colorOuter = Color.RED;
		}
		else if(gObj.collides)
		{
			colorOuter = Color.blue;
		}
		else if(gObj.closest)
		{
			colorOuter = Color.WHITE;
		}

		int rs = (int) gObj.getRadius(true);
		int rb = (int) gObj.getRadius(false);
		
		g.setColor(colorOuter.darker().darker());
		g.fillOval((int) (gObj.x - rb + 0.5), (int) (896.5 - gObj.y - rb), rb << 1, rb << 1);
		g.setColor(colorInner.darker().darker());
		g.fillOval((int) (gObj.x - rs + 0.5), (int) (896.5 - gObj.y - rs), rs << 1, rs << 1);

		g.setColor(c.darker().darker().darker());
		g.drawLine((int) (gObj.x + 0.5), (int) (896.5 - gObj.y), 
				(int) (gObj.x + 60 * gObj.vx), 
				896 - (int) (gObj.y + 60 * gObj.vy));

		g.setColor(c);
		
		g.drawOval((int) (gObj.x - rs + 0.5), (int) (896.5 - gObj.y - rs), rs << 1, rs << 1);
		g.drawOval((int) (gObj.x - rb + 0.5), (int) (896.5 - gObj.y - rb), rb << 1, rb << 1);
		
		g.setColor(new Color(255, 255, 255, 64));
		g.drawString("" + gObj.id, (int) gObj.x, (int) (896.5 - gObj.y));
	}
}
