import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Vector;

public class State implements Serializable
{
	private static final long serialVersionUID = 1L;

	static int idNext = 1;
	
	static int[] digits = new int[] { 0xADD, 0xB2E, 0xB32, 0xB3A, 0xB41, 0xB48, 0xB4F, 0xB56, 0xB5B, 0xB63 };
		
	static class GameObject implements Serializable, Cloneable
	{
		private static final long serialVersionUID = 1L;
		
		int id = -1;
		int sinceFrame;
		double x;
		double y;
		double pathX = 0;
		double pathY = 0;
		double vx = 0;
		double vy = 0;
		boolean willBeHit = false;
		boolean attacked = false;
		boolean nextAttacked = false;
		boolean collides = false;
		boolean collidesNext = false;
		boolean closest = false;
		
		GameObject(int frame, double x, double y)
		{
			this.sinceFrame = frame;
			this.x = x;
			this.y = y;
		}
		
		public boolean matches(GameObject go)
		{
			return true;
		}

		public double getRadius(boolean small) 
		{
			return 0.0;
		}
		
		public void match(int frame, GameObject go)
		{
			id = go.id;
			sinceFrame = go.sinceFrame;
			pathX = go.pathX + calcDiffX(go.x, x);
			pathY = go.pathY + calcDiffY(go.y, y);
			
//			if(!isSpeedFixed(frame))
//			{
				vx = pathX / (frame - sinceFrame);
				vy = pathY / (frame - sinceFrame);
//			}
//			else
//			{
//				vx = go.vx;
//				vy = go.vy;
//			}
		}

		public boolean isSpeedFixed(int frame)
		{
			return frame - sinceFrame > 8;
		}
		
		@Override
		protected Object clone() throws CloneNotSupportedException 
		{
			return super.clone();
		}
	}
	
	static class Asteroid extends GameObject
	{
		private static final long serialVersionUID = 1L;
		
		int type;
		int size;
		
		Asteroid(int frame, double x, double y, int type, int size)
		{
			super(frame, x, y);
			this.type = type;
			this.size = (size == 0 ? 0 : size - 16);
		}
		
		public String toString()
		{
			return "Asteroid: id=" + id + " P:" + x + "/" + y 
				+ " V:" + vx + "/" + vy + " T:" + type + " S:" + size;
		}

		@Override
		public boolean matches(GameObject go) 
		{
			Asteroid a = (Asteroid) go;
			if(a.type != type || a.size != size)
			{
				return false;
			}
			return super.matches(go);
		}

		@Override
		public double getRadius(boolean small) 
		{
			return (small ? 32 : 40) >> -size;
		}
	}
	
	static class Shot extends GameObject implements Cloneable
	{
		private static final long serialVersionUID = 1L;

		double hitAtFrame = -1;
		boolean myShot = false;
		int shipAngle = -1;
		double shipX;
		double shipY;
		int shotStartX;
		int shotStartY;
		int shotX8;
		int shotY8;
		
		Shot(int frame, double x, double y)
		{
			super(frame, x, y);
			shipX = x;
			shipY = y;
		}

		public String toString()
		{
			return "Shot: id=" + id + " P:" + x + "/" + y 
			+ " V:" + vx + "/" + vy;
		}

		@Override
		public void match(int frame, GameObject go) 
		{
			super.match(frame, go);
			myShot = ((Shot) go).myShot;
			shotStartX = ((Shot) go).shotStartX;
			shotStartY = ((Shot) go).shotStartY;
			shotX8 = ((Shot) go).shotX8;
			shotY8 = ((Shot) go).shotY8;
			shipX = ((Shot) go).shipX;
			shipY = ((Shot) go).shipY;
			shipAngle = ((Shot) go).shipAngle;
			
			if(frame - sinceFrame == 8)
			{
				shotX8 = (int) x;
				shotY8 = (int) y;
			}
			if(frame - sinceFrame == 16)
			{
				vx = calcDiffX(shotX8, x) / 8.0;
				vy = calcDiffY(shotY8, y) / 8.0;
			}
			else if(isSpeedFixed(frame))
			{
				vx = go.vx;
				vy = go.vy;
			}
		}

		
		@Override
		public boolean isSpeedFixed(int frame) 
		{
			return frame - sinceFrame > 16;
		}

		@Override
		protected Object clone() throws CloneNotSupportedException 
		{
			return super.clone();
		}
	}
	
	static class Ufo extends GameObject
	{
		private static final long serialVersionUID = 1L;
		
		int size;
		
		Ufo(int frame, double x, double y, int size)
		{
			super(frame, x, y);
			this.size = (size == 0 ? 0 : size - 16);
		}
		
		@Override
		public double getRadius(boolean small) 
		{
			return (small ? 30 : 40) >> -size;
		}
		
		@Override
		public void match(int frame, GameObject go) 
		{
			super.match(frame, go);

			vx = x - go.x;
			vy = y - go.y;
		}
	}
	
	static class Ship extends GameObject
	{
		private static final long serialVersionUID = 1L;
		
		int dx;
		int dy;
		
		int angle;
		int angleCandidates = 0;
		
		Ship(int frame, double x, double y, int dx, int dy)
		{
			super(frame, x, y);
			this.dx = dx;
			this.dy = dy;
		}

		@Override
		public double getRadius(boolean small) 
		{
			return 20;
		}

		@Override
		public void match(int frame, GameObject go) 
		{
			super.match(frame, go);

			vx = x - go.x;
			vy = y - go.y;
		}
	}
	
	Vector<Asteroid> asteroids = new Vector<Asteroid>();
	Vector<Shot> shots = new Vector<Shot>();
	Vector<Shot> shotsLaunched = new Vector<Shot>();
	Ufo ufo = null;
	Ship ship = null;
	int myShotCnt;

	int score = 0;
	long timeGameStarted = 0;
	long currentTime = System.currentTimeMillis();

	byte keys = 0;
	
	int ships = 0;
	
	static int vectorRam[] = new int[512];
	
	int frameByte = 0;
	int pingByte = 0;
	int frame = 0;
	int latency = 0;
	
	// statistics
	int shotsFired = 0;
	int shotsMissed = 0;
	int shotsHit = 0;
	int sumFramesToHit = 0;
	int sumShotsActive = 0;
	int framesActive = 0;
	int asteroidsDestroyedSmall = 0;
	int asteroidsDestroyedMedium = 0;
	int asteroidsDestroyedLarge = 0;
	int ufosDestroyedSmall = 0;
	int ufosDestroyedLarge = 0;
	int shipsLost = 0;
	int jumps = 0;
	
	void fromBytes(byte[] b, State stateOld, int pingByteSent)
	{
		asteroids.clear();
		shots.clear();
		ufo = null;
		ship = null;
		
		frameByte = (b[1024] + 256) & 0xFF;
		pingByte = (b[1025] + 256) & 0xFF;
		latency = (pingByteSent - pingByte + 257) & 0xFF;
		if(stateOld != null)
		{
			frame = stateOld.frame + ((frameByte - stateOld.frameByte + 256) & 0xFF);
		}
		
		for(int r = 0, w = 0; w < 512; r += 2, w++) 
		{
			vectorRam[w] = 
				(toInt(b[r])) | (toInt(b[r + 1]) << 8);
		}
		
		if (vectorRam[0] != 0xe001 && vectorRam[0] != 0xe201) 
		{
System.out.println("Error: does not start with JMPL!");			
			return;
		}

		int scaleFactor = 0;
		int posX = 0;
		int posY = 0;
		int v1x = 0;
		int v1y = 0;
		int brightness;
		int shipdetect = 0;
		
		int pc = 1;
loop:		
		while(pc < 511)
		{
			int opcode = vectorRam[pc] >> 12;
			switch(opcode)
			{
			case 0xa:	// LABS
				posY = vectorRam[pc] & 0x3ff;
				posX = vectorRam[pc + 1] & 0x3ff;
				scaleFactor = vectorRam[pc + 1] >> 12;
				break;
				
			case 0xb:	// HALT
				break loop;
				
			case 0xc:	// JSRL
				int sr = vectorRam[pc] & 0xfff;
				switch (sr)
				{
				case 0x8f3:
					asteroids.add(new Asteroid(frame, posX, posY, 1, scaleFactor));
					break;
					
				case 0x8ff:
					asteroids.add(new Asteroid(frame, posX, posY, 2, scaleFactor));
					break;
					
				case 0x90d:
					asteroids.add(new Asteroid(frame, posX, posY, 3, scaleFactor));
					break;
					
				case 0x91a:
					asteroids.add(new Asteroid(frame, posX, posY, 4, scaleFactor));
					break;
					
				case 0x929:
					ufo = new Ufo(frame, posX, posY, scaleFactor);
					break;
					
				case 0xA6D:
					ships++;
					break;
				}
				
				if(posX == 100 && posY == 876 && scaleFactor == 1)
				{
					score *= 10;
					
					for(int i = 0; i <= 9; i++)
					{
						if(sr == digits[i])
						{
							score += i;
						}
					}
				}
				
				break;
				
			case 0xd:	// RTSL
				break loop;

			case 0xe:	// JMPL
				break loop;
				
			case 0xf:	// SVEC
				break;
				
			default:	// VCTR
				
				int dy = vectorRam[pc] & 0x3ff;
				if ((vectorRam[pc] & 0x400) != 0)
					dy = -dy;
				
				int dx = vectorRam[pc+1] & 0x3ff;
				if ((vectorRam[pc+1] & 0x400) != 0)
					dx = -dx;
				
				brightness = vectorRam[pc+1] >> 12;
				
				if(dx == 0 && dy == 0 && brightness == 15)
				{
					shots.add(new Shot(frame, posX, posY));
				}
				
				if(opcode == 6 && brightness == 12 && dx != 0 && dy != 0)
				{
					switch (shipdetect)
					{
					case 0:
						v1x = dx;
						v1y = dy;
						++shipdetect;
						break;
						
					case 1:
						ship = new Ship(frame, posX, posY, v1x - dx, v1y - dy);
						++shipdetect;
						break;
					}
				}
				else if(shipdetect == 1)
				{
					shipdetect = 0;
				}
			}
			
			pc++;
			if(opcode <= 0x0A)
				pc++;
		}
	}
	
	int toInt(byte b)
	{
		return b >= 0 ? b : b + 256;
	}
	
	public String toString()
	{
		StringBuilder sb = new StringBuilder();
		sb.append("*** time: " + System.currentTimeMillis() + "\n");
		for(Asteroid a : asteroids)
		{
			sb.append(a + "\n");
		}
		for(Shot s : shots)
		{
			sb.append(s + "\n");
		}
		return sb.toString();
	}
	
	public void match(State stateOld)
	{
		if(stateOld == null)
		{
			for(Asteroid a : asteroids)
			{
				a.id = idNext++;
			}
			for(Shot s : shots)
			{
				s.id = idNext++;
			}
			if(ship != null)
			{
				ship.id = idNext++;
			}
			if(ufo != null)
			{
				ufo.id = idNext++;
			}
			return;
		}
		
		timeGameStarted = stateOld.timeGameStarted;
		shipsLost = stateOld.shipsLost;
		framesActive = stateOld.framesActive;
		if(score == 0 && ships > 0 && stateOld.ships == 0)
		{
			timeGameStarted = System.currentTimeMillis();
			shipsLost = 0;
			framesActive = 0;
		}
		else if(score < 1000 && stateOld.score % 100000 > 99000)
		{
			int s = score;
			score = stateOld.score - stateOld.score % 100000 + 100000 + s;
		}
		else
		{
			int s = score;
			score = stateOld.score - stateOld.score % 100000 + s;
		}

		for(Shot s : stateOld.shotsLaunched)
		{
			try
			{
				shotsLaunched.add((Shot) s.clone());
			}
			catch(Exception exc)
			{
			}
		}
		for(int i = shotsLaunched.size() - 1; i >= 0; i--)
		{
			Shot sl = shotsLaunched.get(i);
			if(sl.hitAtFrame < frame
					|| (sl.sinceFrame < frame - 3 && sl.id < 0))
			{
				shotsLaunched.remove(i);
			}
		}
		
		HashMap<Integer, Integer> oldToNewAst = new HashMap<Integer, Integer>();
		HashMap<Integer, Integer> newToOldAst = new HashMap<Integer, Integer>();
		
		matchObjects(stateOld, oldToNewAst, newToOldAst, asteroids, stateOld.asteroids);
		setVelocity(newToOldAst, asteroids, stateOld.asteroids);
		
		HashMap<Integer, Integer> oldToNewShot = new HashMap<Integer, Integer>();
		HashMap<Integer, Integer> newToOldShot = new HashMap<Integer, Integer>();
		
		matchObjects(stateOld, oldToNewShot, newToOldShot, shots, stateOld.shots);
		setVelocity(newToOldShot, shots, stateOld.shots);
		
		matchShotsLaunched();
		
		if(ship != null && stateOld.ship != null)
		{
			ship.match(frame, stateOld.ship);
		}
		if(ufo != null && stateOld.ufo != null)
		{
			ufo.match(frame, stateOld.ufo);
		}
		
		for(Shot s : shots)
		{
			if(s.myShot)
			{
				myShotCnt++;
			}
		}
		for(Shot sl : shotsLaunched)
		{
			if(sl.id < 0)
			{
				myShotCnt++;
			}
		}
		
		shotsFired = stateOld.shotsFired;
		shotsHit = stateOld.shotsHit;
		shotsMissed = stateOld.shotsMissed;
		asteroidsDestroyedLarge = stateOld.asteroidsDestroyedLarge;
		asteroidsDestroyedMedium = stateOld.asteroidsDestroyedMedium;
		asteroidsDestroyedSmall = stateOld.asteroidsDestroyedSmall;
		ufosDestroyedLarge = stateOld.ufosDestroyedLarge;
		ufosDestroyedSmall = stateOld.ufosDestroyedSmall;
		sumFramesToHit = stateOld.sumFramesToHit;
		sumShotsActive = stateOld.sumShotsActive;
		jumps = stateOld.jumps;
		
		// match missing shots to missing objects
		HashSet<Shot> shotsLost = new HashSet<Shot>();
		for(int i = 0; i < stateOld.shots.size(); i++)
		{
			Shot s = stateOld.shots.get(i);
			if(oldToNewShot.get(i) == null && s.myShot)
			{
				shotsLost.add(s);
				shotsFired++;
			}
		}
		
		HashSet<GameObject> goLost = new HashSet<GameObject>();
		if(stateOld.ufo != null && ufo == null)
		{
			goLost.add(stateOld.ufo);
		}
		for(int i = 0; i < stateOld.asteroids.size(); i++)
		{
			Asteroid a = stateOld.asteroids.get(i);
			if(oldToNewAst.get(i) == null)
			{
				goLost.add(a);
			}
		}

		while(true)
		{
			double distMin = Double.MAX_VALUE;
			Shot sBest = null;
			GameObject goBest = null;
			for(Shot s : shotsLost)
			{
				for(GameObject go : goLost)
				{
					double dist = Math.abs((go.getRadius(true) + go.getRadius(false)) / 2.0 
											- calcDist(go, s));
					if(dist < distMin)
					{
						distMin = dist;
						sBest = s;
						goBest = go;
					}
				}
			}
			
			if(distMin < 30)
			{
				if(goBest instanceof Ufo)
				{
					if(((Ufo) goBest).size == -1)
					{
						ufosDestroyedLarge++;
					}
					else
					{
						ufosDestroyedSmall++;
					}
				}
				else
				{
					switch(((Asteroid) goBest).size)
					{
					case 0:
						asteroidsDestroyedLarge++;
						break;
						
					case -1:
						asteroidsDestroyedMedium++;
						break;
						
					case -2:
						asteroidsDestroyedSmall++;
					}
				}
				
				shotsHit++;
				sumFramesToHit += stateOld.frame - sBest.sinceFrame;
				
				shotsLost.remove(sBest);
				goLost.remove(goBest);

				continue;
			}
			
			break;
		}
		
		shotsMissed += shotsLost.size();
		
		if(asteroids.size() > 2 && ship != null)
		{
			sumShotsActive += myShotCnt;
			framesActive++;
		}
		
		if(ships < stateOld.ships)
		{
			shipsLost++;
		}
	}

	private void setVelocity(HashMap<Integer, Integer> newToOld,
			Vector<? extends GameObject> gameObjectsNew,
			Vector<? extends GameObject> gameObjectsOld)
	{
		for(int iNew = 0; iNew < gameObjectsNew.size(); iNew++)
		{
			GameObject goNew = gameObjectsNew.get(iNew);

			if(newToOld.containsKey(iNew))
			{
				int iOld = newToOld.get(iNew);
				
				GameObject goOld = gameObjectsOld.get(iOld);

				goNew.match(frame, goOld);
			}
			else
			{
				goNew.id = idNext++;
				goNew.sinceFrame = frame;
			}
		}
	}

	private void matchObjects(State stateOld,
			HashMap<Integer, Integer> oldToNew,
			HashMap<Integer, Integer> newToOld,
			Vector<? extends GameObject> gameObjectsNew,
			Vector<? extends GameObject> gameObjectsOld) 
	{
		while(true)
		{
			double distMin = 20.0;	// TODO nderung bei Latenz != 1
			int bestOld = -1;
			int bestNew = -1;
			for(int iOld = 0; iOld < gameObjectsOld.size(); iOld++)
			{
				GameObject goOld = gameObjectsOld.get(iOld);
				if(goOld == null || oldToNew.containsKey(iOld))
				{
					continue;
				}
				
				for(int iNew = 0; iNew < gameObjectsNew.size(); iNew++)
				{
					GameObject goNew = gameObjectsNew.get(iNew);
					if(goNew == null || newToOld.containsKey(iNew)
							|| !goNew.matches(goOld))
					{
						continue;
					}
					
					double dist = calcDist(goOld, goNew);
					if(dist < distMin)
					{
						distMin = dist;
						bestOld = iOld;
						bestNew = iNew;
					}
				}
			}
			
			if(bestOld == -1)
			{
				break;
			}
			
			oldToNew.put(bestOld, bestNew);
			newToOld.put(bestNew, bestOld);
		}
	}
	
	private void matchShotsLaunched()
	{
		for(Shot sl : shotsLaunched)
		{
			int framesSinceShot = frame - sl.sinceFrame;
			sl.x = sl.shipX + framesSinceShot * sl.vx;
			sl.x -= Math.floor(sl.x / 1024) * 1024;
			sl.y = sl.shipY + framesSinceShot * sl.vy;
			sl.y -= Math.floor((sl.y - 128) / 768) * 768;
		}
		
		for(Shot s : shots)
		{
			if(s.myShot || s.sinceFrame < frame - 10)	 // TODO genauer, Latenz?
			{
				continue;
			}
			
			for(Shot sl : shotsLaunched)
			{
				if(sl.id >= 0)
				{
					continue;
				}
				
				double dvx = s.vx - sl.vx;
				double dvy = s.vy - sl.vy;
				
				if(dvx * dvx + dvy * dvy > 8)
				{
					continue;
				}
				
				double dx = s.x - sl.x; 
				double dy = s.y - sl.y;
				
				if(dx * dx + dy * dy > 100)
				{
					continue;
				}
				
				s.myShot = true;
				s.sinceFrame = sl.sinceFrame;
				s.pathX = s.x - sl.x;
				s.pathY = s.y - sl.y;
				s.shipAngle = sl.shipAngle;
				s.shotStartX = (int) s.x;
				s.shotStartY = (int) s.y;
				s.shipX = (int) sl.shipX;
				s.shipY = (int) sl.shipY;
				sl.id = s.id;
			}
		}
		
		// remove launched shots, which have an assigned ID and do not exist
		// any longer
		for(int i = shotsLaunched.size() - 1; i >= 0; i--)
		{
			Shot sl = shotsLaunched.get(i);
			if(sl.id < 0)
			{
				continue;
			}
			
			boolean found = false;
			for(Shot s : shots)
			{
				if(sl.id == s.id)
				{
					found = true;
					break;
				}
			}
			
			if(!found)
			{
				shotsLaunched.remove(i);
			}
		}
	}
	
	public double calcDist(GameObject objOld, GameObject objNew)
	{
		double px = objOld.x + objOld.vx;
		double py = objOld.y + objOld.vy;
		
		double dx = calcDiffX(objNew.x, px);
		double dy = calcDiffY(objNew.y, py);
		
		return Math.sqrt(dx * dx + dy * dy);
	}
	
	static double calcDiffX(double x1, double x2)
	{
		double dx = x2 - x1;
		
		if(dx < -512)
		{
			dx += 1024;
		}
		if(dx > 512)
		{
			dx -= 1024;
		}
		
		return dx;
	}
	
	static double calcDiffY(double y1, double y2)
	{
		double dy = y2 - y1;
		
		if(dy < -384)
		{
			dy += 768;
		}
		if(dy > 384)
		{
			dy -= 768;
		}
		
		return dy;
	}
}
