package de.heise.anniversary.contest.asteroids;

import java.net.DatagramPacket;

/**
 * The <code>KeysPacket</code> class represents the client reply packet.
 * <p>
 * It contains the id-prefix, the pressed keys and the ping value.
 *
 * @see ServerConnection
 */
public class KeysPacket {
	public static final byte KEY_HYPERSPACE = 1;
	public static final byte KEY_FIRE = 2;
	public static final byte KEY_THRUST = 4;
	public static final byte KEY_RIGHT = 8;
	public static final byte KEY_LEFT = 0x10;

	private byte[] payload = new byte[8];
	private DatagramPacket datagramPacket;

	/**
	 * Initializes a new <code>KeysPacket</code>.
	 */
	public KeysPacket() {
		datagramPacket = new DatagramPacket(payload, payload.length);
		payload[0] = 'c';
		payload[1] = 't';
		payload[2] = 'm';
		payload[3] = 'a';
		payload[4] = 'm';
		payload[5] = 'e';
		payload[6] = '@';
		payload[7] = 0;
	}

	/**
	 * Resets the key state for this <code>KeysPacket</code>.
	 */
	public void clear() {
		payload[6] = '@';
	}

	/**
	 * Sets the hyperspace-key state.
	 * @param pressed
	 * 			<code>true</code> for pressed hyperspace-key
	 */
	public void hyperspace(boolean pressed) {
		if (pressed)
			payload[6] |= KEY_HYPERSPACE;
		else
			payload[6] &= ~KEY_HYPERSPACE;
	}

	/**
	 * Sets the fire-key state.
	 * @param pressed
	 * 			<code>true</code> for pressed fire-key
	 */
	public void fire(boolean pressed) {
		if (pressed)
			payload[6] |= KEY_FIRE;
		else
			payload[6] &= ~KEY_FIRE;
	}

	/**
	 * Sets the thrust-key state.
	 * @param pressed
	 * 			<code>true</code> for pressed thrust-key
	 */
	public void thrust(boolean pressed) {
		if (pressed)
			payload[6] |= KEY_THRUST;
		else
			payload[6] &= ~KEY_THRUST;
	}

	/**
	 * Sets the left-key state.
	 *
	 * A pressed left-key resets the right-key.
	 *
	 * @param pressed
	 * 			<code>true</code> for pressed left-key
	 */
	public void left(boolean pressed) {
		if (pressed) {
			payload[6] |= KEY_LEFT;
			right(false);
		} else
			payload[6] &= ~KEY_LEFT;
	}

	/**
	 * Sets the right-key state.
	 *
	 * A pressed right-key resets the left-key.

	 * @param pressed
	 * 			<code>true</code> for pressed right-key
	 */
	public void right(boolean pressed) {
		if (pressed) {
			payload[6] |= KEY_RIGHT;
			left(false);
		} else
			payload[6] &= ~KEY_RIGHT;
	}

	/**
	 * Returns ping value.
	 *
	 * @return
	 * 		the ping value
	 */
	public byte getPing() {
		return payload[7];
	}

	/**
	 * Sets ping value.
	 *
	 * The ping value from this <code>KeysPacket</code>.
	 * Use this to detect network latency.
	 *
	 * @param ping
	 * 			the ping value
	 *
	 * @see FramePacket#getPing()
	 */
	public void setPing(byte ping) {
		payload[7] = ping;
	}

	/**
	 * Returns the <code>DatagramPacket</code> for sending the network data
	 *
	 * @return the <code>DatagramPacket</code>
	 *
	 * @see ServerConnection
	 */
	public DatagramPacket getDatagramPacket() {
		return datagramPacket;
	}
	
	
	
	public byte getKeyByte()
	{
	  return payload[6];
	}
}
