/*
created: Jun 17, 2008  Gereon Fassbender

$Revision$
$Date$
$Log$
*/

package asteroids;

import java.util.*;

import de.heise.anniversary.contest.asteroids.*;



public class TargetMonitor
{
  private final static int POS_TOLERANCE = 3;
  
  private List<Target> targets = new ArrayList<Target>();
  private boolean ufo;
  
  
  
  private static Target addNewTarget(List<Target> list,
                                     Position pos, TargetKind kind, Asteroid subject)
  {
    Target t = new Target(kind, subject);
    t.setPosition(pos.getX(), pos.getY());
    list.add(t);
    return t;
  }
  
  
  private void addTarget(Position pos, TargetKind kind, Asteroid subject)
  {
    addNewTarget(targets, pos, kind, subject);
  }
  
  
  private Target findAsteroid(Asteroid a, TargetKind kind)
  {
    for (Target t : targets) {
      if (t.getKind() == kind &&
          t.getAsteroidType() == a.getType() &&
          t.getEstimationAccuracy() > 0) {
        int x = t.getX() + Math.round(t.getSpeedX());
        int y = t.getY() + Math.round(t.getSpeedY());
        if (Math.abs(a.getX() - x) <= POS_TOLERANCE &&
            Math.abs(a.getY() - y) <= POS_TOLERANCE) {
          return t;
        }
      }
    }
    
    return null;
  }
  
  
  private void reset(GameStatus game)
  {
    List<Target> newTargets = new ArrayList<Target>();
    List<Asteroid> asteroids = game.getAsteroids();
    
    for (Asteroid a : asteroids) {
      TargetKind kind = TargetKind.getAsteroidKind(a);
      Target t = findAsteroid(a, kind);
      if (t != null) {
        newTargets.add(t);
      }
      else {
        t = addNewTarget(newTargets, a, kind, a);
        //System.out.println("new asteroid: " + t.getKind());
      }
    }
    
    assert newTargets.size() == asteroids.size();
    
    if (game.getSaucer() != null) {
      if (ufo) {
        Target old = targets.get(targets.size() - 1);
        assert old.getKind().isUfo();
        newTargets.add(old);
      }
      else {
        TargetKind kind = TargetKind.getUfoKind(game.getSaucer());
        addNewTarget(newTargets, game.getSaucer(), kind, null);
      }
      ufo = true;
      assert newTargets.size() == asteroids.size() + 1;
    }
    else {
      ufo = false;
    }
    
    targets = newTargets;
    assert isConsistent(asteroids);
  }
  
  
  private boolean isConsistent(List<Asteroid> asteroids)
  {
    Iterator<Asteroid> it = asteroids.iterator();
    
    for (Target t : targets) {
      if (!t.getKind().isUfo()) {
        Asteroid a = it.next();
        if (t.getKind() != TargetKind.getAsteroidKind(a)) {
          return false;
        }
      }
    }
    
    return true;
  }
  
  
  private void updateDistances(GameStatus game)
  {
    Position ship = game.getShip();
    if (ship == null) {
      return;
    }
    
    for (Target t : targets) {
      t.updateDistance(ship);
    }
  }
  
  
  public void sync(GameStatus game) //List<Asteroid> list)
  {
    int sizeWithoutUfo = (ufo ? targets.size() - 1 : targets.size());
    List<Asteroid> asteroids = game.getAsteroids();
    
    if (asteroids.size() != sizeWithoutUfo) {
      reset(game);
    }
    else if (!isConsistent(asteroids)) {
      System.out.println("* not consistent *");
      reset(game);
    }
    else {
      int i = 0;
      for (Asteroid a : asteroids) {
        targets.get(i++).setPosition(a.getX(), a.getY());
      }
      
      if (ufo && game.getSaucer() != null) {
        targets.get(i).setPosition(game.getSaucer().getX(), game.getSaucer().getY());
      }
      
      if (ufo != (game.getSaucer() != null)) {
        if (ufo) {
          // remove UFO
          targets.remove(targets.size() - 1);
          ufo = false;
        }
        else {
          addTarget(game.getSaucer(), TargetKind.getUfoKind(game.getSaucer()), null);
          ufo = true;
          assert targets.size() == asteroids.size() + 1;
        }
      }
    }
    
    updateDistances(game);
  }
  
  
  public int getTargetCount()
  {
    return targets.size();
  }
  
  
  public int getTargetCount(TargetKind kind)
  {
    int count = 0;
    for (Target t : targets) {
      if (t.getKind() == kind) {
        count++;
      }
    }
    return count;
  }
  
  
  public boolean isUfo()
  {
    return ufo;
  }
  
  
  public List<Target> getTargets()
  {
    return targets;
  }

  
  public Target getTarget(int index)
  {
    return targets.get(index);
  }
  
  
  public boolean isAlive(Target target)
  {
    for (Target t: targets) {
      if (t == target) {
        return true;
      }
    }
    
    return false;
  }
  
  
  public Target getUfo()
  {
    assert ufo;
    Target t = targets.get(targets.size() - 1);
    assert t.getKind().isUfo();
    return t;
  }
  
  
  public int getTargetsOnCollisionCount()
  {
    int count = 0;
    for (Target t : targets) {
      if (t.isOnCollision()) {
        count++;
      }
    }
    return count;
  }
  
  
  public Target getTargetOnCollision(GameWorld world)
  {
    Target ct = null;
    
    for (Target t : targets) {
      if (t.isOnCollision() &&
          (ct == null || t.getEtcFramesLeft(world) < ct.getEtcFramesLeft(world))) {
        ct = t;
      }
    }
    
    return ct;
  }
}
