// ----------------------------------------------------------------------------
// 'The Sniper'  -  c't Asteroids bot  -  Thorsten Denhard, 2008
// ----------------------------------------------------------------------------

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <winsock2.h>

#include "helper.h"
#include "networkengine.h"
#include "viewer.h"

Viewer::Viewer (SOCKET socket, ADDRESS serverIP, bool batch) : 
  mSocket                    (socket), 
  mServerIP                  (serverIP),
  mBatch                     (batch),
  mFrame                     (),
  mGame                      (),
  mPrevFrame                 (0),
  mPrevFullFrameNumber       (0),
  mFrameDelay                (0),
  mFrameLoss                 (0),
  mLastSaucerFullFrameNumber (0),
  mNetworkEngine             (NULL)
{
  mNetworkEngine = new NetworkEngine (mSocket, mServerIP);

  if (true == mBatch)
  {
    this->scheduleAction (KeysPacket::START ());
  }
};

void 
Viewer::run ()
{
  while (true)
  {
    this->iteration ();
  }
}

void 
Viewer::iteration ()
{
  this->sendReceive       ();
  this->updateFrameStatus ();
  this->updateGameStatus  ();
  this->updateAction      ();

  // Special: propagate frame-persistent info set by player
  {
    int currentIndex = mGame.mCurrentFrameIndex;
    FrameStatus& currentFrame = mGame.mFrameStates[currentIndex];

    int i = 0;
    
    for (i=0; i<MAX_SCENE_OBJECTS; ++i)
    {
      currentFrame.mSceneObjects[i].mDestructFrame = mGame.mSceneObjects[i].mDestructFrame;
      currentFrame.mSceneObjects[i].mDestructID    = mGame.mSceneObjects[i].mDestructID;
      currentFrame.mSceneObjects[i].mDestructVX    = mGame.mSceneObjects[i].mDestructVX;
      currentFrame.mSceneObjects[i].mDestructVY    = mGame.mSceneObjects[i].mDestructVY;
    }
  }  
}

void 
Viewer::sendReceive ()
{
  mNetworkEngine->sendPacket           ();
  mNetworkEngine->receivePacket        ();  
  mNetworkEngine->getLatestFramePacket (mFrame, mFrameDelay);  
}

void 
Viewer::updateFrameStatus ()
{
  // Get situation for current frame into the next free slot
  
  mGame.mCurrentFrameIndex = (mGame.mCurrentFrameIndex + 1) % MAX_REMEMBERED_FRAMES;
  FrameStatus& currentFrame = mGame.mFrameStates[mGame.mCurrentFrameIndex];
  
  this->interpretScreen (mFrame, currentFrame);
}

void 
Viewer::updateGameStatus ()
{
  // --------------------------------------------------------------------------
  // Preliminaries
  // --------------------------------------------------------------------------

  int i = 0;
  int k = 0;

  int currentIndex = mGame.mCurrentFrameIndex;
  const FrameStatus& currentFrame = mGame.mFrameStates[currentIndex];

  mGame.FrameStatus::clear ();
    
  // --------------------------------------------------------------------------
  // Take static info from current frame
  // --------------------------------------------------------------------------
  
  {
    mGame.mShipPresent      = currentFrame.mShipPresent;
    mGame.mShipX            = currentFrame.mShipX;
    mGame.mShipY            = currentFrame.mShipY;
    mGame.mShipDX           = currentFrame.mShipDX;      
    mGame.mShipDY           = currentFrame.mShipDY;
    mGame.mTime             = currentFrame.mTime;
    mGame.mFullFrameNumber  = currentFrame.mFullFrameNumber;
    mGame.mSceneObjectCount = currentFrame.mSceneObjectCount;
    
    for (i=0; i<MAX_SCENE_OBJECTS; ++i)
    {
      mGame.mSceneObjects[i] = currentFrame.mSceneObjects[i];
    }    
  }

  // --------------------------------------------------------------------------
  // Determine ship velocity
  // --------------------------------------------------------------------------

  if (currentFrame.mShipPresent)
  {
    // Preliminaries

    double pointsX[MAX_REMEMBERED_FRAMES];
    double pointsY[MAX_REMEMBERED_FRAMES];
    double pointsT[MAX_REMEMBERED_FRAMES];
    int   pointCount = 0;
    
    // Check current and previously remembered frames

    pointsX[pointCount] = double (currentFrame.mShipX);
    pointsY[pointCount] = double (currentFrame.mShipY);
    pointsT[pointCount] = double (currentFrame.mTime);

    ++pointCount;
    
    for (i=1; i<MAX_REMEMBERED_FRAMES; ++i)
    {
      int prevIndex = (currentIndex - i + MAX_REMEMBERED_FRAMES) % MAX_REMEMBERED_FRAMES;
      const FrameStatus& prevFrame = mGame.mFrameStates[prevIndex];

      bool prevFrameValid = true;
      
      prevFrameValid &= (prevFrame.mTime >= 0.0f);
      prevFrameValid &= (prevFrame.mTime <  currentFrame.mTime);
      prevFrameValid &= (true == prevFrame.mShipPresent);
      
      if (true != prevFrameValid)
      {        
        break;
      }

      pointsX[pointCount] = double (prevFrame.mShipX);
      pointsY[pointCount] = double (prevFrame.mShipY);
      pointsT[pointCount] = double (prevFrame.mTime);

      // Wrap-around issues
     
      pointsX[pointCount] = wrapRel (pointsX[pointCount], pointsX[0], GAME_WIDTH);
      pointsY[pointCount] = wrapRel (pointsY[pointCount], pointsY[0], GAME_HEIGHT);

      ++pointCount;
    }

    // Regression

    {
      double meanX = 0.0f;
      double meanY = 0.0f;
      double meanT = 0.0f;
      
      for (i=0; i<pointCount; ++i)
      {
        meanX += pointsX[i];
        meanY += pointsY[i];
        meanT += pointsT[i];
      }

      meanX /= double (pointCount);
      meanY /= double (pointCount);
      meanT /= double (pointCount);

      double nomX = 0.0f;
      double denX = 0.0f;
      
      for (i=0; i<pointCount; ++i)
      {
        nomX += ((pointsT[i]-meanT) * (pointsX[i]-meanX));
        denX += ((pointsT[i]-meanT) * (pointsT[i]-meanT));
      }

      double nomY = 0.0f;
      double denY = 0.0f;
      
      for (i=0; i<pointCount; ++i)
      {
        nomY += ((pointsT[i]-meanT) * (pointsY[i]-meanY));
        denY += ((pointsT[i]-meanT) * (pointsT[i]-meanT));
      }
      
      if (denX > 0.0f &&
          denY > 0.0f)
      {
        mGame.mShipVelocityX = double (nomX / denX);
        mGame.mShipVelocityY = double (nomY / denY);
      }
    }
  }

  // --------------------------------------------------------------------------
  // Determine sceneobject velocities
  // --------------------------------------------------------------------------
  
  this->calculateVelocities (TYPE_ASTEROID_S, currentIndex);
  this->calculateVelocities (TYPE_ASTEROID_M, currentIndex);
  this->calculateVelocities (TYPE_ASTEROID_L, currentIndex);
  this->calculateVelocities (TYPE_SAUCER,     currentIndex);
  this->calculateVelocities (TYPE_SHOT,       currentIndex);

  // --------------------------------------------------------------------------
  // Account for roundtrip-time by moving objects according to delay
  // --------------------------------------------------------------------------

  {
    for (i=0; i<mGame.mSceneObjectCount; ++i)
    {
      if (mGame.mSceneObjects[i].mType == TYPE_NONE)
      {
        continue;
      }

      double x  = double (mGame.mSceneObjects[i].mX);
      double y  = double (mGame.mSceneObjects[i].mY);
      double vx = double (mGame.mSceneObjects[i].mVX);
      double vy = double (mGame.mSceneObjects[i].mVY);

      x += vx * double (mFrameDelay+mFrameLoss+1) / GAME_FPS;
      y += vy * double (mFrameDelay+mFrameLoss+1) / GAME_FPS;

      x = wrapAbs (x, GAME_WIDTH);
      y = wrapAbs (y, GAME_HEIGHT);
      
      mGame.mSceneObjects[i].mX = int (x);
      mGame.mSceneObjects[i].mY = int (y);
    }
  }
  
  // --------------------------------------------------------------------------
  // Update ship-relative positions
  // --------------------------------------------------------------------------

  if (true == mGame.mShipPresent)
  {  
    for (i=0; i<mGame.mSceneObjectCount; ++i)
    {
      mGame.mSceneObjects[i].mRelX = wrapRel (mGame.mSceneObjects[i].mX, mGame.mShipX, GAME_WIDTH);
      mGame.mSceneObjects[i].mRelY = wrapRel (mGame.mSceneObjects[i].mY, mGame.mShipY, GAME_HEIGHT);
    }
  }
  else
  {
    for (i=0; i<mGame.mSceneObjectCount; ++i)
    {
      mGame.mSceneObjects[i].mRelX = mGame.mSceneObjects[i].mX;
      mGame.mSceneObjects[i].mRelY = mGame.mSceneObjects[i].mY;
    }
  }
  
  // --------------------------------------------------------------------------
  // Update target positions (now and n frames into the future)
  // --------------------------------------------------------------------------
  
  for (k=0; k<TARGET_LOOKAHEAD; ++k)
  {
    for (i=0; i<mGame.mSceneObjectCount; ++i)
    {
      mGame.mSceneObjects[i].mTargetX[k]   = mGame.mSceneObjects[i].mX;
      mGame.mSceneObjects[i].mTargetY[k]   = mGame.mSceneObjects[i].mY;
      mGame.mSceneObjects[i].mHasTarget[k] = false;
      
      int type = mGame.mSceneObjects[i].mType;
      
      if (type != TYPE_ASTEROID_S &&
          type != TYPE_ASTEROID_M &&
          type != TYPE_ASTEROID_L &&
          type != TYPE_SAUCER)
      {
        continue;
      }

      if (true != mGame.mShipPresent)
      {
        continue;
      }

      double shipX = double (mGame.mShipX);
      double shipY = double (mGame.mShipY);

      double shipVX = double (mGame.mShipVelocityX);
      double shipVY = double (mGame.mShipVelocityY);
      
      double ox = double (mGame.mSceneObjects[i].mX);
      double oy = double (mGame.mSceneObjects[i].mY);

      double orx = double (mGame.mSceneObjects[i].mRelX);
      double ory = double (mGame.mSceneObjects[i].mRelY);

      double ovx = double (mGame.mSceneObjects[i].mVX);
      double ovy = double (mGame.mSceneObjects[i].mVY);
     
      ox = ox + ovx * double(k) / GAME_FPS;
      oy = oy + ovy * double(k) / GAME_FPS;
      
      orx = orx + ovx * double(k) / GAME_FPS;
      ory = ory + ovy * double(k) / GAME_FPS;
      
      double relVX = ovx - shipVX;
      double relVY = ovy - shipVY;
      
      // Warparound peculiarities: we must check both scenarios
      // and take the nearest result
      
      double tR         = 0.0f;
      double tA         = 0.0f;
      double shotAngleR = getShotAngle (orx, ory, relVX, relVY, shipX, shipY, SHOT_SPEED, tR);
      double shotAngleA = getShotAngle (ox,  oy,  relVX, relVY, shipX, shipY, SHOT_SPEED, tA);

      double tx = ox + tA * relVX;
      double ty = oy + tA * relVY;      

      double ttrx = orx + tR * relVX;
      double ttry = ory + tR * relVY;      

      double rdx = tx - shipX;
      double rdy = ty - shipY;
      double rs  = rdx*rdx + rdy*rdy;      
      
      double ttrdx = ttrx - shipX;
      double ttrdy = ttry - shipY;
      double ttrs  = ttrdx*ttrdx + ttrdy*ttrdy;

      double margin = 0.98f;
      
      if (rs <= ttrs)
      {
        // Check if target can in fact be reached before the shot would die
                
        if (rs < (margin*SHOT_DURATION*SHOT_SPEED)*(margin*SHOT_DURATION*SHOT_SPEED))
        {      
          mGame.mSceneObjects[i].mTargetX[k]   = int (tx);
          mGame.mSceneObjects[i].mTargetY[k]   = int (ty);
          mGame.mSceneObjects[i].mHasTarget[k] = true;
        }
      }
      else
      {
        // Check if target can in fact be reached before the shot would die
                
        if (ttrs < (margin*SHOT_DURATION*SHOT_SPEED)*(margin*SHOT_DURATION*SHOT_SPEED))
        {      
          mGame.mSceneObjects[i].mTargetX[k]   = int (ttrx);
          mGame.mSceneObjects[i].mTargetY[k]   = int (ttry);
          mGame.mSceneObjects[i].mHasTarget[k] = true;
        }
      }      
    }
  }
}

void 
Viewer::interpretScreen (const FramePacket &packet, FrameStatus& frame)
{ 
  frame.clear ();

  unsigned long int oldNum = mPrevFullFrameNumber;
  
  if (packet.mFrameNum > mPrevFrame)
  {
    mPrevFullFrameNumber   += (packet.mFrameNum - mPrevFrame);
    frame.mFullFrameNumber  = mPrevFullFrameNumber;
    mPrevFrame              = packet.mFrameNum;
  }
  else if (mPrevFrame - packet.mFrameNum > 128)
  {
    // Wrap-around
    mPrevFullFrameNumber   += (255 - (mPrevFrame - packet.mFrameNum) + 1);
    frame.mFullFrameNumber  = mPrevFullFrameNumber;
    mPrevFrame              = packet.mFrameNum;    
  }
  else
  {
    // Too many packet losses to keep track. Start counting again.
    mPrevFrame                 = packet.mFrameNum;
    mPrevFullFrameNumber       = mPrevFrame;
    frame.mFullFrameNumber     = mPrevFullFrameNumber;
    mLastSaucerFullFrameNumber = 0;
  }
  
  mFrameLoss  = frame.mFullFrameNumber - oldNum - 1;  
  frame.mTime = double (frame.mFullFrameNumber) / GAME_FPS;
    
  static const int OP_LABS = 0xa;
  static const int OP_HALT = 0xb;
  static const int OP_JSRL = 0xc;
  static const int OP_RTSL = 0xd;
  static const int OP_JMPL = 0xe;
  static const int OP_SVEC = 0xf;

  const unsigned short* ramP = (const unsigned short*) (packet.mVectorRAM);

  int dx         = 0;
  int dy         = 0;
  int sf         = 0;
  int vx         = 0;
  int vy         = 0;
  int vz         = 0;
  int vs         = 0;
  int v1x        = 0;
  int v1y        = 0;
  int shipDetect = 0;
  int pc         = 1;
  
  if ((unsigned char) (packet.mVectorRAM[1]) != 0xe0 && 
      (unsigned char) (packet.mVectorRAM[1]) != 0xe2)
  {
    // Sollte nicht vorkommen; erster Befehl ist immer ein JMPL
    return; 
  }
  
  while (true)
  {
    int op = ramP[pc] >> 12;

    if (op == OP_LABS)
    {
      vy = ramP[pc  ] & 0x3ff;
      vx = ramP[pc+1] & 0x3ff;
      vs = ramP[pc+1] >> 12;

      vy -= 128;
    }
    else if (op == OP_HALT)
    {
      return;
    }
    else if (op == OP_JSRL)
    {
      switch (ramP[pc] & 0xfff)
      {
        case 0x8f3:
        case 0x8ff:
        case 0x90d:
        case 0x91a:
          {
            frame.mSceneObjects[frame.mSceneObjectCount++].setAsteroid (vx, vy, vs);
          }
          break;
          
        case 0x929:
          {
            frame.mSceneObjects[frame.mSceneObjectCount++].setSaucer (vx, vy, vs);
            mLastSaucerFullFrameNumber = frame.mFullFrameNumber;
          }
          break;
      }  
    }
    else if (op == OP_RTSL)
    {
      return;
    }
    else if (op == OP_JMPL)
    {
      AST_NOP;
      return;
    }
    else if (op == OP_SVEC)
    {
      AST_NOP;
    }
    else
    {
      dx = ramP[pc+1] & 0x3ff;
      dy = ramP[pc  ] & 0x3ff;
      vz = ramP[pc+1] >> 12;
      sf = op;

      if ((ramP[pc] & 0x400) != 0)
      {
        dy = -dy;
      }

      if ((ramP[pc+1] & 0x400) != 0)
      {
        dx = -dx;
      }
      
      if (dx == 0 && dy == 0 && vz == 15)
      {
        frame.mSceneObjects[frame.mSceneObjectCount++].setShot (vx, vy);
      }

      if (op == 6  && 
          vz == 12 && 
          dx != 0  && 
          dy != 0)
      {
        switch (shipDetect)
        {
        case 0:
          {
            v1x = dx;
            v1y = dy;
            ++shipDetect;
          }
          break;

        case 1:
          {
            frame.mShipPresent = true;
            frame.mShipX       = vx;
            frame.mShipY       = vy;
            frame.mShipDX      = v1x - dx;
            frame.mShipDY      = v1y - dy;
            ++shipDetect;
          }
          break;
        }
      }
      else if (shipDetect == 1)
      {
        shipDetect = 0;
      }
    }

    if (op <= 0xa)
    {
      ++pc;
    }

    if (op != OP_JMPL) 
    {
      ++pc;
    }
  }   
}

void 
Viewer::calculateVelocities (int type, int currentIndex)
{
  int i = 0;
  int k = 0;

  FrameStatus& currentFrame = mGame.mFrameStates[currentIndex];
  
  int currentObjectIndexList[MAX_SCENE_OBJECTS];  
  int currentCount = currentFrame.findObjects (type, currentObjectIndexList);
  
  if (currentCount < 1)
  {
    return;
  }
  
  double pointsX    [MAX_SCENE_OBJECTS][MAX_REMEMBERED_FRAMES];
  double pointsY    [MAX_SCENE_OBJECTS][MAX_REMEMBERED_FRAMES];
  double pointsT    [MAX_SCENE_OBJECTS][MAX_REMEMBERED_FRAMES];
  int    pointCounts[MAX_SCENE_OBJECTS];
  
  int    prevListX   [MAX_SCENE_OBJECTS];
  int    prevListY   [MAX_SCENE_OBJECTS];
  double prevListT   [MAX_SCENE_OBJECTS];
  int    currentListX[MAX_SCENE_OBJECTS];
  int    currentListY[MAX_SCENE_OBJECTS];
  double currentListT[MAX_SCENE_OBJECTS];
  int    idList      [MAX_SCENE_OBJECTS];
  
  for (i=0; i<currentCount; ++i)
  {
    int objectIndex = currentObjectIndexList[i];
    
    pointCounts[i] = 0;

    pointsX[i][pointCounts[i]] = double (currentFrame.mSceneObjects[objectIndex].mX);
    pointsY[i][pointCounts[i]] = double (currentFrame.mSceneObjects[objectIndex].mY);
    pointsT[i][pointCounts[i]] = double (currentFrame.mTime);

    pointCounts[i] += 1;
  }
  
  for (i=0; i<currentCount; ++i)
  {
    int objectIndex = currentObjectIndexList[i];

    currentListX[i] = currentFrame.mSceneObjects[objectIndex].mX;
    currentListY[i] = currentFrame.mSceneObjects[objectIndex].mY;
    currentListT[i] = currentFrame.mTime;
  }

  // --------------------------------------------------------------------------  
  // Extrapolate speeds based on current frame and consecutive previous frames
  // --------------------------------------------------------------------------  

  const int  neededFrameCount = 5;
        int  neededCount      = currentCount;
        bool matchedOnce      = false;

  // Init idlist with 1-to-1 mapping

  for (k=0; k<neededCount; ++k)
  {
    idList[k] = k;
  }
 
  int lookBack = (type == TYPE_SAUCER) ? MAX_REMEMBERED_FRAMES_SAUCER : MAX_REMEMBERED_FRAMES;
  
  for (i=1; i<lookBack; ++i)
  {
    int prevIndex = (currentIndex - i + MAX_REMEMBERED_FRAMES) % MAX_REMEMBERED_FRAMES;
    const FrameStatus& prevFrame = mGame.mFrameStates[prevIndex];

    int prevObjectIndexList[MAX_SCENE_OBJECTS];  
    int prevCount = prevFrame.findObjects (type, prevObjectIndexList);
    
    bool prevFrameValid = true;

    prevFrameValid &= (prevFrame.mTime >= 0.0f);
    prevFrameValid &= (prevFrame.mTime <  currentFrame.mTime);
    prevFrameValid &= (prevCount       >  0);
    
    if (true != prevFrameValid)
    {
      // Never inspect frames before an invalid frame 
      // (chain is broken, things are likely to get bad)
      break;
    }

    // Init prevlists
    
    for (k=0; k<prevCount; ++k)
    {
      int objectIndex = prevObjectIndexList[k];
      
      prevListX[k] = prevFrame.mSceneObjects[objectIndex].mX;
      prevListY[k] = prevFrame.mSceneObjects[objectIndex].mY;
      prevListT[k] = prevFrame.mTime;
    }
      
    // Check counts
    
    bool sameCount = (prevCount == neededCount);

    if (true != sameCount)
    {
      // Count mismatch. Check if we can try nn-matching

      if (true == matchedOnce)
      {
        // We already did a nn-matching for some other frame.
        // More than once is no good, the info we have gathered so far must suffice...
        break;
      }

      if (i > neededFrameCount)
      {
        // We already have enough data, so we dont introduce an error-prone
        // matching, but are content with what we have...
        break;
      }
     
      // Try nn-matching

      for (k=0; k<currentCount; ++k)
      {
        idList[k] = -1;
      }

      double knownSpeed = (type == TYPE_SHOT) ? SHOT_SPEED : 0.0f;

      matchObjects (currentListX, currentListY, currentListT,
                    prevListX, prevListY, prevListT,
                    currentCount, prevCount, knownSpeed, idList);

      matchedOnce = true;
      neededCount = prevCount;
    }

    // Check for odd, buggy matchings: matches too far away 
    // from each other are to be ignored

    for (k=0; k<currentCount; ++k)
    {
      int prevID = idList[k];
      
      if (prevID == -1)
      {
        continue;
      }

      int prevObjectIndex = prevObjectIndexList[prevID];

      double xc = double (currentListX[k]);
      double yc = double (currentListY[k]);
      double tc = double (currentListT[k]);

      double xp = double (prevListX[prevID]);
      double yp = double (prevListY[prevID]);
      double tp = double (prevListT[prevID]);

      xp = wrapRel (xp, xc, GAME_WIDTH);
      yp = wrapRel (yp, yc, GAME_HEIGHT);

      double xd = xc - xp;
      double yd = yc - yp;
      double ds = xd*xd + yd*yd;

      double td       = fabs (tc - tp);
      double maxDist  = MAX_SPEED * td;
      double maxDistS = maxDist*maxDist;

      if (ds > maxDistS)
      {
        idList[k] = -1;
      }
    }

    // Add position information at prevFrame for all found objects

    for (k=0; k<currentCount; ++k)
    {
      int prevID = idList[k];
      
      if (prevID == -1)
      {
        continue;
      }
          
      int prevObjectIndex = prevObjectIndexList[prevID];

      // Special: propagate frame-persistent info set by player
      
      if (i == 1)
      {
        int objectIndex = currentObjectIndexList[k];        
        mGame.mSceneObjects[objectIndex].mDestructFrame = prevFrame.mSceneObjects[prevObjectIndex].mDestructFrame;
        mGame.mSceneObjects[objectIndex].mDestructID    = prevFrame.mSceneObjects[prevObjectIndex].mDestructID;
        mGame.mSceneObjects[objectIndex].mDestructVX    = prevFrame.mSceneObjects[prevObjectIndex].mDestructVX;
        mGame.mSceneObjects[objectIndex].mDestructVY    = prevFrame.mSceneObjects[prevObjectIndex].mDestructVY;
      }
      
      // Current frame will also be reference for possible further matching-process 

      currentListX[k] = prevListX[prevID];
      currentListY[k] = prevListY[prevID];
      currentListT[k] = prevListT[prevID];

      // Store data

      pointsX[k][pointCounts[k]] = double (prevFrame.mSceneObjects[prevObjectIndex].mX);
      pointsY[k][pointCounts[k]] = double (prevFrame.mSceneObjects[prevObjectIndex].mY);
      pointsT[k][pointCounts[k]] = double (prevFrame.mTime);

      // Wrap-around issues
      
      pointsX[k][pointCounts[k]] = wrapRel (pointsX[k][pointCounts[k]], pointsX[k][0], GAME_WIDTH);
      pointsY[k][pointCounts[k]] = wrapRel (pointsY[k][pointCounts[k]], pointsY[k][0], GAME_HEIGHT);

      pointCounts[k] += 1;        
    }
  }   

  // Regression

  for (k=0; k<currentCount; ++k)
  {
    const int& pc = pointCounts[k];
    
    double meanX = 0.0f;
    double meanY = 0.0f;
    double meanT = 0.0f;
    
    for (i=0; i<pc; ++i)
    {
      meanX += pointsX[k][i];
      meanY += pointsY[k][i];
      meanT += pointsT[k][i];
    }

    meanX /= double (pc);
    meanY /= double (pc);
    meanT /= double (pc);

    double nomX = 0.0f;
    double denX = 0.0f;
    
    for (i=0; i<pc; ++i)
    {
      nomX += double (((pointsT[k][i]-meanT) * (pointsX[k][i]-meanX)));
      denX += double (((pointsT[k][i]-meanT) * (pointsT[k][i]-meanT)));
    }

    double nomY = 0.0f;
    double denY = 0.0f;
    
    for (i=0; i<pc; ++i)
    {
      nomY += ((pointsT[k][i]-meanT) * (pointsY[k][i]-meanY));
      denY += ((pointsT[k][i]-meanT) * (pointsT[k][i]-meanT));
    }
    
    if (denX > 0.0f &&
        denY > 0.0f)
    {
      double vx  = double (nomX / denX);
      double vy  = double (nomY / denY);
      double vss = vx*vx + vy*vy;
                
      // Ignore odd, buggy speeds
     
      if (type == TYPE_SHOT)
      {
        if (vss > 0.25f*SHOT_SPEED*SHOT_SPEED &&
            vss < MAX_SPEED*MAX_SPEED)
        {
          int objectIndex = currentObjectIndexList[k];
      
          mGame.mSceneObjects[objectIndex].mVX = vx;
          mGame.mSceneObjects[objectIndex].mVY = vy;
        }
      }
      else
      {        
        if (vss < MAX_SPEED*MAX_SPEED)
        {
          int objectIndex = currentObjectIndexList[k];
      
          mGame.mSceneObjects[objectIndex].mVX = vx;
          mGame.mSceneObjects[objectIndex].mVY = vy;
        }
      } 
    }
  }
}

void 
Viewer::updateAction ()
{
  // To be implemented in subclass
  AST_NOP;
}

void 
Viewer::scheduleAction (const KeysPacket& keys)
{
  mNetworkEngine->scheduleAction (keys);
}

void 
Viewer::removeScheduledActions ()
{
  mNetworkEngine->removeScheduledActions ();
}

void 
Viewer::forceLeftRightBalance (unsigned char value)
{
  mNetworkEngine->forceLeftRightBalance (value);
}

unsigned char 
Viewer::getLeftRightBalance ()
{
  return mNetworkEngine->getLeftRightBalance ();
}

unsigned char 
Viewer::getLastLeftRightBalance ()
{
  return mNetworkEngine->getLastLeftRightBalance ();
}

