#ifndef INC_GAMEOBJ
#define INC_GAMEOBJ

#include "tools.h"

#define TIME_INFINITE (unsigned int)-1
#define OBJ_MAX_SPEED 8.0

enum gameObjTypeE { objTypeUnk=0,objTypeAstroidSmall=1,objTypeAstroidMed=2,objTypeAstroidLarge=3,objTypeSaucerSmall=4,objTypeSaucerLarge=5,objTypeShot=6,objTypeShip=7 };

/// this class hold the physical properties for each type
struct gameObjPropS {
	/// a single bit for each item ..
	int    bit;
	/// crash radius
	int		 radius;
	/// the known continues speed of an object (if this is 0 it is not contant)
	float speed;
	/// maximum live time ..
	unsigned int maxTime;
	/// maximum predict time (because they will change direction, this timer is <=maxTime)
	unsigned int maxPredictTime;
	/// bit field which object can kill us
	int   killBits;
	/// points per kill
	int   points;
};

extern gameObjPropS gameObjProp[objTypeShip+1];

///
struct rotationDataS {
	/// vector from ram
	int    dx,dy;
	/// angle in degree
	double angDegree;
};

#define MAX_ANGLE 256

extern rotationDataS rotationData[MAX_ANGLE];

inline int getHeadIdx(int idx)
{
	while (idx>=MAX_ANGLE) idx=idx-MAX_ANGLE;
	while (idx<0) idx=MAX_ANGLE+idx;
	return idx;
}

/// use the first as base and return the change index with sign
inline int getHeadDiff(int idx1,int idx2)
{
	int full=MAX_ANGLE;
	int half=full/2;
	
	idx1=getHeadIdx(idx1);
	idx2=getHeadIdx(idx2);

	int diff=idx2-idx1;
  // the shortest way is important so it cant be larger than halve ..
	if (abs(diff)>half)
	{
		diff=full-abs(diff);
		if (idx2>idx1)
			diff*=-1;
	}
	
	return diff;
}
///
class gameObjBaseC {
protected:
	/// type
	gameObjTypeE type;
	/// the position on which we detect the object the first time
	vecIntC      pos;
	/// marks if this is a good or bad object
	int          goodObject;
public:
	///
	gameObjBaseC() { type=objTypeUnk;goodObject=0;}
	///
	void set(const gameObjBaseC & sec) 
	{
		*this=sec;
	}
	/// set
	void set(gameObjTypeE t,const vecIntC & p) { type=t;pos=p;}
	///
	gameObjTypeE getType() const { return type;}
	///
	void setType(gameObjTypeE t) { type=t;}
	/// return the init start pos
	const vecIntC & getStartPos() const { return pos;}
	///
	int isGood() const { return goodObject;}	
	///
	void setGood() { goodObject=1;}
	void setBad()  { goodObject=0;}
};

/// class to present a to point system with time
class gameObjTimePointsC {

	struct pointDataS {
		float   t;
		vecDblC  p;
	};

	constFlowArrayT<32,pointDataS> points;	
	///
	float        predictTime;
	///
	vecDblC      predictPos;
	///
	vecDblC      head;
	///
	float        speed;
protected:
	/// set one value
	void set(unsigned int inT0,const vecIntC & inP0) 
	{ points.clear();
		pointDataS point;
	  point.t=inT0;
		point.p=convVec(inP0);
		points.add(point);
		speed=0;
		predictTime=0;
	}
	///
	void update(unsigned int inT1,const vecIntC & inP1);	
public:
	///
	gameObjTimePointsC() { speed=0.0;predictTime=0;}
	///
	vecDblC getPredictPos(float t) const;		
	/// if predict is possible use getSpeed to get current speed
	float getSpeed() const { return speed;}
	///		
	float getT0() const { return points.first().t;}
	float getT1() const { return points.last().t;}
	///
	const vecDblC & getP0() const { return points.first().p;}
	const vecDblC & getP1() const { return points.last().p;}	
	/// get heading as norm vector
	vecDblC getHead() const { return head;}
	/// get direction (including speed)
	vecDblC getDir() const { return head.norm(getSpeed());}
	

};

/// this is not a good c++ style but it is the best for performance
class gameObjC : public gameObjBaseC ,public gameObjTimePointsC{
	
	/// very important a unique key for each object
	unsigned int key;
	/// time on which the object appears
	unsigned int t0;
	/// time on which the object will die
	unsigned int tx;
	/// id of killer object
	unsigned int killerId;	
	/// count of how many tracks we do for this object
	int          trackCount;		
public:
	/// ctor
	gameObjC() { trackCount=0;t0=0;tx=TIME_INFINITE;killerId=TIME_INFINITE;}
	///
	unsigned int getTX() const { return tx;}
	///
	int getKillData(unsigned int & id,unsigned int & time) const
	{
		if (killerId==TIME_INFINITE) return 0;
		id=killerId;
		time=tx;
		return 1;
	}
	/// return if the object could be predicted to the given time
	int predict(unsigned int t) const { return trackCount!=0 && alive(t);}
	/// alive at given time index
	int alive(unsigned int t) const { return t>=t0 && t<tx;}
	/// return the left livetime for this object
	unsigned int getLiveTime(unsigned int t) const
	{
		if (killerId==TIME_INFINITE)
		{			
			int rest=gameObjProp[getType()].maxTime-(t-t0);
			fatalAssert(rest>=0,"time negative");
			return rest;
		}
		return tx-t;
	}
	///
	unsigned int getKey() const { return key;}
	void         setKey(unsigned int k) { key=k;}
	///
	void set(unsigned int updateTimer,unsigned int inKey,const gameObjBaseC & base)
	{
		key=inKey;
		t0=updateTimer;
		gameObjTimePointsC::set(updateTimer,base.getStartPos());		
		gameObjBaseC::set(base);
		trackCount=0;				
	}
	///
	void resetTrack(unsigned int updateTimer,const vecIntC & inPos)
	{
		trackCount=0;		
		gameObjTimePointsC::set(updateTimer,inPos);		
		gameObjBaseC::set(getType(),inPos);		
	}
	/// return the number of tracks
	int getTrackCount() const { return trackCount;}
	///
	void update(unsigned int updateTimer,const vecIntC & inPos) 
	{ trackCount++;
		gameObjTimePointsC::update(updateTimer,inPos);
	}
	///
	int isUsed(unsigned int updateTimer) { return getT1()==updateTimer;}	
	/** set my killing time and the object id of the killer
	    will return earlier killer in free1 or/and free2 (value of 0 indicate no earlier killer)
			return 1 if the killer was accepted
	*/
	int setKillTime(unsigned int currentTime,unsigned int time,gameObjC & killer,unsigned int & free1,unsigned int & free2);	
	/// same code as above but with single call
	int setKillTime(unsigned int currentTime,unsigned int time,unsigned int callerId,unsigned int callerTx,unsigned int & free);
	///
	void delKiller() { tx=TIME_INFINITE;killerId=0;}
	/// return start time of object
	unsigned int getStartime() const { return t0;}	
};

class gameObjListC : public constArrayT<128,gameObjC> {
public:
	/// return itme from lsit by given key (runtime n/2)
	int getByKey(unsigned int key) const
	{
		for (int i=0;i<count();i++)
			if ((*this)[i].getKey()==key)
				return i;
		return -1;
	}
	///
	unsigned int getHighKey() const
	{
		unsigned int k=0;

		for (int i=0;i<count();i++)
			if ((*this)[i].getKey()>k)
				k=(*this)[i].getKey();
		return k;
	}
	///
	int countTargets() const
	{
		int c=0;

		for (int i=0;i<count();i++)
			if ((*this)[i].getType()!=objTypeShot)
				c++;
		return c;
	}
	///
	int countTypes(gameObjTypeE type) const
	{
		int c=0;
		for (int i=0;i<count();i++)
		{
			if ((*this)[i].getType()==type)
				c++;
		}
		return c;								
	}

	/* this methode is not enough because we have also to take care about the tx items 
	/// are there maximal count of asteroids here
	int maximalAsteroids() const 
	{
		if (count()<25) return 0;
		int astroCount=0;

		for (int i=0;i<count();i++)
		{
			switch ((*this)[i].getType()) {
				case objTypeAstroidSmall:
				case objTypeAstroidMed:
				case objTypeAstroidLarge:
					astroCount++;
					break;
			}
		}
		return astroCount>=24;
	}
	*/
};

/// my ship definition
class spaceShipC {	
	///
	vecIntC position;	
	///	
	unsigned int updateTime;
	///
	int     rotationIndex;			
	///
	unsigned int lastFire;	
public:	
	/// ctor
	spaceShipC() { rotationIndex=0;updateTime=0;lastFire=0;};
	/// new time .. new position from frame .. rotate indicates if we press last round l(-1) or r(+1) .. head is the value from frame buffer
	void update(unsigned int time,const vecIntC & pos,const vecIntC & head,int rotate);
	/// artificail update
	void updateExt(unsigned int time,const vecIntC & pos,int head)
	{ updateTime=time;position=pos;rotationIndex=head;}
	/// get current direction in index
	int getRotationIndex() const { return rotationIndex;}
	///
	void hitRotate(int head) { rotationIndex=getHeadIdx(rotationIndex+head);}
	/// return head as vecotr
	vecDblC getHead() const { return vecDblC(cos(rotationData[rotationIndex].angDegree/180*PI),sin(rotationData[rotationIndex].angDegree/180*PI));}
	///
	const vecIntC &  getPosition() const {return position;}
	/// add a shot to our list
	void shot(unsigned int time) 
	{ // fatalAssert(shots.count()<4,"shots too high");
		lastFire=time;
	}
	///
	unsigned int getLastFireTime() { return lastFire;}

};

#endif
