
namespace PlayerLogic
{
    public class GameStatus
    {
        public const int MAXASTEROIDS =  30;
        public const int MAXBULLETS   =  10;
        public const int BULLETSPEED  =   8;

        public long frameNumber;
        public int latenz;
        public int framesToFire;
        public long score;

        private struct EventBufferEntry
        {
            byte frameId;
            byte keys;
        }

        public bool hasShip;
        public int shipX;
        public int shipY;
        public byte shipAngle;
        public byte shipExpectedAngle;
        public double shipAX; // Angle Direction X
        public double shipAY; // Angle Direction Y
        public double shipEX; // Expected Direction X
        public double shipEY; // Expected Direction Y
        public double shipPX;
        public double shipPY;
        public double shipRX;
        public double shipRY;
        public double shipDX; // Movement Direction X
        public double shipDY; // Movement Direction Y

        public int targetX;
        public int targetY;
        public double interceptX;
        public double interceptY;

        public bool hasSaucer;
        public bool hadSaucer;
        public Saucer saucer;

        public int numAsteroids;
        public Asteroid[] asteroids;
        public int numAsteroidsOld;
        public Asteroid[] asteroidsOld;

        public void AddAsteroid(int x, int y, int type, int size, int frameFactor)
        {
            asteroids[numAsteroids].X = x;
            asteroids[numAsteroids].Y = y;
            asteroids[numAsteroids].Type = type;
            asteroids[numAsteroids].Size = size;
            asteroids[numAsteroids].HasDirection = false;
            int prevId = FindPreviousAsteroid(numAsteroids, frameFactor);
            if (prevId >= 0)
            {
                // Good chance to get a correct direction info
                int dx = (x - asteroidsOld[prevId].X) / frameFactor;
                int dy = (y - asteroidsOld[prevId].Y) / frameFactor;

                // TODO: Check for direction changes to detect destroyed asteroids overlaid with new ones
                asteroids[numAsteroids].HasDirection = true;
                if (asteroidsOld[prevId].HasDirection)
                {
                    asteroids[numAsteroids].DX = asteroidsOld[prevId].DX * 0.9f + dx * 0.1f;
                    asteroids[numAsteroids].DY = asteroidsOld[prevId].DY * 0.9f + dy * 0.1f;
                }
                else
                {
                    asteroids[numAsteroids].DX = dx;
                    asteroids[numAsteroids].DY = dy;
                }
            }
            ++numAsteroids;
        }

        private int FindPreviousAsteroid(int current, int frameFactor)
        {
            int closestMatch = -1;
            int closestDist = 1024*1024;
            int closestGuess = -1;
            int closestGuessDist = 1024 * 1024;
            int cpx = asteroids[current].X;
            int cpy = asteroids[current].Y;

            for  (int i = 0; i < numAsteroidsOld; ++i)
            {
                if (asteroids[current].Size == asteroidsOld[i].Size && asteroids[current].Type == asteroidsOld[i].Type)
                {
                    int epx, epy;
                    if (asteroidsOld[i].HasDirection)
                    {
                        epx =
                            Utility.normalizeX(asteroidsOld[i].X + (int)(asteroidsOld[i].DX * frameFactor + 0.5f) -
                                                    cpx);
                        epy =
                            Utility.normalizeY(asteroidsOld[i].Y + (int)(asteroidsOld[i].DY * frameFactor + 0.5f) -
                                                    cpy);
                        int dist = epx*epx + epy*epy;
                        if (dist < 9 && dist < closestDist)
                        {
                            closestDist = dist;
                            closestMatch = i;
                        }
                    }
                    else
                    {
                        epx = Utility.normalizeX(asteroidsOld[i].X - cpx);
                        epy = Utility.normalizeY(asteroidsOld[i].Y - cpy);
                        int dist = epx*epx + epy*epy;
                        if(dist < 64 && dist < closestGuessDist)
                        {
                            closestGuessDist = dist;
                            closestGuess = i;
                        }
                    }
                }
            }
            return (closestMatch >= 0 ? closestMatch : closestGuess);
        }

        public bool hasCriticalBullet;
        public int numBullets;
        public Bullet[] bullets;
        public int numBulletsOld;
        public Bullet[] bulletsOld;

        public void AddBullet(int x, int y, int frameFactor)
        {
            bullets[numBullets].X = x;
            bullets[numBullets].Y = y;
            bullets[numBullets].HasDirection = false;
            bullets[numBullets].IsDangerous = false;
            bullets[numBullets].FromSaucer = false;
            bullets[numBullets].LifeCounter = 0;
            ++numBullets;
        }

        public GameStatus()
        {
            frameNumber = 0;
            latenz = 2;
            framesToFire = 0;
            score = 0;
            hasShip = false;
            shipX = 0; shipY = 0;
            shipAngle = 0;
            shipAX = 0; shipAY = 0;
            shipDX = 0; shipDY = 0;
            shipRX = 0; shipRY = 0;
            shipPX = 0; shipPY = 0;
            hasSaucer = false;
            hadSaucer = false;
            saucer.X = 0;
            saucer.Y = 0;
            saucer.DX = 0;
            saucer.DY = 0;
            saucer.Size = 0;
            numAsteroids = 0;
            asteroids = new Asteroid[MAXASTEROIDS];
            numAsteroidsOld = 0;
            asteroidsOld = new Asteroid[MAXASTEROIDS];
            hasCriticalBullet = false;
            numBullets = 0;
            bullets = new Bullet[MAXBULLETS];
            numBulletsOld = 0;
            bulletsOld = new Bullet[MAXBULLETS];
            for (int i = 0; i < MAXBULLETS; ++i)
            {
                bullets[i].X = 0;
                bullets[i].Y = 0;
            }
        }

        public void Clear()
        {
            numAsteroidsOld = numAsteroids;
            for (int i = 0; i < numAsteroids; ++i)
            {
                asteroidsOld[i] = asteroids[i];
            }
            numBulletsOld = numBullets;
            for (int i = 0; i < numBullets; ++i)
            {
                bulletsOld[i] = bullets[i];
            }
            hasShip = false;
            hadSaucer = hasSaucer;
            hasSaucer = false;
            hasCriticalBullet = false;
            numAsteroids = 0;
            numBullets = 0;
            targetX = -1;
        }

        public void PostprocessStatus(int frameFactor)
        {
            for (int i = 0; i < numBullets; ++i)
            {
                int x = bullets[i].X;
                int y = bullets[i].Y;
                int prevId = FindPreviousBullet(i, frameFactor);
                if (prevId >= 0)
                {
                    bullets[i].LifeCounter = bullets[prevId].LifeCounter + frameFactor;
                    int dx = Utility.normalizeX(x - bulletsOld[prevId].X) / frameFactor;
                    int dy = Utility.normalizeY(y - bulletsOld[prevId].Y) / frameFactor;
                    bullets[i].FromSaucer = bulletsOld[prevId].FromSaucer;
                    bullets[i].HasDirection = true;
                    if (bulletsOld[prevId].HasDirection)
                    {
                        bullets[i].DX = bulletsOld[prevId].DX * 0.9f + dx * 0.1f;
                        bullets[i].DY = bulletsOld[prevId].DY * 0.9f + dy * 0.1f;
                    }
                    else
                    {
                        bullets[i].DX = dx;
                        bullets[i].DY = dy;
                    }

                    double rx = Utility.normalizeX(x - shipX);
                    double ry = Utility.normalizeY(y - shipY);
                    double fdx = bullets[i].DX;
                    double fdy = bullets[i].DY;
                    double currentDist = rx * rx + ry * ry;
                    double nextDist = (fdx + rx) * (fdx + rx) + (fdy + ry) * (fdy + ry);
                    if (currentDist > nextDist)
                    {
                        if (nextDist <= 27 * 27)
                        {
                            bullets[i].IsDangerous = true;
                            hasCriticalBullet = true;
                        }
                    }
                }
                else
                {
                    int shipDist = 0x7fffffff;
                    int saucerDist = 0x7fffffff;

                    if (hasShip)
                    {
                        int shipDistX = Utility.normalizeX(x - shipX);
                        int shipDistY = Utility.normalizeY(y - shipY);
                        shipDist = shipDistX * shipDistX + shipDistY * shipDistY;
                    }
                    if (hasSaucer)
                    {
                        int saucerDistX = Utility.normalizeX(x - saucer.X);
                        int saucerDistY = Utility.normalizeY(y - saucer.Y);
                        saucerDist = saucerDistX * saucerDistX + saucerDistY * saucerDistY;
                    }
                    if (saucerDist < shipDist)
                    {
                        bullets[i].FromSaucer = true;
                    }
                    else
                    {
                        bullets[i].DX = shipAX * 8.0f + shipDX;
                        bullets[i].DY = shipAY * 8.0f + shipDY;
                        bullets[i].HasDirection = true;
                    }
                }
            }

            for (int i = 0; i < numAsteroids; ++i)
            {
                asteroids[i].IsHit = IsHitSoon(i);
            }

            saucer.IsHit = false;
            for (int i = 0; i < numBullets; ++i)
            {
                if (CheckForHitIterative(bullets[i].X, bullets[i].Y, bullets[i].DX, bullets[i].DY, bullets[i].FramesLeft, saucer) >= 0)
                    saucer.IsHit = true;
            }
        }

        private int FindPreviousBullet(int current, int frameFactor)
        {
            int closestMatch = -1;
            int closestDist = 1024 * 1024;
            int closestGuess = -1;
            int closestGuessDist = 1024 * 1024;
            int cpx = bullets[current].X;
            int cpy = bullets[current].Y;

            for (int i = 0; i < numBulletsOld; ++i)
            {
                int epx;
                int epy;
                if (bulletsOld[i].HasDirection)
                {
                    epx = Utility.normalizeX(bulletsOld[i].X + (int)(bulletsOld[i].DX * frameFactor) - cpx);
                    epy = Utility.normalizeY(bulletsOld[i].Y + (int)(bulletsOld[i].DY * frameFactor) - cpy);
                    int dist = epx * epx + epy * epy;
                    if (dist < 9 && dist < closestDist)
                    {
                        closestDist = dist;
                        closestMatch = i;
                    }
                }
                else
                {
                    epx = Utility.normalizeX(cpx - bulletsOld[i].X);
                    epy = Utility.normalizeY(cpy - bulletsOld[i].Y);
                    int dist = epx * epx + epy * epy - 64;
                    if (dist < closestGuessDist)
                    {
                        closestGuessDist = dist;
                        closestGuess = i;
                    }
                }
            }
            return (closestMatch >= 0 ? closestMatch : closestGuess);
        }

        public bool IsHitSoon(int idx)
        {
            for (int i = 0; i < numBullets; ++i)
            {
                if (CheckForHitIterative(bullets[i].X, bullets[i].Y, bullets[i].DX, bullets[i].DY, bullets[i].FramesLeft, asteroids[idx]) >= 0)
                    return true;
            }
            return false;
        }

        public static int CheckForHitIterative(double bx, double by, double bdx, double bdy, int framesLeft, GameObject obj)
        {
            bx = Utility.normalizeX(bx - obj.X);
            by = Utility.normalizeY(by - obj.Y);
            for (int i = 0; i < framesLeft; ++i)
            {
                if (Utility.IsInside(bx, by, obj.Radius))
                    return i;
                //if (Utility.Distance(0, 0, bx, by) + i/12 < obj.Radius)
                //    return i;
                bx = Utility.normalizeX(bx + bdx - obj.DX);
                by = Utility.normalizeY(by + bdy - obj.DY);
            }
            return -1;
        }

        public int NextHitDistance()
        {
            double bx = shipX + shipEX * (19.5 - 16.0);
            double by = shipY + shipEY * (19.5 - 16.0);
            double bdx = shipEX * 8.0;
            double bdy = shipEY * 8.0;
            int min_dist = 0x7fffffff;
            int dist;
            for (int i = 0; i < numAsteroids; ++i)
            {
                if (!asteroids[i].IsHit)
                {
                    dist = CheckForHitIterative(bx, by, bdx, bdy, 70, asteroids[i]);
                    if (dist >= 0 && dist < min_dist)
                        min_dist = dist;
                }                
            }
            if (hasSaucer)
            {
                dist = CheckForHitIterative(bx, by, bdx, bdy, 70, saucer);
                if (dist >= 0 && dist < min_dist)
                    min_dist = dist;
            }
            return min_dist < 0x7ffffff ? min_dist : -1;
        }

        public bool WillHitSomething()
        {
            double bx = shipX + shipEX * (19.5 - 16.0);
            double by = shipY + shipEY * (19.5 - 16.0);
            double bdx = shipEX * 8.0;
            double bdy = shipEY * 8.0;
            for (int i = 0; i < numAsteroids; ++i)
            {
                if (!asteroids[i].IsHit && CheckForHitIterative(bx, by, bdx, bdy, 70, asteroids[i]) >= 0)
                    return true;
            }
            if (hasSaucer)
                return (CheckForHitIterative(bx, by, bdx, bdy, 70, saucer) >= 0);
            return false;
        }

        public bool WillHitSaucer()
        {
            if (hasSaucer)
            {
                double bx = shipX + shipEX * (19.5 - 16.0);
                double by = shipY + shipEY * (19.5 - 16.0);
                double bdx = shipEX * 8.0;
                double bdy = shipEY * 8.0;
                return (CheckForHitIterative(bx, by, bdx, bdy, 70, saucer) >= 0);
            }
            return false;
        }

        public void PredictStatus(GameStatus real, int frames)
        {
            frameNumber = real.frameNumber + frames;
            latenz = real.latenz;
            hasShip = true;
            shipX = (int)(real.shipX + real.shipDX * frames);
            shipY = (int)(real.shipY + real.shipDY * frames);
            shipAngle = real.shipAngle;
            shipAX = real.shipAX; shipAY = real.shipAY;
            shipEX = real.shipEX; shipEY = real.shipEY;
            shipDX = real.shipDX; shipDY = real.shipDY;
            hasSaucer = real.hasSaucer;
            saucer.X = (int)(real.saucer.X + (real.saucer.HasDirection ? real.saucer.DX * frames : 0));
            saucer.Y = (int)(real.saucer.Y + (real.saucer.HasDirection ? real.saucer.DY * frames : 0));
            saucer.Size = real.saucer.Size;
            saucer.HasDirection = real.saucer.HasDirection;
            saucer.DX = real.saucer.DX;
            saucer.DY = real.saucer.DY;

            numAsteroids = real.numAsteroids;
            for (int i = 0; i < numAsteroids; ++i)
            {
                asteroids[i].Type = real.asteroids[i].Type;
                asteroids[i].Size = real.asteroids[i].Size;
                asteroids[i].HasDirection = real.asteroids[i].HasDirection;
                if (asteroids[i].HasDirection)
                {
                    asteroids[i].DX = real.asteroids[i].DX;
                    asteroids[i].DY = real.asteroids[i].DY;
                    asteroids[i].X = (int)(real.asteroids[i].X + asteroids[i].DX * frames + 0.5f);
                    asteroids[i].Y = (int)(real.asteroids[i].Y + asteroids[i].DY * frames + 0.5f);
                }
                else
                {
                    asteroids[i].X = real.asteroids[i].X;
                    asteroids[i].Y = real.asteroids[i].Y;
                }
            }

            numBullets = real.numBullets;
            for (int i = 0; i < numBullets; ++i)
            {
                bullets[i].HasDirection = real.bullets[i].HasDirection;
                if (bullets[i].HasDirection)
                {
                    bullets[i].DX = real.bullets[i].DX;
                    bullets[i].DY = real.bullets[i].DY;
                    bullets[i].X = (int)(real.bullets[i].X + bullets[i].DX * frames + 0.5f);
                    bullets[i].Y = (int)(real.bullets[i].Y + bullets[i].DY * frames + 0.5f);
                }
                else
                {
                    bullets[i].X = real.bullets[i].X;
                    bullets[i].Y = real.bullets[i].Y;
                }
            }

            for (int i = 0; i < numAsteroids; ++i)
            {
                asteroids[i].IsHit = IsHitSoon(i);
            }

            saucer.IsHit = false;
            for (int i = 0; i < numBullets; ++i)
            {
                if (CheckForHitIterative(bullets[i].X, bullets[i].Y, bullets[i].DX, bullets[i].DY, bullets[i].FramesLeft, saucer) >= 0)
                    saucer.IsHit = true;
            }
        }
    }
}
