package de.curdreinert.asteroids.intelligence;

import java.util.ArrayList;
import java.util.List;

import de.curdreinert.asteroids.base.Log;
import de.curdreinert.asteroids.base.UnexpectedException;
import de.curdreinert.asteroids.base.Utility;
import de.curdreinert.asteroids.screenobject.Asteroid;
import de.curdreinert.asteroids.screenobject.Saucer;
import de.curdreinert.asteroids.screenobject.Ship;
import de.curdreinert.asteroids.screenobject.Shot;

public class Screen {

	private static int currentId = 0;

	private int id = currentId++;

	private int[] data = new int[512];

	private byte frame;

	private byte ping;

	private List<Asteroid> asteroids = new ArrayList<Asteroid>();

	private Saucer saucer = null;

	private List<Shot> shots = new ArrayList<Shot>();

	private Ship ship = null;

	private String message = "";

	private int lives = 0;

	public Screen(byte[] data) {
		this.frame = data[1024];
		this.ping = data[1025];
		for (int i = 0; i < 512; i++) {
			this.data[i] = Utility.word(data[2 * i], data[2 * i + 1]);
		}
		parse();
	}

	public byte getFrame() {
		return frame;
	}

	public byte getPing() {
		return ping;
	}

	public String toString() {
		StringBuffer result = new StringBuffer();
		result.append("Frame: " + frame + " Ping: " + ping + "\n");

		if (ship != null) {
			result.append(ship + "\n");
		} else {
			result.append("No ship\n");
		}

		for (Asteroid asteroid : asteroids) {
			result.append(asteroid + "\n");
		}

		for (Shot shot : shots) {
			result.append(shot + "\n");
		}

		if (saucer != null) {
			result.append(saucer + "\n");
		} else {
			result.append("No saucer\n");
		}

		result.append("Score: " + getScore() + "\n");
		result.append("Lives: " + lives + "\n");

		return result.toString();
	}

	private void parse() {
		int dx = 0;
		int dy = 0;
		int x = 0;
		int y = 0;
		int brigthness = 0;
		int scale = 0;
		int v1x = 0;
		int v1y = 0;
		int shipdetect = 0;

		int pc = 1;
		for (;;) {
			int op = data[pc] >> 12;
			switch (op) {

			case 0xa: // LABS
				y = data[pc] & 0x3ff;
				x = data[pc + 1] & 0x3ff;
				scale = data[pc + 1] >> 12;
				break;

			case 0xb: // HALT
				return;

			case 0xc: // JSRL
				switch (data[pc] & 0xfff) {
				case 0x8f3:
					asteroids.add(new Asteroid(x, y, 1, scale));
					break;
				case 0x8ff:
					asteroids.add(new Asteroid(x, y, 2, scale));
					break;
				case 0x90d:
					asteroids.add(new Asteroid(x, y, 3, scale));
					break;
				case 0x91a:
					asteroids.add(new Asteroid(x, y, 4, scale));
					break;
				case 0x929:
					saucer = new Saucer(x, y, scale);
					break;
				case 0xA6D:
					lives++;
					break;
				default:
					handleCharacter(data[pc] & 0xfff);
				}
				break;

			case 0xd: // RTSL
				throw new UnexpectedException("Cannot handle RTSL");

			case 0xe: // JMPL
				throw new UnexpectedException("Cannot handle JMPL");

			case 0xf: // SVEC
				/*
				 * dy = data[pc] & 0x300; if ((data[pc] & 0x400) != 0) dy = -dy;
				 * dx = (data[pc] & 3) << 8; if ((data[pc] & 4) != 0) dx = -dx;
				 * sf = (((data[pc] & 8) >> 2) | ((data[pc] & 0x800) >> 11)) +
				 * 2; vz = (data[pc] & 0xf0) >> 4;
				 */
				break;

			default:
				dy = data[pc] & 0x3ff;
				if ((data[pc] & 0x400) != 0) {
					dy = -dy;
				}
				dx = data[pc + 1] & 0x3ff;
				if ((data[pc + 1] & 0x400) != 0) {
					dx = -dx;
				}
				brigthness = data[pc + 1] >> 12;

				if (dx == 0 && dy == 0 && brigthness == 15) {
					shots.add(new Shot(x, y));
				}

				if (op == 6 && brigthness == 12 && dx != 0 && dy != 0) {
					switch (shipdetect) {
					case 0:
						v1x = dx;
						v1y = dy;
						++shipdetect;
						break;
					case 1:
						ship = new Ship(x, y, v1x - dx, v1y - dy);
						++shipdetect;
						break;
					}
				} else if (shipdetect == 1) {
					shipdetect = 0;
				}
				break;
			}
			if (op <= 0xa) { // 2Word-Opcodes
				pc += 2;
			} else {
				pc += 1;
			}
		}
	}

	private void handleCharacter(int address) {
		switch (address) {
		case 0x852:
			message += "(C) 1979 ATARI INC";
			break;
		case 0xA78:
			message += "A";
			break;
		case 0xA80:
			message += "B";
			break;
		case 0xA8D:
			message += "C";
			break;
		case 0xA93:
			message += "D";
			break;
		case 0xA9B:
			message += "E";
			break;
		case 0xAA3:
			message += "F";
			break;
		case 0xAAA:
			message += "G";
			break;
		case 0xAB3:
			message += "H";
			break;
		case 0xABA:
			message += "I";
			break;
		case 0xAC1:
			message += "J";
			break;
		case 0xAC7:
			message += "K";
			break;
		case 0xACD:
			message += "L";
			break;
		case 0xAD8:
			message += "N";
			break;
		case 0xADD:
			message += "0";
			break;
		case 0xAE3:
			message += "P";
			break;
		case 0xAEA:
			message += "Q";
			break;
		case 0xAF3:
			message += "R";
			break;
		case 0xAFB:
			message += "S";
			break;
		case 0xB02:
			message += "T";
			break;
		case 0xB08:
			message += "U";
			break;
		case 0xB0E:
			message += "V";
			break;
		case 0xB13:
			message += "W";
			break;
		case 0xB1A:
			message += "X";
			break;
		case 0xB1F:
			message += "Y";
			break;
		case 0xB26:
			message += "Z";
			break;
		case 0xB2C:
			message += " ";
			break;
		case 0xB2E:
			message += "1";
			break;
		case 0xB32:
			message += "2";
			break;
		case 0xB3A:
			message += "3";
			break;
		case 0xB41:
			message += "4";
			break;
		case 0xB48:
			message += "5";
			break;
		case 0xB4F:
			message += "6";
			break;
		case 0xB56:
			message += "7";
			break;
		case 0xB5B:
			message += "8";
			break;
		case 0xB63:
			message += "9";
			break;
		default:
			message += "[" + String.format("%03X", address) + "]";
		}
	}

	public List<Asteroid> getAsteroids() {
		return asteroids;
	}

	public Saucer getSaucer() {
		return saucer;
	}

	public Ship getShip() {
		return ship;
	}

	public List<Shot> getShots() {
		return shots;
	}

	public int getScore() {
		try {
			String scoreString = message.substring(
					message.indexOf("ATARI INC") + 9, message.lastIndexOf(' '))
					.trim();
			if (scoreString.indexOf(' ') != -1) {
				scoreString = scoreString
						.substring(0, scoreString.indexOf(' '));
			}
			return Integer.parseInt(scoreString);
		} catch (Exception e) {
			Log.warn("Error parsing " + message);
			e.printStackTrace();
			return -1;
		}
	}

	public boolean sameAsteroids(Screen asThis) {
		List<Asteroid> others = asThis.getAsteroids();
		if (asteroids.size() != others.size()) {
			return false;
		}
		for (int i = 0; i < asteroids.size(); i++) {
			if (!asteroids.get(i).mayBeSame(others.get(i))) {
				return false;
			}
		}
		return true;
	}

	public int getId() {
		return id;
	}

	public String getMessage() {
		return message;
	}

	public int getLives() {
		return lives;
	}

	public boolean sameShots(Screen asThis) {
		List<Shot> others = asThis.getShots();
		if (shots.size() != others.size()) {
			return false;
		}
		for (int i = 0; i < shots.size(); i++) {
			if (!shots.get(i).mayBeSame(others.get(i))) {
				return false;
			}
		}
		return true;
	}

	public int[] getData() {
		return data;
	}

}
