package de.curdreinert.asteroids.display;

import java.awt.Color;
import java.awt.Graphics2D;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import de.curdreinert.asteroids.base.Utility;

public class VectorPlotter {

	private int[] mem = new int[0xC00];

	private int x = 0;

	private int y = 0;

	private int gsf = 0;

	private int op = 0;

	private int base = -1;

	private Graphics2D graphics;

	private static final String[] MNEMONICS = new String[] { "VCTR", "VCTR",
			"VCTR", "VCTR", "VCTR", "VCTR", "VCTR", "VCTR", "VCTR", "VCTR",
			"LABS", "HALT", "JSRL", "RTSL", "JMPL", "SVEC" };

	public VectorPlotter() {
		loadVectorRom();
	}

	private void loadVectorRom() {
		try {
			InputStream stream = getClass().getClassLoader()
					.getResourceAsStream("ressources/035127.02");
			for (int i = 0; i < 0x400; i++) {
				int low = stream.read();
				int high = stream.read();
				mem[0x800 + i] = Utility.word((byte) low, (byte) high);
			}
			assert (stream.read() == -1);
			stream.close();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public void plot(Graphics2D graphics, int[] data) {
		this.graphics = graphics;
		base = (data[0] == 0xE001) ? 0 : 0x200;
		System.arraycopy(data, 0, mem, base, 0x200);
		simulate(base);
	}

	public void simulate(int start) {

		int pc = start;
		List<Integer> stack = new ArrayList<Integer>();
		int dx = 0;
		int dy = 0;
		int z = 0;

		do {
			int word1 = mem[pc];
			int word2 = mem[pc + 1];
			op = mem[pc] >> 12;

			switch (op) {
			case 0xA: // LABS
				x = word2 & 0xFFF;
				y = word1 & 0x3FF;
				gsf = word2 >> 12;
				pc += 2;
				break;

			case 0xB: // HALT
				break;

			case 0xC: // JSRL
				stack.add(pc + 1);
				pc = word1 & 0xFFF;
				break;

			case 0xD: // RTSL
				pc = stack.remove(stack.size() - 1);
				break;

			case 0xE: // JMPL
				pc = word1 & 0xFFF;
				break;

			case 0xF: // SVEC
				dx = (word1 & 0x003) << 8;
				if (isSet(word1, 0x004)) {
					dx = -dx;
				}
				dy = (word1 & 0x300);
				if (isSet(word1, 0x400)) {
					dy = -dy;
				}
				z = (word1 & 0x0F0) >> 4;
				int sf = ((isSet(word1, 0x800) ? 1 : 0)) + 2
						* ((isSet(word1, 0x008) ? 1 : 0));
				vector(dx, dy, z, svecDiv(sf));
				pc++;
				break;

			default: // VCTR
				dx = word2 & 0x3FF;
				if (isSet(word2, 0x400)) {
					dx = -dx;
				}
				dy = word1 & 0x3FF;
				if (isSet(word1, 0x400)) {
					dy = -dy;
				}
				z = word2 >> 12;
				vector(dx, dy, z, vctrDiv(op));
				pc += 2;
				break;
			}
		} while (op != 0x0B);
	}

	private void vector(int dx, int dy, int z, int scale) {
		graphics.setColor(color(z));
		int newX = x + dx / scale;
		int newY = y + dy / scale;
		if (z != 0) {
			graphics.drawLine(x, y, newX, newY);
		}
		x = newX;
		y = newY;
	}

	private Color color(int z) {
		return new Color(z * 16, z * 16, z * 16);
	}

	public void disassemble(int start) {
		int pc = start;
		int x = 0;
		int y = 0;
		int gsf = 0;
		int op = 0;
		int z = 0;
		int to;

		while (pc - start < 0x200) {
			int word1 = mem[pc];
			int word2 = mem[pc + 1];
			String line = "";
			op = mem[pc] >> 12;

			if (op <= 0xA) {
				line += String.format("%03X: %04X %04X ", pc, word1, word2);
			} else {
				line += String.format("%03X: %04X      ", pc, word1);
			}
			line += MNEMONICS[op];

			switch (op) {
			case 0xA: // LABS
				x = word2 & 0xFFF;
				y = word1 & 0x3FF;
				gsf = word2 >> 12;
				line += String.format(" (%d, %d), s%d", x, y, gsf);
				break;

			case 0xB: // HALT
				break;

			case 0xC: // JSRL
				to = word1 & 0xFFF;
				line += String.format(" $%03X", to);
				break;

			case 0xD: // RTSL
				break;

			case 0xE: // JMPL
				to = word1 & 0xFFF;
				line += String.format(" $%03X", to);
				break;

			case 0xF: // SVEC
				x = (word1 & 0x003) << 8;
				if (isSet(word1, 0x004)) {
					x = -x;
				}
				y = (word1 & 0x300);
				if (isSet(word1, 0x400)) {
					y = -y;
				}
				z = (word1 & 0x0F0) >> 4;
				int sf0 = isSet(word1, 0x800) ? 1 : 0;
				int sf1 = isSet(word1, 0x008) ? 1 : 0;
				int sf = sf0 + 2 * sf1;

				line += String.format(" (%d, %d), s%d, z%d", x, y, sf, z);
				break;

			default:
				x = word2 & 0xFFF;
				if (isSet(word2, 0x400)) {
					x = -x;
				}
				y = word1 & 0x3FF;
				if (isSet(word1, 0x400)) {
					y = -y;
				}
				z = word2 >> 12;
				line += String.format(" (%d, %d), s%d, z%d", x, y, op, z);
				break;
			}

			System.out.println(line);

			if (op <= 0xA) {
				pc += 2;
			} else {
				pc += 1;
			}
		}
		System.out.println();
	}

	private boolean isSet(int word, int mask) {
		return (word & mask) == mask;
	}

	private int vctrDiv(int scale) {
		switch ((scale + gsf) % 16) {
		case 9:
			return 1;
		case 8:
			return 2;
		case 7:
			return 4;
		case 6:
			return 8;
		case 5:
			return 16;
		case 4:
			return 32;
		case 3:
			return 64;
		case 2:
			return 128;
		case 1:
			return 256;
		case 0:
			return 512;
		}
		return 1;
	}

	private int svecDiv(int scale) {
		switch ((scale + gsf) % 16) {
		case 3:
			return 16;
		case 2:
			return 32;
		case 1:
			return 64;
		case 0:
			return 128;
		}
		return 1;
	}
}
