(in-package :weltraumputze)

(defun random-strategy (scene fire-command-available)
  (if (ship scene)
      (create-command (delete nil
                              (list
                               (if (= 0 (random 3))
                                   'right
                                   (if (= 0 (random 3))
                                       'left))
                               (if (= 0 (random 10))
                                   'fire))))
      0))

(defun win-strategy (current-scene fire-command-available current-command)
  ;; Without visible ship, no command can reasonably be given
  (when (null (ship current-scene))
    (return-from win-strategy 0))

  ;; Base calculations on a centered world so that we can ignore
  ;; screen edge wrap
  (let ((scene (create-centered-scene current-scene))
        (commands nil))

    ;; For each shot, find the nearest target and remove it
    ;; from the list of possible targets
    (setf (targets scene) (remove-shot-targets scene))

    ;; The angle byte used for a new shot is influenced by the
    ;; next ship movement. Since we are interested here for the
    ;; effective firing angle, just use the modified angle-byte
    (setf (angle-byte scene) (trace-angle-byte (angle-byte scene) current-command))

    ;(dump-scene scene)

    ;; If a shot seems to be likely successful, shoot
    (if (and fire-command-available (angle-byte scene))
        (multiple-value-bind (target estimated-time) (shot-will-hit scene)
          (if (and target
                   (or (eq (type-of target) 'flying-saucer) ; shoot on flying saucers
                       (collision 0 0 0 0 (dim (ship scene)) ; and also on colliding asteroids
                                  (x target) (y target) (or (dx target) 0) (or (dy target) 0) (dim target))
                       (<= estimated-time (determine-max-time scene)))) ; or if near / few asteroids near
              (setf commands (push 'fire commands)))))

    ;; Old shooting logic
;    (if (and fire-command-available (angle-byte scene) (shot-will-hit scene))
;       (setf commands (push 'fire commands)))

    ;; Turn to the most threatening target
    (let ((mt-target (car (collision-targets scene))))
      ; Second biggest threat (and also most valuable) are flying saucers
      (if (not mt-target) (setf mt-target (flying-saucer scene)))
      ; Should there be only one target left, turn to the target
      (if (= 1 (length (targets scene))) (setf mt-target (car (targets scene))))
      (if mt-target
          (multiple-value-bind (gx gy) (angle-byte-to-gun (angle-byte scene))
            (let ((angle (- (angle (- gx (or (dx mt-target) 0)) (- gy (or (dy mt-target) 0)))
                            (angle (x mt-target) (y mt-target)))))
              (if (< angle 0)
                  (setf angle (+ angle (* 2 pi))))

              (if (< angle pi)
                  (setf commands (push 'right commands))
                  (setf commands (push 'left commands)))))

          (setf commands (push 'right commands)))) ; Otherwise, just turn right

    ; Add thrust at random for testing purposes
    ;(if (= (random 2) 0)
    ;    (setf commands (push 'thrust commands)))

    (create-command commands)
    ))

(defun shot-will-hit (scene)
  (multiple-value-bind (gx gy) (angle-byte-to-gun (angle-byte scene))
    (let ((shot-x (* gx -5/8)) ; start position would be in distance 3
          (shot-y (* gy -5/8)) ; from ship position in the next scene
          (shot-dx gx) ; this is
          (shot-dy gy) ; simplified
          (closest-target nil)
          (estimated-collision-time nil))
      (dolist (target (targets scene))
        (if (dx target)
            (let ((estimated-collision
                   (collision shot-x shot-y shot-dx shot-dy 0.5
                              (x target) (y target) (dx target) (dy target) (dim target))))
              (if (and estimated-collision (< estimated-collision 72)
                       (or (null closest-target) (< estimated-collision estimated-collision-time)))
                  (progn
                    (setf closest-target target)
                    (setf estimated-collision-time estimated-collision))))))
      (values closest-target estimated-collision-time))))

;; Returns the maximum time acceptable for a shot to reach
;; its target
(defun determine-max-time (scene)
  (let ((n 0))
    (dolist (target (targets scene))
      (if (< (magnitude (x target) (y target)) 240)
          (incf n (size target))))
    (if (>= n 6)
        30
        72)))

;; Returns the list of targets of the supplied scene
;; with all targets remove that are likely to be shot
(defun remove-shot-targets (scene)
  (let ((targets (targets scene)))
    (dolist (shot (shots scene))
      (when (dx shot)
        (let ((shot-target nil)
              (shortest-target-distance nil))
          (dolist (target targets)
            (when (dx target)
              (let ((distance (collision (x shot) (y shot) (dx shot) (dy shot) (dim shot)
                                         (x target) (y target) (dx target) (dy target) (dim target))))
                (when (and distance
                           (> distance 0)
                           (< distance (- 71 (- (frame-no scene) (first-frame shot))))
                           (or (null shortest-target-distance)
                               (< distance shortest-target-distance)))
                  (setf shot-target target)
                  (setf shortest-target-distance distance)))))
          (when shot-target
            (setf targets (remove shot-target targets))))))
    targets))

;; Returns all targets going to collide with the ship
;; in chronological order
(defun collision-targets (scene)
  (flet ((collides (target)
           (collision 0 0 0 0 (dim (ship scene))
                      (x target) (y target) (or (dx target) 0) (or (dy target) 0) (dim target))))
    (let ((targets (remove-if-not #'collides (targets scene))))
      (sort targets #'< :key #'collides))))

;; Returns the flying saucer if it exists in the supplied scene
(defun flying-saucer (scene)
  (find 'flying-saucer (targets scene) :key #'type-of))
