/* Functions and datatypes for vector computations */
/* $Id: vector.c,v 1.7 2008/05/17 21:12:01 raap Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "vector.h"

/* Compute the 2D length of the line segment between P1 and P2 */
double LineLength2D (XY p1, XY p2){
	return sqrt( 	(p1.x - p2.x) * (p1.x - p2.x) + 
			(p1.y - p2.y) * (p1.y - p2.y)
		   );
};


/* Compute the length of the line segment between P1 and P2 */
double LineLength3D (XYZ p1, XYZ p2){
	return sqrt( 	(p1.x - p2.x) * (p1.x - p2.x) + 
			(p1.y - p2.y) * (p1.y - p2.y) +
			(p1.z - p2.z) * (p1.z - p2.z) 
		   );
};
	
/* The length of the given vector */
double VectorLength2D (XY p1){
	return sqrt( p1.x * p1.x + p1.y * p1.y);
};
		

/* Calculates the intersection of Lines P1P2 and P3P4 in 2D
	Stores results in mua and mub, where mua and mub are defined by:
	P(intersect) = P1 + mua * P1P2
	P(intersect) = P3 + mub * P3P4
	
	The lines are given in parameter form
		
	Returns -1 if the lines are (nearly) parallel, but not coincident
	Returns 0  if the lines are (nearly) coincident
	Returns 1  if one intersection point exists
		Onyl in this case are mua and mub set.
*/
int LineIntersect2D( XY p1, XY p1p2, XY p3, XY p3p4, double *mua, double *mub){
	double numera, numerb, denom;
	
	denom = p3p4.y * p1p2.x - p3p4.x * p1p2.y;
	numera = ( p3p4.x * (p1.y - p3.y) - p3p4.y * (p1.x -p3.x) );
	numerb = ( p1p2.x * (p1.y - p3.y) - p1p2.y * (p1.x -p3.x) );
	
	if (abs(denom) < EPS){
		if (abs(numera) < EPS)
			return 0;	/* coincident lines */ 
		else
			return -1;	/* parallele lines */
	} else {
		*mua = numera / denom;
		*mub = numerb / denom;
	};
	return 1;
};

/* We have a very special case. We want to get the intersection point of two lines.
	We know, they are at right angles to each other.
	So we need to know the two starting points and the direction of one of the two, lets say the first
	We use the standard LineIntersect2D algorithm above
	but use p3p4.x = -p1p2.y and p3p4.y = p1p2.x
*/
int LineIntersectRightAngle(XY p1, XY p1p2, XY p3, double *mua, double *mub){
	double numera, numerb, denom;
	
	denom = p1p2.x * p1p2.x + p1p2.y * p1p2.y;	
	numera = ( (-p1p2.y) * (p1.y - p3.y) - p1p2.x * (p1.x -p3.x) );
	numerb = ( p1p2.x * (p1.y - p3.y) - p1p2.y * (p1.x -p3.x) );
	*mua = numera / denom;
	*mub = numerb / denom;
	return 1;
};
		
		
/*	
   Calculate the line segment PaPb that is the shortest route between
   two lines P1P2 and P3P4. Calculate also the values of mua and mub where
      Pa = P1 + mua (P2 - P1)
      Pb = P3 + mub (P4 - P3)
   Return 0 if no solution exists,
	else 1.
*/
int LineLineIntersect( XYZ p1, XYZ p2, XYZ p3, XYZ p4, XYZ *pa, XYZ *pb, double *mua, double *mub)
{
	XYZ p13,p43,p21;
	double d1343,d4321,d1321,d4343,d2121;
	double numer,denom;
	
	
	p13.x = p1.x - p3.x;
	p13.y = p1.y - p3.y;
	p13.z = p1.z - p3.z;
	p43.x = p4.x - p3.x;
	p43.y = p4.y - p3.y;
	p43.z = p4.z - p3.z;
	if ((abs(p43.x)  < EPS) && (abs(p43.y)  < EPS) && (abs(p43.z)  < EPS))
		return(0);
	p21.x = p2.x - p1.x;
	p21.y = p2.y - p1.y;
	p21.z = p2.z - p1.z;
	if ((abs(p21.x)  < EPS) && (abs(p21.y)  < EPS) && (abs(p21.z)  < EPS))
		return(0);

	d1343 = p13.x * p43.x + p13.y * p43.y + p13.z * p43.z;
	d4321 = p43.x * p21.x + p43.y * p21.y + p43.z * p21.z;
	d1321 = p13.x * p21.x + p13.y * p21.y + p13.z * p21.z;
	d4343 = p43.x * p43.x + p43.y * p43.y + p43.z * p43.z;
	d2121 = p21.x * p21.x + p21.y * p21.y + p21.z * p21.z;

	denom = d2121 * d4343 - d4321 * d4321;
	if (abs(denom) < EPS)
		return(0);
	numer = d1343 * d4321 - d1321 * d4343;

	*mua = numer / denom;
	*mub = (d1343 + d4321 * (*mua)) / d4343;

	pa->x = p1.x + *mua * p21.x;
	pa->y = p1.y + *mua * p21.y;
	pa->z = p1.z + *mua * p21.z;
	pb->x = p3.x + *mub * p43.x;
	pb->y = p3.y + *mub * p43.y;
	pb->z = p3.z + *mub * p43.z;

	return(1);
}

/* Solve a quadratic equation a*x^2 + b*x +c = 0.
	Returns TRUE, if 1 or 2 real solutions exist
	else FALSE
	Solutions are stored in *x1, *x2
*/
BOOL quadratic_solve(double a, double b, double c, double *x1, double *x2){
	double det;
	
	det = b*b - 4*a*c;
	if (det < 0) return FALSE;
	det = sqrt(det);
	*x1 = (det - b) / (2 * a);
	*x2 = (-b - det) / (2 * a); 
	return TRUE;
};

/*
   Calculate the intersection of a ray and a sphere
   The line segment is defined from p1 to p2
   The sphere is of radius r and centered at sc
   There are potentially two points of intersection given by
   p = p1 + mu1 (p2 - p1)
   p = p1 + mu2 (p2 - p1)
   Return FALSE if the ray doesn't intersect the sphere.
   Taken from http://local.wasp.uwa.edu.au/~pbourke/geometry/sphereline/raysphere.c
*/
BOOL RaySphere(XYZ p1, XYZ p2, XYZ sc, double r, double *mu1, double *mu2)
{
	double a,b,c;
	double bb4ac;
	XYZ dp;

	dp.x = p2.x - p1.x;
	dp.y = p2.y - p1.y;
	dp.z = p2.z - p1.z;
	a = dp.x * dp.x + dp.y * dp.y + dp.z * dp.z;
	b = 2 * (dp.x * (p1.x - sc.x) + dp.y * (p1.y - sc.y) + dp.z * (p1.z - sc.z));
	c = sc.x * sc.x + sc.y * sc.y + sc.z * sc.z;
	c += p1.x * p1.x + p1.y * p1.y + p1.z * p1.z;
	c -= 2 * (sc.x * p1.x + sc.y * p1.y + sc.z * p1.z);
	c -= r * r;
	bb4ac = b * b - 4 * a * c;
	if (abs(a) < EPS || bb4ac < 0) {
		*mu1 = 0;
		*mu2 = 0;
		return(FALSE);
	}

	*mu1 = (-b + sqrt(bb4ac)) / (2 * a);
	*mu2 = (-b - sqrt(bb4ac)) / (2 * a);

	return(TRUE);
}

/* See above */
/* dp is the difference P2 - P1 or the vector P1P2 */
BOOL RayCircle(XY p1, XY dp, XY sc, double r, double *mu1, double *mu2){
	double a,b,c;
	double bb4ac;

	a = dp.x * dp.x + dp.y * dp.y;
	b = 2 * (dp.x * (p1.x - sc.x) + dp.y * (p1.y - sc.y) );
	c = sc.x * sc.x + sc.y * sc.y;
	c += p1.x * p1.x + p1.y * p1.y;
	c -= 2 * (sc.x * p1.x + sc.y * p1.y);
	c -= r * r;
	bb4ac = b * b - 4 * a * c;
	if (abs(a) < EPS || bb4ac < 0) {
		*mu1 = 0;
		*mu2 = 0;
		return(FALSE);
	}

	*mu1 = (-b + sqrt(bb4ac)) / (2 * a);
	*mu2 = (-b - sqrt(bb4ac)) / (2 * a);

	return(TRUE);
};



/* Computes the integer square root very fast */
static unsigned int
isqrt26(unsigned int input) {
	unsigned long   s = 0;
	unsigned long   s21 = 1073741824;

	if (s21 <= input)
		s += 32768, input -= s21;
	s21 = (s << 15) + 268435456;
	if (s21 <= input)
		s += 16384, input -= s21;
	s21 = (s << 14) + 67108864;
	if (s21 <= input)
		s += 8192, input -= s21;
	s21 = (s << 13) + 16777216;
	if (s21 <= input)
		s += 4096, input -= s21;
	s21 = (s << 12) + 4194304;
	if (s21 <= input)
		s += 2048, input -= s21;
	s21 = (s << 11) + 1048576;
	if (s21 <= input)
		s += 1024, input -= s21;
	s21 = (s << 10) + 262144;
	if (s21 <= input)
		s += 512, input -= s21;
	s21 = (s << 9) + 65536;
	if (s21 <= input)
		s += 256, input -= s21;
	s21 = (s << 8) + 16384;
	if (s21 <= input)
		s += 128, input -= s21;
	s21 = (s << 7) + 4096;
	if (s21 <= input)
		s += 64, input -= s21;
	s21 = (s << 6) + 1024;
	if (s21 <= input)
		s += 32, input -= s21;
	s21 = (s << 5) + 256;
	if (s21 <= input)
		s += 16, input -= s21;
	s21 = (s << 4) + 64;
	if (s21 <= input)
		s += 8, input -= s21;
	s21 = (s << 3) + 16;
	if (s21 <= input)
		s += 4, input -= s21;
	s21 = (s << 2) + 4;
	if (s21 <= input)
		s += 2, input -= s21;
	s21 = (s << 1) + 1;
	if (s21 <= input)
		s++;
	return s;
}

/* The NULL vector (0,0) often has special meaning in our program */
int is_null(delta *d){
	return ( (d->dx == 0) && (d->dy == 0));
};

/* The integer length of a direction vector */
int length(delta *d){
	return isqrt26(d->dx * d->dx + d->dy * d->dy);
};


void koord_check(koord *k, char *s){
	if ((k->x < 0) || (k->x >1023) || (k->y < 0) || (k->y > 767)){
		fprintf(stderr,"Koord Check failed for (%d,%d) in %s\n", k->x, k->y, s);
		exit(20);
	}
};

/* compute direction from (x1,y1) to (x2,y2) and store in (xres,yres) 
	Resulting invariant: (x1,y1) + (xres,yres) = (x2,y2)
*/
void sto_dir(koord *k1, koord *k2, delta *res){
	int dx = k2->x - k1->x;
	int dy = k2->y - k1->y;
	
	res->dx = dx;
	res->dy = dy;
	return;
}


/* limit of visible coordinates */
#define EXT_X_LIM	1024
#define EXT_Y_LIM	768

#define INT_X_FACTOR	8
#define INT_Y_FACTOR	8

#define INT_X_LIM	(EXT_X_LIM * INT_X_FACTOR)
#define INT_Y_LIM	(EXT_Y_LIM * INT_Y_FACTOR)

/* Transform external (visible) coordinates into internal representation 
	Stores result in k
	The internal representation is 8 times as big as the external
	x = [0 .. 8191], y = [0 ..  6143)
*/
void ext2int (int x, int y, koord *k){
	k->x = x * INT_X_FACTOR;
	k->y = y * INT_Y_FACTOR;
};

/* same as above but for direction vectors */
void ext2int_dir (int x, int y, delta *d){
	d->dx = x;
	d->dy = y;
};


/* The C modulus function returns a negative remainder for negative arguments 
	This function returns a positive value between 0..mod
*/
int mod(a,b){
	int r;
	r = a % b;
	if (r < 0) r += b;
	return r;
};

/* The C integer division rounds toward 0 (even negative numbers)
	We need the mathematically correct division (round to smallest integer)
*/
int idiv(a,b){
	return (abs(a)/abs(b) * sign(a)*sign(b));
};


	
/* Add a direction vector to an internal coordinate: (x,y) + (dx,dy) = (newx, newy)
	Store result in newk
	This routine implements wrap around, so that the new coordinates are within bounds
	newk and k can be the same location
*/
void add_dir_wrap( koord *k, delta *d, koord *newk){
	newk->x = mod( (k->x + d->dx), INT_X_LIM);
	newk->y = mod( (k->y + d->dy), INT_Y_LIM);
};


/* returns <> 0 iff the two integers match to within 1 visible pixel */
static int match_truncate(int i1, int i2){
	return (abs(i1 - i2) < 8);	
}


/* returns <> 0 iff the two internal directions are compatible (within 1 visible pixel */
int match_deltas(delta *d1, delta *d2){

	return ( match_truncate(d1->dx, d2->dx) && match_truncate(d1->dy, d2->dy));

};

/* Euklidean distance between (x1,y1) and (x2,y2), always positive 
	We do no wrap around
	resulting distance vector is stored in delta_res
*/
int dist_delta(koord *k1, koord *k2, delta *delta_res){
	sto_dir(k1, k2, delta_res);
	return (length(delta_res));	
};


/* Our playing field is a torus with opposite sides linked together 
	For calculations of distance and collision detection we have to take that into account.
	We simply create 8 extra shadow positions, which are at koordinates outside the playing field.
	All 9 positions are stored in the array k[].
*/	
void create_shadows(koord *k2, koord k[]){
	
	/* Original position */
	k[0] = *k2;
	
	/* Copy k2 at left edge */
	k[1].x = -INT_X_LIM + k2->x;
	k[1].y = k2->y;	
	
	/* Copy k2 at right edge */
	k[2].x = INT_X_LIM + k2->x;
	k[2].y = k2->y;	
	
	/* Copy k2 above */
	k[3].x = k2->x;
	k[3].y = INT_Y_LIM + k2->y;
	
	/* Copy k2 below */
	k[4].x = k2->x;
	k[4].y = -INT_Y_LIM + k2->y;
	
	/* Copy k2 left and above */
	k[5].x = -INT_X_LIM + k2->x;
	k[5].y = INT_Y_LIM + k2->y;	
	
	/* Copy k2 left and below */
	k[6].x = -INT_X_LIM + k2->x;
	k[6].y = -INT_Y_LIM + k2->y;	
	
	/* Copy k2 right and above */
	k[7].x = INT_X_LIM + k2->x;
	k[7].y = INT_Y_LIM + k2->y;	
	
	/* Copy k2 right and below */
	k[8].x = INT_X_LIM + k2->x;
	k[8].y = -INT_Y_LIM + k2->y;	

};
	
	
/* Euklidean distance between (x1,y1) and (x2,y2), always positive 
	We compute this distance in the doubly linked torus.
	The resulting shortest direction is stored in delta_res.
	Direction is from k1 to k2 -> Invariant: (x1,y1) + (xres,yres) = (x2,y2)
*/	
int dist_wrap(koord *k1, koord *k2, delta *delta_res){
	int	i, dist, min_dist;
	delta	best_delta;
	delta	delta;
	koord k[9];
	
	min_dist = 999999;
	
	/* Create shadows and original */
	create_shadows(k2, k);
	
	for (i=0; i<9; i++){
		dist = dist_delta(k1, &(k[i]), &delta);
		if (dist < min_dist){
			min_dist = dist;
			best_delta = delta;
		};
	};
	*delta_res = best_delta;
	return (min_dist);
};

int sign(double x){
	if (x < -EPS) return -1;
	if (x > EPS) return 1;
	return 0;
};


/* compute the scalar product (dot product) of the 2 vectors */
int dot_prod(delta *d1, delta *d2){
	return (d1->dx * d2->dx + d1->dy * d2->dy);	
}

/* compute the z coordinate of the cross product between 2 vectors ( = scalar product) 
	Is positive if d2 is reached by turning left from d1
*/
int cross_prod_z(delta *d1, delta *d2){
	return (d1->dx * d2->dy - d1->dy * d2->dx);	
}

/* The cross product is not really what we need.
	Given two directions, d1 and d2, we need to know in which direction we have to turn d1 to reach d2
*/
int dir_turn (delta *d1, delta *d2) {
	int cp;
	cp = sign (cross_prod_z(d1, d2));
	
	if (cp != 0) return cp;
	
	/* Now the two directions are parallel. Either they have same orientations or directly opposite one */
	
	/* No turn if they are already aligned */
	if ( (sign(d1->dx) == sign(d2->dx)) && (sign(d1->dy) == sign(d2->dy)))
		return 0; 

	/* Choose one direction arbitrarily */
	return LEFT;
};

/* The same function as dir_turn, but for XY arguments */
int dir_turn_XY (XY d1, XY d2) {
	int cp;
	cp = sign (d1.x * d2.y - d1.y * d2.x);	
	
	if (cp != 0) return cp;
	
	/* Now the two directions are parallel. Either they have same orientations or directly opposite one */
	
	/* No turn if they are already aligned */
	if ( (sign(d1.x) == sign(d2.x)) && (sign(d1.y) == sign(d2.y)))
		return 0; 

	/* Choose one direction arbitrarily */
	return LEFT;
};

static const double PI = 3.1415926;
static const double PI2 = 2.0 * 3.1415926; 

/* compute the angle of given vector 0 .. 2PI */
double angle_double (double y, double x){
	double res = atan2(y, x);
	if (res < 0) res += PI2;
	return res;
};	


/* The difference between 2 directions (measured from -PI to +Pi.
	If result is negative, angle2 is to the right of angle 1
	else it is to the left.

	We divide the result by the angle our ship turns at 1 frame, i. e. 2*PI/85
	So we get the number of frames it takes to turn from dir d1 to dir d2
*/
int dir_diff_XY (XY d1, XY d2){
	double res;
	
	res = angle_double(d2.y, d2.x) - angle_double(d1.y, d1.x);
	if (res > PI) 
		res = res - PI2;
	else if (res < -PI)
		res = PI2 + res;
	 
//	printf("res = %.2f\n", res);
	res = res * 85.33 / PI2;
//	printf("res = %.2f\n", res);

	return ( round(res) );
};

double dir_diff_double (XY d1, XY d2){
	double res;
	
	res = angle_double(d2.y, d2.x) - angle_double(d1.y, d1.x);
	if (res > PI) 
		res = res - PI2;
	else if (res < -PI)
		res = PI2 + res;
	 
//	printf("res = %.2f\n", res);
	res = res * 85.33 / PI2;
//	printf("res = %.2f\n", res);

	return ( res );
};


	
/* compute the angle of given vector 0 .. 2PI */
double angle (delta *d){
	double res = atan2( (double) d->dy, (double) d->dx);
	if (res < 0) res += PI2;
	return res;
};

