/* $Id: objects.c,v 1.12 2008/06/15 16:03:41 raap Exp $ */

#include <stdio.h>
#include <stdlib.h>

#include "util.h"
#include "vector.h"
#include "player.h"
#include "dir.h"
#include "aim.h"
#include "objects.h"
#include "player.h"


void PrtObj (object *obj, char *s){
#ifdef DEBUG	
	prt_time();
	printf("%s",s);
	if (obj->aura_time >= 0) 
		printf("Ghost ");
	if (obj->death_time > obj->cur_time)
		printf("Zombie ");
	if (obj->birth_time > obj->cur_time)
		printf("Unborn ");
	switch (obj->type){
		case SHOT:
			if (obj->own) printf("Own ");
			printf("SHOT");
//			idx = get_idx(&(obj->dir));
			break;	
		case SHIP:
			printf("SHIP");
//			idx = get_idx(&(obj->dir));
			break;
		case AST:
			printf("AST");
			break;
		
		case UFO_S:
			printf("Small UFO");
			break;
		
		case UFO_L:
			printf("Large UFO");
			break;
		
		default:
			printf("Object");
			break;
	};

	printf(" at (%d, %d) with dir <%d,%d>=%4.2f deg., dir_idx=%d, dt=%d, curtime=%d, birth at %d, death at %d, size=%d, id=%d \n", 
	       obj->cur_xy.x, obj->cur_xy.y, obj->dir.dx, obj->dir.dy, angle(&(obj->dir)) * 360.0 /(2 * 3.141), obj->dir_idx, 
		       obj->dt, obj->cur_time, obj->birth_time, obj->death_time, obj->size, obj->id );
	
#endif
};


/* Initialize an object with reasonable default values 
	The object has just been observed at time cur_time at position cur_xy 
*/
static void init_obj(object *obj, int cur_time, koord *cur_xy){
	obj->dir.dx = 0;
	obj->dir.dy = 0;
	obj->dir_idx =0;
	obj->distance.dx = 0;
	obj->distance.dy = 0;
	obj->dt = 0;
	obj->last_update = cur_time;
	obj->last_xy = *cur_xy;
	obj->aura_time = -1;
	obj->birth_time = obj->cur_time;
	obj->he.target_id = -1;
};

/* Fill a shot object with reasonable default values */
void init_shot(object *shot){
	shot->type = SHOT;
	shot->form = 0;
	shot->size = 1;
	shot->speed_good = UNKNOWN;	
	shot->own = 0;
	init_obj(shot, shot->cur_time, &(shot->cur_xy));
	return;
};

void init_ast(object *ast){
	ast->type = AST;
	ast->speed_good = UNKNOWN;	
	ast->death_time = -1;
	init_obj(ast, ast->cur_time, &(ast->cur_xy));
	return;
};

void init_ufo(object *ufo){
	ufo->speed_good = UNKNOWN;	
	init_obj(ufo, ufo->cur_time, &(ufo->cur_xy));
	return;
};


void init_ship(object *ship){
	ship->type = SHIP;
	ship->speed_good = ESTIMATED;	
	ship->size = 10 * 8;
	init_obj(ship, ship->cur_time, &(ship->cur_xy));
	return;
};


BOOL is_ufo(object *tar){
	return ((tar->type == UFO_S) || (tar->type == UFO_L));
};	

/* Clear the object table */
void tab_init (ObjTab *tab){
	tab->n = 0;
};

//TODO exit only if debugging
/* Insert this object into the first free place in the object table */
void insert (ObjTab *tab, object *obj){

	if (tab->n < OBJECTS_LIM){
		tab->list[tab->n] = *obj;
		tab->n++;
	}
	else{
#ifdef DEBUG		
		fprintf(stderr,"Inserting object into full object table\n");
		exit(20);
#endif
	};
};
	
/* Remove this object from the object table */
void delete (ObjTab *tab, int idx) {
	if (tab->n > 0){
		tab->list[idx] = tab->list[tab->n - 1];
		tab->n--;
	} else {
#ifdef DEBUG	
		fprintf(stderr,"Removing object from empty object table\n");
		exit(20);
#endif
	};
};

void PrtObjList(ObjTab *tab){
	int i;
	for (i=0; i<tab->n; i++)
		PrtObj(&(tab->list[i]), "");
};


/* Turn the ship 
	either by -1, 0 or 1 turns.
	+1 means counter clockwise
	Sets new dir and dir_idx.
*/
void turn_ship (object *ship, int turndir){
	ship->dir_idx = (ship->dir_idx + (turndir * 3)) & 0xff;
	ship->dir = *get_dir(ship->dir_idx);
};


/* Calculates the position of our own shot at its first appearance relative to the mid-point of the ship
	Stores result in *pos 
  no wrap around !
*/
static void inline shot_pos (koord *ship_cur_xy, delta *dir, koord *pos) {
	pos->x = ship_cur_xy->x + (5 * dir->dx) / 2;
	pos->y = ship_cur_xy->y + (5 * dir->dy) / 2;
};


/* calculate new coordinates for a moving object (currently used by aiming algorithm) */ 
void move_pos(object *obj, int dt, koord *new_xy){
	delta offset;
	
	/* sensible default is old location */
	if ( (obj->speed_good != KNOWN) && (obj->speed_good != ESTIMATED) ){ 
		*new_xy = obj->cur_xy; 
		return;
	};
	/* compute the expected new location */
	offset.dx = obj->dir.dx * dt;
	offset.dy = obj->dir.dy * dt;	
	add_dir_wrap( &(obj->cur_xy), &offset, new_xy );

	return;
};


/* =========================== Updating actual (observed) position in space and time  ======================== */

/* Update the old objects position with observed information about new object 
	Updates distance, dt, cur_xy and cur_time
	NOT ! for own shots decrements lifetime NOT!
*/
void update_pos (object *oldobj, object *newobj){
	delta offset;
	
	oldobj->dt += (newobj->cur_time - oldobj->last_update);
	
	dist_wrap( &(oldobj->last_xy), &(newobj->cur_xy), &offset);
	oldobj->distance.dx += offset.dx;
	oldobj->distance.dy += offset.dy;
	
	oldobj->cur_xy = newobj->cur_xy;
	oldobj->cur_time = newobj->cur_time;
	oldobj->last_update = newobj->cur_time;
	oldobj->last_xy = oldobj->cur_xy;
};


/* =========================== Updating actual (observed) speed / direction ================================= */

/* Calculate the speed of objects.
	Uses distance and dt.
	Updates dir_min, dir_max and dir
	TODO do we need dir_min and dir_max ?
*/
void update_speed(object *obj){
	
	if (obj->dt < 1) return;		/* Not enough information to calculate speed */ 
	
	estimate_dir(&(obj->distance), obj->dt, &(obj->dir));
	obj->speed_good = ESTIMATED;
	
	/* Sometimes we have mismatched an object. We regularily reset dt to recover fast from this situation */
	if (obj->dt == 16){
		obj->distance.dx = obj->distance.dx / 2;
		obj->distance.dy = obj->distance.dy / 2;
		obj->dt = 8;
	};
};

/* The UFOs are the only objects (apart from the ship) which can change their speed and direction 
	Handle them seperately 
	Their speed can be accurately measured from frame to frame, because they move  with +/-16 in x direction
	and -16, 0, or +16 in y direction.
	Uses distance and dt.
	Updates dir 
	Resets distance and dt to account for sudden changes in direction
 Returns TRUE if the UFO has changed direction 
*/
BOOL update_speed_ufo (object *ufo){
	delta old_dir;
	
	if (ufo->dt < 1) {
		/* New UFO. We "know" the speed in x direction is either 16 or -16 */
		if (ufo->cur_xy.x == 16)
			ufo->dir.dx = 16;
		else 
			ufo->dir.dx = -16;

		ufo->dir.dy = 0;	/* sensible default */
		return FALSE;		/* Initial speed does not count as a change */
	};
	
	old_dir = ufo->dir;
	
	/* Calculate direction from known distance and time difference */
	estimate_dir(&(ufo->distance), ufo->dt, &(ufo->dir));
	
	/* ATTENTION: Now we simply forget all that we know about the direction of the UFO */
	ufo->distance.dx = 0;
	ufo->distance.dy = 0;

	ufo->dt = 0;
	
	if ( (old_dir.dx != ufo->dir.dx) || (old_dir.dy != ufo->dir.dy) )
		return TRUE;
	return FALSE;
};

/* Calculate the speed and direction of the ship.
	TODO At the moment we simply assume that the ship is not moving. So distance should be (0,0).
	Uses distance and dt.
	
	We can get a very good estimate of the direction of the ship by using the "winkelbyte"
	We assume that the current dir is the visual dir from VectorRAM 
	
	Updates  dir and dir_idx
*/
void update_speed_ship(object *ship, delta *visual_dir){

	/* dir_idx is what we expect the internal direction of the ship to be */
	/* Synchronize it with visible direction */
	ship->dir_idx = sync_dir(visual_dir, ship->dir_idx);
	
	/* And use the dir_idx ("winkelbyte") to update the real direction vector */
	ship->dir = *get_dir(ship->dir_idx);
};


void update_speed_list(ObjList list, int n){
	int i;
	
	for (i=0; i<n; i++){
		update_speed(&(list[i]));
	};
}

void update_speed_list_ufos(ObjList list, int n){
	int i;
	
	for (i=0; i<n; i++){
		update_speed_ufo(&(list[i]));
	};
}

/* =========================== Calculating expected (predicted) position ================================= */

/* Predict the new position of an object if it moves for dt number of frames
	Uses cur_xy, dir and time difference dt.
	Stores result in cur_xy and updates cur_time
*/
void predict_pos(object *obj, int dt){
	delta offset;
	
	/* sensible default is old location */
	if ( (obj->speed_good != KNOWN) && (obj->speed_good != ESTIMATED)) 
		return;
	
	/* compute the expected new location */
	offset.dx = obj->dir.dx * dt;
	offset.dy = obj->dir.dy * dt;	
	
	/* Add this with wrap around to old location and store it in *pos */
	add_dir_wrap( &(obj->cur_xy), &offset, &(obj->cur_xy) );
	obj->cur_time += dt;
	return;
};

/* Update the predicted position of objects (cur_xy) at time newtime */
void predict_pos_list (ObjList list, int n, int newtime){
	int i;
	
	for (i=0; i < n; i++){	
		predict_pos(&(list[i]), newtime - list[i].cur_time);
	};
};

/* ------------------------------------------------------------------------------------------------------------ */



/* Check if newobj matches oldobj
	Two objects match iff type, form and size are equal
	
	If yes, sets res to point to this object
	Uses newobj->type and newobj->form and newobj->size
*/
BOOL MatchType (object *oldobj, object *newobj){
	
	/* Size and form must match ! (type should match anyway, because the database contains only objects of same type */
	if ((oldobj->form != newobj->form) || (oldobj->size != newobj->size) || (oldobj->type != newobj->type))
		return 0;
		
	return 1;
};


/* Check if this object matches any object in the database 
	If yes, returns the index of the best matching object or -1 if no match
	Uses target->cur_xy and target->type and target->form and target->size and target->cur_time
	Sets *res_dist to the best distance
*/
int MatchObj (ObjList list, int n, object *target, int *res_dist){
	object *oldobj;
	delta  d;
	int i, dist, best_dist, best_idx;
		
	best_dist = 999999;		
	best_idx = -1;		/* is >= 0 if we have found a candidate */
	
	/* for each currently active object */
	for (i=0; i < n; i++){
		oldobj = &(list[i]);	
						
		/* Types must match */
		if (!MatchType(oldobj, target)) continue;
		/* Ghost objects are not matched to real objects */
		if (oldobj->aura_time >= 0) continue;
		
		/* Unborn objects are not matched against real objects */
		if (oldobj->birth_time > target->cur_time) continue;
		
		/* min distance and direction of this object to target object */
		dist = dist_wrap( &(oldobj->cur_xy), &(target->cur_xy), &d);
				
		if (dist < best_dist){ 		/* best match so far */
			best_dist = dist;
			best_idx = i;
		};
	};
	*res_dist = best_dist;
	return best_idx;
}


/* check if this objects koords are near some already seen asteroids 
	The shot object has minimum information filled in, i.e. type and cur_xy
	Returns index of the best/first matching asteroid
	or -1 iff failure
*/
int find_obj(ObjList list, int n, object *obj){
	int res;
	int dist;
	
	if ( (res = MatchObj(list, n, obj, &dist)) != -1) {
		
		/* Do we have a real winner */
		if ( (dist < 10 * 8) || 
			((list[res].dt < 2) && (dist < 50 * 8)) ){
//			PrtObj(obj, "Finding match for ");
//			PrtObj(&(list[res]),"Found match with ");
			return res;
		}
	};
	return -1;
};

/* ------------------------------------------------------------------------------------------------------------ */


/* Returns a time >= 0, iff a shot from the current ship will surely hit the given target 
	We assume that the sips direction and position do NOT change before firing
	Returns -1 if not a sure hit
*/
int sure_shot(object *ship, object *tar, HitEvent *he){ 
	koord start1, start2;
	koord k[9];
	int i, t;
	
	/* The shots start 2.5 times their speed outside of the ship
		But they occur 1 frames after the current time.
		So we move the target by 1 frame also
	*/
	
	/* The shot will appear at position start1, but it will be 1 frame from now */
	shot_pos(&(ship->cur_xy), &(ship->dir), &start1);
	
	/* So we move the target by 1 frame also */
	move_pos(tar, 1, &start2);

	create_shadows(&start1, k);
	
	for (i=0; i<9; i++){
//		printf("Start1=(%d,%d), Start2=(%d,%d)\n",start1.x, start1.y, start2.x, start2.y);
		/* We add 1 to the time hit2 returns, because our shot needs 1 frame to appear */
 		t = hit (&k[i], &(ship->dir), &start2, &(tar->dir), tar->size) + 1; 
//		printf("hit timediff = %d\n", t);
		if (t < 1) continue;
	
		/* We know the length of our shots is around 69 frames, so check that target is within reach */
		if ( t > 68) {		
//			printf("Target out of reach!\n");
			continue;
		};
		return t;
	};
	return -1;
}; 

/* Compute the direction into which we have to turn the ship
	to shoot the target and hit it in the fastest possible time
	Returns the expected time to hit
 	and stores the best direction in variable *turn_dir
 	If we cannot possibly hit the target, we return the previous best result and 
 	*turn_dir is not changed 
 	If we can hit the target, but the best shooting time is larger than 68 frames,
 	we change the direction *turn_dir, but do return the previous best result
 	
 	TODO this routine seems to be doing too much
 	TODO we do not necessarily find the shortest possible time to hit,
 		because we only allow one waiting (=no turning) frame
 		It could be better to wait for 2 or 3 frames
 		
 	We are given a dead_band parameter db. It means that we can only fire after db frames have passed.
 	
  no wrap_around
*/
static int dir_to_hit (object *ship, koord *tar_xy, delta *tar_dir, int size, int *turn_dir, int cur_best){
	/* This is stupidly ineffective, but it should work: 
		Simply iterate through all possible directions:	
	*/
	int idx, new_idx, dt_lim, dt, ts, best_t, best_ts, turn_time;
	delta dir;
	koord start1, tar_new;
	
	best_ts = 99;
	best_t = cur_best;
	idx = get_idx(&(ship->dir));
	
	/* We try all possible left and right turns (including no turn at all at the beginning). We can have from 0 upto 43 turns.
	But if we already know that the best result is cur_best, we need only try (cur_best-1) turns
	*/
	dt_lim = min (44, cur_best);
	/* We alternate between left and right turns */
	for (dt = 0; abs(dt) < dt_lim; dt=( dt <= 0 ? -dt + 1 :  -dt) ){
		
		new_idx = (idx + 3*dt) & 0xff;
		dir = *get_dir( new_idx );
		
		/* Our ship is at direction dir */
		/* We have taken dt frames to come here */
		/* Now we see if we can hit the target */
		/* The shots start 2.5 times their speed outside of the ship
		But they occur 1 frame after the current time.
		So we move the target by 1 extra frame also
		*/
		shot_pos(&(ship->cur_xy), &dir, &start1);
		
		/* Compute the time it takes to turn and shoot */
		/* Waiting (=not turning) takes 1 frame also */
		if (dt == 0) turn_time = 2;
		else turn_time = abs(dt) + 1;
		
		/* We have to move the target for the dt+1 frames */
		tar_new.x = tar_xy->x + tar_dir->dx * turn_time;
		tar_new.y = tar_xy->y + tar_dir->dy * turn_time;	
		
		/* We add 1 to the time hit returns, because our shot needs 1 frame to appear */
		ts = hit (&start1, &dir, &tar_new, tar_dir, size) + 1; 
		if (ts < 1) continue;
		
		if ( (turn_time + ts) < best_t){
			
			if (dt == 0) *turn_dir = NO_TURN;
			else if (dt < 0) *turn_dir = RIGHT;
				else *turn_dir = LEFT;
			
			/* We only update the time to hit, if the target is not out of range */
			if (ts <= 68){
//				printf("turn_time=%d, ts=%d\n", turn_time, ts);
				best_t = turn_time + ts;
				dt_lim = min (dt_lim, best_t);
			};
		};
		
	};
	return best_t;
};



/* We are given a target
	Compute whether the ship has to turn left or right or wait to hit the given target in the shortest possible time

	TODO take into account the ship's own speed
	
	Returns the best direction in variable *turn_dir 
	We return -1 if we cannot hit the target 
*/
int turn_to_target (object *ship, object *tar, int *turn_dir, HitEvent *he){
	koord k[9];
	int i, td, t0, tbest;
	
		
//	printf("\nTurn Dir\n");
	td = NO_TURN;	/* default */
	tbest = 99999;
	create_shadows(&(tar->cur_xy), k);
	for (i=0; i<9; i++){
		t0 = dir_to_hit( ship, &(k[i]), &(tar->dir), tar->size, &td, tbest );
		if ( t0 < tbest ){
			tbest = t0;
		};
	};
	*turn_dir = td;
	
	if (tbest == 99999) tbest = -1;
	
	he->hit_time = tbest;
	he->target_id = tar->id;
	
	return tbest;
};


