// network.hpp
// -----------
//
//  (C) Copyright Gerald Thaler 2008.
//
// Distributed under the Boost Software License, Version 1.0.
//    (See accompanying file LICENSE_1_0.txt or copy at
//          http://www.boost.org/LICENSE_1_0.txt)

#ifndef INTREPID_NETWORK_HPP
#define INTREPID_NETWORK_HPP

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
# pragma once
#endif // defined(_MSC_VER) && (_MSC_VER >= 1200)

#include "algo.hpp"
#include "comm_model.hpp"
#include "scope_exit.hpp"
#include "threading.hpp"

namespace intrepid
{
    template<class PlayerImpl>
    class network
    {
    public:
        network();

        void run(string const &server);

    private:
        network(network const &); // = delete;
        network &operator=(network const &); // = delete;

        void connect_socket();
        void handle_connect(error_code const &ec);
        void handle_receive(error_code const &ec, size_t bytes_transferred);
        void init_keys_packet();
        void init_name_packet();
        void run_io();
        void schedule_connect(int ms);
        void schedule_receive();
        void send_keys();

        static int const connect_retry_unconnected_ms = 5000;
        static size_t const receive_buffer_size
                                        = sizeof(comm_model::frame_packet);
        static int const nof_send_name = 2;

        bool                  connected_;
        PlayerImpl            player_;
        scoped_array<uint8_t> receive_buffer_;
            // can't replace this with array because of alignment requirements
        int                   send_name_counter_;
        string                server_;

        struct keys_packet
        {
            uint8_t signature[6];
            uint8_t keys;
            uint8_t ping;
        };
        BOOST_STATIC_ASSERT(sizeof(keys_packet) == 8);
        keys_packet keys_packet_;

        struct name_packet
        {
            uint8_t signature[6];
            uint8_t name[32];
        };
        BOOST_STATIC_ASSERT(sizeof(name_packet) == 38);
        name_packet name_packet_;

        comm_model::ping_to_keys_map ping_to_keys_;

        // Order dependency: io_service_ must be constructed before
        //                   connect_timer_ and socket_.
        asio::io_service      io_service_;
        asio::deadline_timer  connect_timer_;
        asio::ip::udp::socket socket_;
    };

// class network
// public:

    template<class PlayerImpl>
    network<PlayerImpl>::network()
        :   receive_buffer_(new uint8_t[receive_buffer_size]),
            connect_timer_(io_service_),
            socket_(io_service_)
    {
        init_keys_packet();
        init_name_packet();
        fill(receive_buffer_.get(),
             receive_buffer_.get() + receive_buffer_size, 0);
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::init_keys_packet()
    {
        string const sign = "ctmame";
        assert(sign.size() == sizeof(keys_packet_.signature));
        copy(sign, keys_packet_.signature);
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::init_name_packet()
    {
        string const sign = "ctname";
        assert(sign.size() == sizeof(name_packet_.signature));
        copy(sign, name_packet_.signature);
        fill(name_packet_.name, 0);
        string const name = "Intrepid";
        assert(name.size() <= sizeof(name_packet_.name));
        copy(name, name_packet_.name);
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::run(string const &server)
    {
        // The networking thread shall run with a higher priority.
        threading::high_priority_range hpr;

        server_ = server;
        connected_ = false;
        keys_packet_.keys = comm_model::keys::none;
        keys_packet_.ping = 0;
        fill(ping_to_keys_, 0);
        send_name_counter_ = 0;

        player_.start();
        INTREPID_ON_SCOPE_EXIT(bind(&PlayerImpl::stop, &player_));
        run_io();
    }

// private:

    template<class PlayerImpl>
    void network<PlayerImpl>::connect_socket()
    {
        using asio::ip::udp;
        udp::resolver resolver(io_service_);
        udp::resolver::query query(udp::v4(), server_, "1979",
                                   udp::resolver::query::address_configured |
                                   udp::resolver::query::numeric_service);
        udp::endpoint endpoint = *resolver.resolve(query);
        socket_.connect(endpoint);
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::handle_connect(error_code const &ec)
    {
        if (ec == asio::error::operation_aborted)
        {
            return;
        }
        if (ec)
        {
            throw system_error(ec);
        }
        send_keys();
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::handle_receive(error_code const &ec,
                                             size_t bytes_transferred)
    {
        if (ec == asio::error::operation_aborted)
        {
            return;
        }
        if (ec)
        {
            throw system_error(ec);
        }

        void const *recv_buffer = receive_buffer_.get();
        if (bytes_transferred >= comm_model::frame_packet_size)
        {
            assert(bytes_transferred == comm_model::frame_packet_size
                   || bytes_transferred == sizeof(comm_model::frame_packet));
            if (!connected_)
            {
                connected_ = true;
                cout << "Connected to server." << endl;
                send_name_counter_ = nof_send_name;
            }
            if (send_name_counter_ > 0)
            {
                cout << "Sending name." << endl;
                socket_.send(asio::buffer(&name_packet_, sizeof(name_packet)));
                -- send_name_counter_;
            }
            comm_model::frame_packet const *fp =
                    static_cast<comm_model::frame_packet const *>(recv_buffer);
            keys_packet_.keys = player_.next_frame(keys_packet_.ping,
                                                   *fp,
                                                   ping_to_keys_);
            send_keys();
        }
        else if (bytes_transferred > 2)
        {
            char const *msg_buffer = static_cast<char const *>(recv_buffer);
            string const msg(msg_buffer, msg_buffer + bytes_transferred - 2);
            cout << "Server: " << msg << '.' << std::endl;
            if (connected_)
            {
                io_service_.stop();
            }
        }
        schedule_receive();
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::run_io()
    {
        using asio::ip::udp;

        io_service_.reset();

        socket_.open(udp::v4());
        INTREPID_ON_SCOPE_EXIT(bind(&udp::socket::close, &socket_));

        connect_socket();
        schedule_connect(0);
        schedule_receive();
        io_service_.run();
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::schedule_connect(int ms)
    {
        connect_timer_.expires_from_now(posix_time::milliseconds(ms));
        connect_timer_.async_wait(bind(&network<PlayerImpl>::handle_connect,
                                       this, _1));
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::schedule_receive()
    {
        socket_.async_receive(asio::buffer(receive_buffer_.get(),
                                           receive_buffer_size),
                              bind(&network<PlayerImpl>::handle_receive,
                                   this, _1, _2));
    }

    template<class PlayerImpl>
    void network<PlayerImpl>::send_keys()
    {
        socket_.send(asio::buffer(&keys_packet_, sizeof(keys_packet)));
        ping_to_keys_[keys_packet_.ping++] = keys_packet_.keys;
        ping_to_keys_[uint8_t(keys_packet_.ping + 128)] = 0;
        if (!connected_)
        {
            schedule_connect(connect_retry_unconnected_ms);
        }
    }
} // end of namespace intrepid

#endif // include guard
