/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.codehaus.groovy.transform;

import groovy.transform.NonSealed;
import org.codehaus.groovy.ast.*;
import org.codehaus.groovy.control.CompilePhase;
import org.codehaus.groovy.control.SourceUnit;

import static org.codehaus.groovy.ast.ClassHelper.make;

/**
 * Handles generation of code for the @Sealed annotation.
 */
@GroovyASTTransformation(phase = CompilePhase.SEMANTIC_ANALYSIS)
public class NonSealedASTTransformation extends AbstractASTTransformation {

    private static final Class<?> NON_SEALED_CLASS = NonSealed.class;
    private static final ClassNode NON_SEALED_TYPE = make(NON_SEALED_CLASS);

    @Override
    public void visit(ASTNode[] nodes, SourceUnit source) {
        init(nodes, source);
        AnnotatedNode parent = (AnnotatedNode) nodes[1];
        AnnotationNode anno = (AnnotationNode) nodes[0];
        if (!NON_SEALED_TYPE.equals(anno.getClassNode())) return;

        if (parent instanceof ClassNode) {
            ClassNode cNode = (ClassNode) parent;
            if (cNode.isEnum()) {
                addError("@" + NON_SEALED_CLASS.getSimpleName() + " not allowed for enum", cNode);
                return;
            }
            if (cNode.isAnnotationDefinition()) {
                addError("@" + NON_SEALED_CLASS.getSimpleName() + " not allowed for annotation definition", cNode);
                return;
            }
            cNode.putNodeMetaData(NON_SEALED_CLASS, Boolean.TRUE);
        }
    }
}
