/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/

package org.apache.james.user.memory;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatCode;

import org.apache.james.core.Username;
import org.apache.james.user.api.DelegationUserDeletionTaskStep;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

class DelegationUserDeletionTaskStepTest {
    private static final Username ALICE = Username.of("alice@domain.tld");
    private static final Username BOB = Username.of("bob@domain.tld");

    private MemoryDelegationStore delegationStore;
    private DelegationUserDeletionTaskStep testee;

    @BeforeEach
    void setUp() {
        delegationStore = new MemoryDelegationStore();
        testee = new DelegationUserDeletionTaskStep(delegationStore);
    }

    @Test
    void shouldDeleteDelegatingUser() {
        Mono.from(delegationStore.addAuthorizedUser(BOB)
            .forUser(ALICE))
            .block();

        Mono.from(testee.deleteUserData(ALICE)).block();

        assertThat(Flux.from(delegationStore.authorizedUsers(ALICE)).collectList().block())
            .isEmpty();
    }

    @Test
    void shouldDeleteDelegateeUser() {
        Mono.from(delegationStore.addAuthorizedUser(BOB)
            .forUser(ALICE))
            .block();

        Mono.from(testee.deleteUserData(BOB)).block();

        assertThat(Flux.from(delegationStore.authorizedUsers(ALICE)).collectList().block())
            .isEmpty();
    }

    @Test
    void shouldBeIdempotent() {
        assertThatCode(() -> Mono.from(testee.deleteUserData(BOB)).block())
            .doesNotThrowAnyException();
    }
}
